// WorkerWindow
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "WorkerWindow.h"

#include "Core/Env/Assert.h"
#include "Core/Strings/AString.h"
#include "Core/Strings/AStackString.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/JobQueueRemote.h"

#include "../resource1.h"

#include <commctrl.h>

// Defines
//------------------------------------------------------------------------------
#define ID_TRAY_APP_ICON                5000
#define ID_TRAY_EXIT_CONTEXT_MENU_ITEM  3000
#define WM_TRAYICON ( WM_USER + 1 )
#define	WORKER_VERSION_STRING "v0.2"

// Minimize
//------------------------------------------------------------------------------
void Minimize()
{
	// hide the main window
	ShowWindow( WorkerWindow::Get().GetWindowHandle(), SW_HIDE );
}

// Restore
//------------------------------------------------------------------------------
void Restore()
{
	// show the main window
	ShowWindow( WorkerWindow::Get().GetWindowHandle(), SW_SHOW );
}

// WorkerWindowWndProc
//------------------------------------------------------------------------------
LRESULT CALLBACK WorkerWindowWndProc( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
    switch( msg )
    {
		case WM_SYSCOMMAND:
		{
			switch( wParam & 0xfff0 )
			{
				case SC_MINIMIZE:
				{
					Minimize();
					return 0;
				}
				case SC_CLOSE:
				{
					WorkerWindow::Get().SetWantToQuit();
					return 0;
				}
			}
			break;
		}
		case WM_TRAYICON:
		{
			if (lParam == WM_LBUTTONUP)
			{
				Restore();
				return 0;
			}
			else if (lParam == WM_RBUTTONDOWN)
			{
				// display popup menu at mouse position
				POINT curPoint ;
				GetCursorPos( &curPoint ) ;
				SetForegroundWindow( hwnd );

				// Show menu and block until hidden
				UINT item = TrackPopupMenu( WorkerWindow::Get().GetMenu(),
											TPM_RETURNCMD | TPM_NONOTIFY,
											curPoint.x,
											curPoint.y,
											0,
											hwnd,
											nullptr );
				if ( item == ID_TRAY_EXIT_CONTEXT_MENU_ITEM )
				{
					WorkerWindow::Get().SetWantToQuit();
					return 0;
				}
			}
			break;
		}
        case WM_CLOSE:
		{
			Minimize();
			return 0;
		}
        case WM_DESTROY:
		{
			WorkerWindow::Get().SetWantToQuit();
			return 0;
		}
		default:
		{
			// nothing...  fall through
		}
    }

	return DefWindowProc(hwnd, msg, wParam, lParam);
}

// CONSTRUCTOR
//------------------------------------------------------------------------------
WorkerWindow::WorkerWindow( void * hInstance )
	: m_UIState( NOT_READY )
	, m_UIThreadHandle( INVALID_THREAD_HANDLE )
	, m_WindowHandle( nullptr )
	, m_Menu( nullptr )
	, m_HInstance( (HINSTANCE)hInstance )
{
	// spawn UI thread - this creates the window, which must be done
	// on the same thread as we intend to process messages on 
	m_UIThreadHandle = Thread::CreateThread( &UIUpdateThreadWrapper,
											 "UIThread",
											 ( 32 * KILOBYTE ) );
	ASSERT( m_UIThreadHandle != INVALID_THREAD_HANDLE );

	// wait for the UI thread to hit the main update loop
	while ( m_UIState != UPDATING ) { Sleep( 1 ); } 
}

// DESTRUCTOR
//------------------------------------------------------------------------------
WorkerWindow::~WorkerWindow()
{
	// make sure we're only here if we're exiting
	ASSERT( m_UIState == ALLOWED_TO_QUIT );

	// ensure the thread is shutdown
	VERIFY( WaitForSingleObject( m_UIThreadHandle, INFINITE ) == WAIT_OBJECT_0 );

	// clean up the ui thread
	CloseHandle( m_UIThreadHandle );

	// clean up UI resources
	if ( m_WindowHandle )
	{
		DestroyWindow( m_ThreadListView );
		DestroyWindow( m_WindowHandle );
		DestroyMenu( m_Menu );

		Shell_NotifyIcon(NIM_DELETE, &m_NotifyIconData);
	}
}

// SetStatus
//------------------------------------------------------------------------------
void WorkerWindow::SetStatus( const char * statusText )
{
	AStackString< 512 > text;
	text.Format( "FBuildWorker %s - %s", WORKER_VERSION_STRING, statusText );

	VERIFY( SetWindowText( m_WindowHandle, text.Get() ) );
}

// SetWorkerState
//------------------------------------------------------------------------------
void WorkerWindow::SetWorkerState( size_t index, const AString & status )
{
	(void)index; (void)status;
	LVITEM item;
	memset( &item, 0, sizeof( LVITEM ) );
	item.mask = LVIF_TEXT;
	item.pszText = (LPSTR)status.Get();
	item.iItem = (uint32_t)index;
	item.iSubItem = 1;
	SendMessage( m_ThreadListView, LVM_SETITEM, (WPARAM)0, (LPARAM)&item );
}

// UIUpdateThreadWrapper
//------------------------------------------------------------------------------
/*static*/ uint32_t WorkerWindow::UIUpdateThreadWrapper( void * )
{
	WorkerWindow::Get().UIUpdateThread();
	return 0;
}

// UIUpdateThread
//------------------------------------------------------------------------------
void WorkerWindow::UIUpdateThread()
{
	// Create the Window
	const char * windowClass = "mainWindowClass";

	WNDCLASSEX wc;
    wc.cbSize        = sizeof(WNDCLASSEX);
    wc.style         = 0;
    wc.lpfnWndProc   = WorkerWindowWndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = m_HInstance;
    wc.hIcon         = LoadIcon(NULL, IDI_ASTERISK);
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
    wc.lpszMenuName  = NULL;
    wc.lpszClassName = windowClass;
    wc.hIconSm       = LoadIcon(NULL, IDI_ASTERISK);

    if( !RegisterClassEx( &wc ) )
    {
        return;
    }

	// center the window on screen
	int w = 700;
	int h = 250;
	int x = GetSystemMetrics(SM_CXSCREEN)-w;
	int y = 0; // GetSystemMetrics(SM_CYSCREEN)/2-(h/2);

	m_WindowHandle = CreateWindow( windowClass,			// LPCTSTR lpClassName,
								   nullptr,				// LPCTSTR lpWindowName,
								   WS_SYSMENU | WS_MINIMIZEBOX, // DWORD dwStyle,
								   x,y,					// int x, int y,
								   w, h,				// int nWidth, int nHeight,
								   nullptr,				// HWND hWndParent,
								   nullptr,				// HMENU hMenu,
								   nullptr,				// HINSTANCE hInstance,
								   nullptr );			// LPVOID lpParam
	ASSERT( m_WindowHandle );

	// Create the tray icon
	ZeroMemory( &m_NotifyIconData, sizeof( NOTIFYICONDATA ) );
	m_NotifyIconData.cbSize = sizeof(NOTIFYICONDATA);
	m_NotifyIconData.hWnd = m_WindowHandle;
	m_NotifyIconData.uID = ID_TRAY_APP_ICON;
	m_NotifyIconData.uFlags = NIF_ICON |	// provide icon
							  NIF_MESSAGE | // want click msgs
							  NIF_TIP;      // provide tool tip
	m_NotifyIconData.uCallbackMessage = WM_TRAYICON; //this message must be handled in hwnd's window procedure. more info below.
	m_NotifyIconData.hIcon = LoadIcon(NULL, IDI_ASTERISK);//(HICON)LoadImage( NULL, MAKEINTRESOURCE(IDI_ICON1), IMAGE_ICON, 32,32, 0 );
	ASSERT( m_NotifyIconData.hIcon );
	AStackString<> toolTip;
	toolTip.Format( "FBuildWorker %s", WORKER_VERSION_STRING );
	AString::Copy( toolTip.Get(), m_NotifyIconData.szTip, toolTip.GetLength() + 1 );

	// init windows common controls
	INITCOMMONCONTROLSEX icex;           // Structure for control initialization.
	icex.dwICC = ICC_LISTVIEW_CLASSES;
	InitCommonControlsEx(&icex);

	// get main window dimensions for positioning/sizing child controls
	RECT rcClient;                       // The parent window's client area.
	GetClientRect( m_WindowHandle, &rcClient ); 

	// listview
	{
		// Create the list-view window in report view with label editing enabled.
		m_ThreadListView = CreateWindow(WC_LISTVIEW, 
										 "zzz",
										 WS_CHILD | LVS_REPORT | WS_VISIBLE | LVS_NOSORTHEADER,
										 0, 0,
										 rcClient.right - rcClient.left,
										 rcClient.bottom - rcClient.top,
										 m_WindowHandle,
										 nullptr, //(HMENU)IDM_CODE_SAMPLES,
										 nullptr,
										 nullptr); 

		// columns
		LV_COLUMN col;
		memset( &col, 0, sizeof( col ) );
		col.mask = LVCF_WIDTH | LVCF_TEXT;
		col.cx =40;
		col.pszText = "CPU";
		SendMessage(m_ThreadListView,LVM_INSERTCOLUMN,0,(LPARAM)&col);
		col.cx = 640;
		col.pszText = "Status";
		SendMessage(m_ThreadListView,LVM_INSERTCOLUMN,1,(LPARAM)&col);

		size_t numWorkers = JobQueueRemote::Get().GetNumWorkers();
		SendMessage( m_ThreadListView, LVM_SETITEMCOUNT, (WPARAM)numWorkers, (LPARAM)0 );

		for ( size_t i=0; i<numWorkers; ++i )
		{
			LVITEM item;
			memset( &item, 0, sizeof( LVITEM ) );
			item.mask = LVIF_TEXT;
			AStackString<> string;
			string.Format( "%u", (uint32_t)( numWorkers - i ) );
			item.pszText = (LPSTR)string.Get();
			SendMessage( m_ThreadListView, LVM_INSERTITEM, (WPARAM)0, (LPARAM)&item );
		}
	}

	// popup menu for tray icon
	m_Menu = CreatePopupMenu();
    AppendMenu( m_Menu, MF_STRING, ID_TRAY_EXIT_CONTEXT_MENU_ITEM, TEXT( "Exit" ) );

	// Display tray icon
	Shell_NotifyIcon(NIM_ADD, &m_NotifyIconData);

	// Display the window, and minimize it
	// - we do this so the user can see the application has run
    ShowWindow( m_WindowHandle, SW_SHOW );
    UpdateWindow( m_WindowHandle );
	//Minimize();  // TODO:C Enable this when not debugging

	SetStatus( "Idle" );

	// we can now accept manipulation from the main thread
	m_UIState = UPDATING;

	// process messages until wo need to quit
	MSG msg;
	do
	{
		// any messages pending?
		if ( PeekMessage( &msg, nullptr, 0, 0, PM_NOREMOVE ) )
		{
			// message available, process it
			VERIFY( GetMessage( &msg, NULL, 0, 0 ) != 0 );
			TranslateMessage( &msg );
			DispatchMessage( &msg );
			continue; // immediately handle any new messages
		}
		else
		{
			// no message right now - prevent CPU thrashing by having a sleep
			Sleep( 1 );
		}
	} while ( m_UIState < ALLOWED_TO_QUIT );
}

//------------------------------------------------------------------------------
