// TestPrecompiledHeaders.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FBuildTest.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"

#include "Core/FileIO/FileIO.h"
#include "Core/Strings/AStackString.h"

// TestPrecompiledHeaders
//------------------------------------------------------------------------------
class TestPrecompiledHeaders : public FBuildTest
{
private:
	DECLARE_TESTS

	// Helpers
	FBuildStats Build( FBuildOptions options = FBuildOptions(), bool useDB = true ) const;
	const char * GetPCHDBFileName() const { return "../../../../ftmp/Test/PrecompiledHeaders/pch.fdb"; }

	// Tests
	void TestPCH() const;
	void TestPCH_NoRebuild() const;
	void TestPCHWithCache() const;
	void TestPCHWithCache_NoRebuild() const;
};

// Register Tests
//------------------------------------------------------------------------------
REGISTER_TESTS_BEGIN( TestPrecompiledHeaders )
	REGISTER_TEST( TestPCH )
	REGISTER_TEST( TestPCH_NoRebuild )	
	REGISTER_TEST( TestPCHWithCache )
	REGISTER_TEST( TestPCHWithCache_NoRebuild )
REGISTER_TESTS_END

// Build
//------------------------------------------------------------------------------
FBuildStats TestPrecompiledHeaders::Build( FBuildOptions options, bool useDB ) const
{
	options.m_ConfigFile = "Data/TestPrecompiledHeaders/fbuild.bff";
	options.m_ShowSummary = true; // required to generate stats for node count checks

	FBuild fBuild( options );
	TEST_ASSERT( fBuild.Initialize( useDB ? GetPCHDBFileName() : nullptr ) );

	AStackString<> target( "PCHTest" );

	TEST_ASSERT( fBuild.Build( target ) );
	TEST_ASSERT( fBuild.SaveDependencyGraph( GetPCHDBFileName() ) );	

	return fBuild.GetStats();
}

// TestPCH
//------------------------------------------------------------------------------
void TestPrecompiledHeaders::TestPCH() const
{
	FBuildOptions options;
	options.m_ForceCleanBuild = true;
	options.m_UseCacheWrite = true;
	options.m_ShowSummary = true; // required to generate stats for node count checks

	AStackString<> obj( "../../../../ftmp/Test/PrecompiledHeaders/PCHUser.obj" );
	AStackString<> pch( "../../../../ftmp/Test/PrecompiledHeaders/PrecompiledHeader.pch" );
	AStackString<> lib( "../../../../ftmp/Test/PrecompiledHeaders/TestPCH.lib" );
	EnsureFileDoesNotExist( obj );
	EnsureFileDoesNotExist( pch );
	EnsureFileDoesNotExist( lib );

	FBuildStats stats = Build( options, false ); // don't use DB

	EnsureFileExists( obj );
	EnsureFileExists( pch );
	EnsureFileExists( lib );

	// Check stats
	//						Seen,	Built,	Type
	CheckStatsNode ( stats,	4,		3,		Node::FILE_NODE );	// cpp + pch + compiler
	CheckStatsNode ( stats,	2,		2,		Node::OBJECT_NODE );// obj + pch obj
	CheckStatsNode ( stats,	1,		1,		Node::LIBRARY_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DIRECTORY_LIST_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::ALIAS_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DLL_NODE );
	CheckStatsTotal( stats,	10,		8 );

	// check we wrote all objects to the cache
	TEST_ASSERT( stats.GetStatsFor( Node::OBJECT_NODE ).m_NumCacheStores == 1 ); // only the main obj can be cached
}

// TestPCH_NoRebuild
//------------------------------------------------------------------------------
void TestPrecompiledHeaders::TestPCH_NoRebuild() const
{
	FBuildStats stats = Build();

	// Check stats
	//						Seen,	Built,	Type
	CheckStatsNode ( stats,	4,		4,		Node::FILE_NODE );	// cpp + pch cpp + pch .h + compiler
	CheckStatsNode ( stats,	2,		0,		Node::OBJECT_NODE );// obj + pch obj
	CheckStatsNode ( stats,	1,		0,		Node::LIBRARY_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DIRECTORY_LIST_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::ALIAS_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::DLL_NODE );
	CheckStatsTotal( stats,	10,		5 );
}

// TestPCHWithCache
//------------------------------------------------------------------------------
void TestPrecompiledHeaders::TestPCHWithCache() const
{
	FBuildOptions options;
	options.m_ForceCleanBuild = true;
	options.m_UseCacheRead = true;
	options.m_ShowSummary = true; // required to generate stats for node count checks

	AStackString<> obj( "../../../../ftmp/Test/PrecompiledHeaders/PCHUser.obj" );
	AStackString<> pch( "../../../../ftmp/Test/PrecompiledHeaders/PrecompiledHeader.pch" );
	AStackString<> lib( "../../../../ftmp/Test/PrecompiledHeaders/TestPCH.lib" );
	EnsureFileDoesNotExist( obj );
	EnsureFileDoesNotExist( pch );
	EnsureFileDoesNotExist( lib );

	FBuildStats stats = Build( options, false ); // don't use DB

	EnsureFileExists( obj );
	EnsureFileExists( pch );
	EnsureFileExists( lib );

	// Check stats
	//						Seen,	Built,	Type
	CheckStatsNode ( stats,	4,		3,		Node::FILE_NODE );	// cpp + pch + compiler
	CheckStatsNode ( stats,	2,		1,		Node::OBJECT_NODE );// obj + pch obj (build pch only)
	CheckStatsNode ( stats,	1,		1,		Node::LIBRARY_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DIRECTORY_LIST_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::ALIAS_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DLL_NODE );
	CheckStatsTotal( stats,	10,		7 );

	// check all objects came from the cache
	TEST_ASSERT( stats.GetStatsFor( Node::OBJECT_NODE ).m_NumCacheHits == 1 );
}

// TestPCHWithCache_NoRebuild
//------------------------------------------------------------------------------
void TestPrecompiledHeaders::TestPCHWithCache_NoRebuild() const
{
	FBuildStats stats = Build();

	// Check stats
	//						Seen,	Built,	Type
	CheckStatsNode ( stats,	4,		4,		Node::FILE_NODE );	// cpp + pch cpp + pch .h + compiler
	CheckStatsNode ( stats,	2,		0,		Node::OBJECT_NODE );// obj + pch obj
	CheckStatsNode ( stats,	1,		0,		Node::LIBRARY_NODE );
	CheckStatsNode ( stats,	1,		1,		Node::DIRECTORY_LIST_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::ALIAS_NODE );
	CheckStatsNode ( stats,	1,		0,		Node::DLL_NODE );
	CheckStatsTotal( stats,	10,		5 );
}

//------------------------------------------------------------------------------
