// JobQueueRemote
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "JobQueueRemote.h"
#include "Job.h"
#include "WorkerThreadRemote.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/Graph/Node.h"

#include "Core/Containers/AutoPtr.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Time/Timer.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
JobQueueRemote::JobQueueRemote( uint32_t numWorkerThreads ) :
	m_PendingJobs( 1024, true ),
	m_CompletedJobs( 1024, true ),
	m_CompletedJobsFailed( 1024, true ),
	m_Workers( numWorkerThreads, false )
{
	WorkerThread::InitTmpDir();

	for ( uint32_t i=0; i<numWorkerThreads; ++i )
	{
		// identify each worker with an id starting from 1
		// (the "main" thread is considered 0)
		uint32_t threadIndex = ( i + 1001 );
		m_Workers.Append( new WorkerThreadRemote( threadIndex ) );
	}
}

// DESTRUCTOR
//------------------------------------------------------------------------------
JobQueueRemote::~JobQueueRemote()
{
	// signal all workers to stop - ok if this has already been done
	SignalStopWorkers();

	// wait for workers to finish - ok if they stopped before this
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		m_Workers[ i ]->WaitForStop();
		delete m_Workers[ i ];
	}
}

// SignalStopWorkers (Main Thread)
//------------------------------------------------------------------------------
void JobQueueRemote::SignalStopWorkers()
{
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		m_Workers[ i ]->Stop();
	}
}

// HaveWorkersStopped
//------------------------------------------------------------------------------
bool JobQueueRemote::HaveWorkersStopped() const
{
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		if ( m_Workers[ i ]->HasExited() == false )
		{
			return false;
		}
	}
	return true;
}

// GetWorkerStatus
//------------------------------------------------------------------------------
void JobQueueRemote::GetWorkerStatus( size_t index, AString & status ) const
{
	( (WorkerThreadRemote *)m_Workers[ index ] )->GetStatus( status );
}

// QueueJob (Main Thread)
//------------------------------------------------------------------------------
void JobQueueRemote::QueueJob( Job * job )
{
	MutexHolder m( m_PendingJobsMutex );
	m_PendingJobs.Append( job );
}


// GetCompletedJob
//------------------------------------------------------------------------------
Job * JobQueueRemote::GetCompletedJob()
{
	MutexHolder m( m_CompletedJobsMutex );

	// completed jobs
	if ( !m_CompletedJobs.IsEmpty() ) 
	{
		Job * job = m_CompletedJobs[ 0 ];
		m_CompletedJobs.PopFront();
		job->GetNode()->SetState( Node::UP_TO_DATE );
		return job;
	}

	// failed jobs
	if ( !m_CompletedJobsFailed.IsEmpty() ) 
	{
		Job * job = m_CompletedJobsFailed[ 0 ];
		m_CompletedJobsFailed.PopFront();
		job->GetNode()->SetState( Node::FAILED );
		return job;
	}

	return nullptr;
}

// CancelJobsWithUserData
//------------------------------------------------------------------------------
void JobQueueRemote::CancelJobsWithUserData( void * userData )
{
	// delete queued jobs
	{
		MutexHolder m( m_PendingJobsMutex );
		auto it = m_PendingJobs.Begin();
		while ( it != m_PendingJobs.End() )
		{
			if ( ( *it )->GetUserData() == userData )
			{
				delete *it;
				m_PendingJobs.Erase( it );
				continue;
			}
			++it;
		}
	}

	// delete completed jobs
	{
		MutexHolder m( m_CompletedJobsMutex );
		auto it = m_CompletedJobs.Begin();
		while ( it != m_CompletedJobs.End() )
		{
			if ( ( *it )->GetUserData() == userData )
			{
				delete *it;
				m_CompletedJobs.Erase( it );
				continue;
			}
			++it;
		}
	}

	// unhook in-flight jobs
	// (we can't delete these now, so we let them complete and delete
	// them upon completion - see FinishedProcessingJob)
	MutexHolder mh( m_InFlightJobsMutex );
	auto it = m_InFlightJobs.Begin();
	while ( it != m_InFlightJobs.End() )
	{
		if ( ( *it )->GetUserData() == userData )
		{
			( *it )->SetUserData( nullptr );
		}
		++it;
	}
}

// GetJobToProcess (Worker Thread)
//------------------------------------------------------------------------------
Job * JobQueueRemote::GetJobToProcess()
{
	MutexHolder m( m_PendingJobsMutex );
	if ( m_PendingJobs.IsEmpty() )
	{
		return nullptr;
	}

	// building jobs in the order they are queued
	Job * job = m_PendingJobs[ 0 ];
	m_PendingJobs.PopFront();

	MutexHolder mh( m_InFlightJobsMutex );
	m_InFlightJobs.Append( job );

	return job;
}

// FinishedProcessingJob (Worker Thread)
//------------------------------------------------------------------------------
void JobQueueRemote::FinishedProcessingJob( Job * job, bool success )
{
//	ASSERT( job->GetNode()->GetState() == Node::BUILDING );

	// remove from in-flight
	{
		MutexHolder mh( m_InFlightJobsMutex );
		auto it = m_InFlightJobs.Find( job );
		ASSERT( it != nullptr );
		m_InFlightJobs.Erase( it );
	}

	// handle jobs which were cancelled while in flight
	if ( job->GetUserData() == nullptr )
	{
		delete job;
		return;
	}

	// push to appropriate completion queue
	MutexHolder m( m_CompletedJobsMutex );
	if ( success )
	{
		m_CompletedJobs.Append( job );
	}
	else
	{
		m_CompletedJobsFailed.Append( job );
	}
}

// DoBuild
//------------------------------------------------------------------------------
/*static*/ Node::BuildResult JobQueueRemote::DoBuild( Job * job )
{
	Timer timer; // track how long the item takes

	Node * node = job->GetNode();

	// remote tasks must output to a tmp file
	if ( job->IsLocal() == false )
	{
		AStackString<> tmpFileName;
		WorkerThread::CreateTempFilePath( "remote.obj", tmpFileName );
		node->ReplaceDummyName( tmpFileName);
	}

	ASSERT( node->IsAFile() );

	// make sure the output path exists
	if ( Node::EnsurePathExistsForFile( node->GetName() ) == false )
	{
		// error already output by EnsurePathExistsForFile
		return Node::NODE_RESULT_FAILED;
	}

	Node::BuildResult result = node->DoBuild2( job );

	uint32_t timeTakenMS = uint32_t( timer.GetElapsedMS() );

	if ( result == Node::NODE_RESULT_OK )
	{
		// record new build time only if built (i.e. if failed, the time
		// does not represent how long it takes to create this resource)
		node->SetLastBuildTime( timeTakenMS );
		node->SetStatFlag( Node::STATS_BUILT );
		//FLOG_INFO( "-Build: %u ms\t%s", timeTakenMS, node->GetName().Get() );
	}

	if ( result == Node::NODE_RESULT_FAILED )
	{
		// build of file failed - if there is a file....
		if ( FileIO::FileExists( node->GetName().Get() ) )
		{
			// ... it is invalid, so try to delete it
			if ( FileIO::FileDelete( node->GetName().Get() ) == false )
			{
				// failed to delete it - this might cause future build problems!
				FLOG_ERROR( "Post failure deletion failed for '%s'", node->GetName().Get() );
			}
		}
	}
	else
	{
		// build completed ok, or retrieved from cache...
		ASSERT( result == Node::NODE_RESULT_OK );

		// read results into memory to send back to client
		FileStream fs;
		if ( fs.Open( node->GetName().Get() ) == false )
		{
			FLOG_ERROR( "File missing despite success for '%s'", node->GetName().Get() );
			result = Node::NODE_RESULT_FAILED;
		}
		else
		{
			uint32_t size = (uint32_t)fs.GetFileSize();
			AutoPtr< char > mem( (char *)::Alloc( size ) );
			if ( fs.Read( mem.Get(), size ) == false )
			{
				FLOG_ERROR( "File read error for '%s'", node->GetName().Get() );
				result = Node::NODE_RESULT_FAILED;
			}
			else
			{
				job->OwnData( mem.Release(), size );
			}
		}
	}

	// log processing time
	node->AddProcessingTime( timeTakenMS );
	
	return result;
}

//------------------------------------------------------------------------------
