// JobQueue - list of pending jobs
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_WORKERPOOL_JOBQUEUE_H
#define FBUILD_WORKERPOOL_JOBQUEUE_H

// Includes
//------------------------------------------------------------------------------
#include "Core/Containers/Array.h"
#include "Core/Containers/Singleton.h"

#include "Tools/FBuild/FBuildCore/Graph/Node.h"
#include "Core/Process/Mutex.h"

// Forward Declarations
//------------------------------------------------------------------------------
class Node;
class Job;
class WorkerThread;


// JobSubQueue
//------------------------------------------------------------------------------
class JobSubQueue
{
public:
	explicit JobSubQueue();
	~JobSubQueue();

	inline uint32_t GetCount() const { return m_Count; }

	// jobs pushed by the main thread
	void QueueJob( Job * job );

	// jobs consumed by workers
	Job * RemoveJob();
private:
	uint32_t	m_Count;	// access the current count
	Mutex		m_Mutex;	// lock to add/remove jobs
	Job *		m_Head;		// remove oldest from head
	Job *		m_Tail;		// push newest to tail
};

// JobQueue
//------------------------------------------------------------------------------
class JobQueue : public Singleton< JobQueue >
{
public:
	JobQueue( uint32_t numWorkerThreads );
	~JobQueue();

	// main thread calls these
	void QueueJob( Node * node );
	void FinalizeCompletedJobs();

	// handle shutting down
	void SignalStopWorkers();
	bool HaveWorkersStopped() const;

	// access state
	size_t GetNumDistributableJobsAvailable() const;
	inline size_t GetDistributableJobsMemUsage() const { return m_PendingJobs2MemoryUsage; }

	void GetJobStats( uint32_t & numJobs, uint32_t & numJobsActive, 
					  uint32_t & numJobsDist, uint32_t & numJobsDistActive ) const;

private:
	// worker threads call these
	friend class WorkerThread;
	Job *		GetJobToProcess();
	static Node::BuildResult DoBuild( Job * job );
	void		FinishedProcessingJob( Job * job, bool result, bool wasRemote );

	void	QueueJob2( Job * job );

	// client side of protocol consumes jobs via this interface
	friend class Client;
	Job *		GetDistributableJobToProcess();
	void		ReturnUnfinishedDistributableJob( Job * job );

	JobSubQueue			m_PendingJobs[ Node::NUM_PRIORITY_LEVELS ];

	mutable Mutex		m_PendingJobsMutex;
	Array< Job * >		m_PendingJobs2;
	size_t				m_PendingJobs2MemoryUsage;
	uint32_t			m_NumJobsActive;
	uint32_t			m_NumJobsDistActive;

	// completed jobs
	mutable Mutex		m_CompletedJobsMutex;
	Array< Job * >		m_CompletedJobs;
	Array< Job * >		m_CompletedJobsFailed;

	// we have pair of arrays to enable a swap, avoiding locking the mutex too long
	Array< Job * >		m_CompletedJobs2;
	Array< Job * >		m_CompletedJobsFailed2;

	Array< WorkerThread * > m_Workers;
};

//------------------------------------------------------------------------------
#endif // FBUILD_WORKERPOOL_JOBQUEUE_H 