// Protocol.cpp - Defines network communication protocol
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Protocol.h"

#include "Core/FileIO/MemoryStream.h"
#include "Core/Network/TCPConnectionPool.h"

// Forward Declarations
//------------------------------------------------------------------------------

// IMessage
//------------------------------------------------------------------------------
Protocol::IMessage::IMessage( Protocol::MessageType msgType, uint32_t msgSize, bool hasPayload )
	: m_MsgType( msgType )
	, m_MsgSize( msgSize )
	, m_HasPayload( hasPayload )
{
}

// IMessage::Send
//------------------------------------------------------------------------------
void Protocol::IMessage::Send( const ConnectionInfo * connection ) const
{
	ASSERT( connection );
	ASSERT( m_HasPayload == false ); // must use Send with payload

	// the packet is the contents of the derived class
	TCPConnectionPool & pool = connection->GetTCPConnectionPool();
	pool.Send( connection, this, m_MsgSize );
}

// IMessage::Send (with payload)
//------------------------------------------------------------------------------
void Protocol::IMessage::Send( const ConnectionInfo * connection, const MemoryStream & payload ) const
{
	ASSERT( connection );
	ASSERT( m_HasPayload == true ); // must NOT use Send with payload

	TCPConnectionPool & pool = connection->GetTCPConnectionPool();
	pool.Send( connection, this, m_MsgSize );
	pool.Send( connection, payload.GetData(), payload.GetSize() );
}

// IMessage::Broadcast
//------------------------------------------------------------------------------
void Protocol::IMessage::Broadcast( TCPConnectionPool * pool ) const
{
	ASSERT( pool );
	ASSERT( m_HasPayload == false ); // must implement custom function

	// the packet is the contents of the derived class
	pool->Broadcast( this, m_MsgSize );
}

// MsgConnection
//------------------------------------------------------------------------------
Protocol::MsgConnection::MsgConnection( uint32_t numJobsAvailable )
	: Protocol::IMessage( Protocol::MSG_CONNECTION, sizeof( MsgConnection ), false )
	, m_ProtocolVersion( PROTOCOL_VERSION )
	, m_NumJobsAvailable( numJobsAvailable )
{
}

// MsgStatus
//------------------------------------------------------------------------------
Protocol::MsgStatus::MsgStatus( uint32_t numJobsAvailable )
	: Protocol::IMessage( Protocol::MSG_STATUS, sizeof( MsgStatus ), false )
	, m_NumJobsAvailable( numJobsAvailable )
{
}

// MsgRequestJob
//------------------------------------------------------------------------------
Protocol::MsgRequestJob::MsgRequestJob()
	: Protocol::IMessage( Protocol::MSG_REQUEST_JOB, sizeof( MsgRequestJob ), false )
{
}

// MsgNoJobAvailable
//------------------------------------------------------------------------------
Protocol::MsgNoJobAvailable::MsgNoJobAvailable()
	: Protocol::IMessage( Protocol::MSG_NO_JOB_AVAILABLE, sizeof( MsgNoJobAvailable ), false )
{
}

// MsgJob
//------------------------------------------------------------------------------
Protocol::MsgJob::MsgJob( uint32_t toolId )
	: Protocol::IMessage( Protocol::MSG_JOB, sizeof( MsgJob ), true )
	, m_ToolId( toolId )
{
}

// MsgJobResult
//------------------------------------------------------------------------------
Protocol::MsgJobResult::MsgJobResult()
	: Protocol::IMessage( Protocol::MSG_JOB_RESULT, sizeof( MsgJobResult ), true )
{
}

// MsgRequestManifest
//------------------------------------------------------------------------------
Protocol::MsgRequestManifest::MsgRequestManifest( uint32_t toolId )
	: Protocol::IMessage( Protocol::MSG_REQUEST_MANIFEST, sizeof( MsgRequestManifest ), false )
	, m_ToolId( toolId )
{
}

// MsgManifest
//------------------------------------------------------------------------------
Protocol::MsgManifest::MsgManifest()
	: Protocol::IMessage( Protocol::MSG_MANIFEST, sizeof( MsgManifest ), true )
{
}

// MsgRequestFile
//------------------------------------------------------------------------------
Protocol::MsgRequestFile::MsgRequestFile( uint32_t toolId, uint32_t fileId )
	: Protocol::IMessage( Protocol::MSG_REQUEST_FILE, sizeof( MsgRequestFile ), false )
	, m_ToolId( toolId )
	, m_FileId( fileId )
{
}

// MsgFile
//------------------------------------------------------------------------------
Protocol::MsgFile::MsgFile( uint32_t toolId, uint32_t fileId )
	: Protocol::IMessage( Protocol::MSG_FILE, sizeof( MsgFile ), false )
	, m_ToolId( toolId )
	, m_FileId( fileId )
{
}

//------------------------------------------------------------------------------
