// Client.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Client.h"

#include "Tools/FBuild/FBuildCore/Protocol/Protocol.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/Graph/CompilerNode.h"
#include "Tools/FBuild/FBuildCore/Graph/FileNode.h"
#include "Tools/FBuild/FBuildCore/Graph/Node.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectNode.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/Job.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h"

#include "Core/FileIO/ConstMemoryStream.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/FileIO/MemoryStream.h"
#include "Core/Math/Random.h"

// Defines
//------------------------------------------------------------------------------
#define CLIENT_STATUS_UPDATE_FREQUENCY_SECONDS ( 1.0f )
#define CONNECTION_LIMIT ( 10 )
#define CONNECTION_REATTEMPT_DELAY_TIME ( 5.0f )

// CONSTRUCTOR
//------------------------------------------------------------------------------
Client::Client( const Array< AString > & workerList )
	: m_WorkerList( workerList )
	, m_ShouldExit( false )
	, m_Exited( false )
{
	// allocate space for server states
	m_ServerList.SetSize( workerList.GetSize() );

	m_Thread = Thread::CreateThread( ThreadFuncStatic,
									 "Client",
									 ( 64 * KILOBYTE ),
									 this );
	ASSERT( m_Thread );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Client::~Client()
{
	m_ShouldExit = true;
	while ( m_Exited == false )
	{
		Sleep( 1 );
	}

	ShutdownAllConnections();

	CloseHandle( m_Thread );
}

//------------------------------------------------------------------------------
/*virtual*/ void Client::OnDisconnected( const ConnectionInfo * connection )
{
	ASSERT( connection );
	ServerState * ss = (ServerState *)connection->GetUserData();
	ASSERT( ss );

	MutexHolder mh( m_ServerListMutex );
	if ( ss->m_Jobs.IsEmpty() == false )
	{
		auto it = ss->m_Jobs.Begin();
		auto end = ss->m_Jobs.End();
		while ( it != end )
		{
			JobQueue::Get().ReturnUnfinishedDistributableJob( *it );
			++it;
		}
		ss->m_Jobs.Clear();
	}
	ss->m_Connection = nullptr;
	ss->m_CurrentMessage = nullptr;
}

// ThreadFuncStatic
//------------------------------------------------------------------------------
/*static*/ uint32_t Client::ThreadFuncStatic( void * param )
{
	Client * c = (Client *)param;
	c->ThreadFunc();
	return 0;
}

// ThreadFunc
//------------------------------------------------------------------------------
void Client::ThreadFunc()
{
	// ensure first status update will be sent more rapidly
	m_StatusUpdateTimer.Start( CLIENT_STATUS_UPDATE_FREQUENCY_SECONDS * 0.5f );

	while ( m_ShouldExit == false )
	{
		LookForWorkers();

		CommunicateJobAvailability();

		Sleep( 1 );
	}

	m_Exited = true;
}

// LookForWorkers
//------------------------------------------------------------------------------
void Client::LookForWorkers()
{
	MutexHolder mh( m_ServerListMutex );

	const size_t numWorkers( m_ServerList.GetSize() );

	// find out how many connections we have now
	size_t numConnections = 0;
	for ( size_t i=0; i<numWorkers; i++ )
	{
		if ( m_ServerList[ i ].m_Connection )
		{
			numConnections++;
		}
	}

	// limit maximum concurrent connections
	if ( numConnections >= CONNECTION_LIMIT )
	{
		return;
	}

	// if we're connected to every possible worker already
	if ( numConnections == numWorkers )
	{
		return;
	}

	// randomize the start index to better distribute workers when there 
	// are many workers/clients - otherwise all clients will attempt to connect 
	// to the first CONNECTION_LIMIT workers
	Random r;
	size_t startIndex = r.GetRandIndex( (uint32_t)numWorkers );

	// find someone to connect to
	for ( size_t j=0; j<numWorkers; j++ )
	{
		const size_t i( ( j + startIndex ) % numWorkers );

		ServerState & ss = m_ServerList[ i ];
		if ( ss.m_Connection )
		{
			continue;
		}

		// lock the server state
		MutexHolder mhSS( ss.m_Mutex );

		ASSERT( ss.m_Jobs.IsEmpty() );

		if ( ss.m_DelayTimer.GetElapsed() < CONNECTION_REATTEMPT_DELAY_TIME )
		{
			continue;
		}

		const ConnectionInfo * ci = Connect( m_WorkerList[ i ], Protocol::PROTOCOL_PORT );
		if ( ci == nullptr )
		{
			ss.m_DelayTimer.Start(); // reset connection attempt delay
		}
		else
		{
			const uint32_t numJobsAvailable( JobQueue::IsValid() ? (uint32_t)JobQueue::Get().GetNumDistributableJobsAvailable() : 0 );

			ci->SetUserData( &ss );
			ss.m_Connection = ci; // success!
			ss.m_NumJobsAvailable = numJobsAvailable;

			// send connection msg
			Protocol::MsgConnection msg( numJobsAvailable );
			MutexHolder mh2( ss.m_Mutex );
			msg.Send( ci );
		}

		// limit to one connection attempt per iteration
		return;
	}
}

// CommunicateJobAvailability
//------------------------------------------------------------------------------
void Client::CommunicateJobAvailability()
{
	// too soon since last status update?
	if ( m_StatusUpdateTimer.GetElapsed() < CLIENT_STATUS_UPDATE_FREQUENCY_SECONDS )
	{
		return;
	}

	m_StatusUpdateTimer.Start(); // reset time

	// possible for job queue to not exist yet
	if ( !JobQueue::IsValid() )
	{
		return;
	}

	// has status changed since we last sent it?
	uint32_t numJobsAvailable = (uint32_t)JobQueue::Get().GetNumDistributableJobsAvailable();
	Protocol::MsgStatus msg( numJobsAvailable );

	MutexHolder mh( m_ServerListMutex );
	if ( m_ServerList.IsEmpty() )
	{
		return; // no servers to communicate with
	}

	// update each server to know how many jobs we have now
	auto it = m_ServerList.Begin();
	auto end = m_ServerList.End();
	while ( it != end )
	{
		if ( it->m_Connection )
		{
			MutexHolder ssMH( it->m_Mutex );
			if ( it->m_NumJobsAvailable != numJobsAvailable )
			{
				msg.Send( it->m_Connection );
				it->m_NumJobsAvailable = numJobsAvailable;
			}
		}
		++it;
	}
}


// OnReceive
//------------------------------------------------------------------------------
/*virtual*/ void Client::OnReceive( const ConnectionInfo * connection, void * data, uint32_t size )
{
	MutexHolder mh( m_ServerListMutex );

	ServerState * ss = (ServerState *)connection->GetUserData();
	ASSERT( ss );

	// are we expecting a msg, or the payload for a msg?
	void * payload = nullptr;
	size_t payloadSize = 0;
	if ( ss->m_CurrentMessage == nullptr )
	{
		// message
		ss->m_CurrentMessage = reinterpret_cast< const Protocol::IMessage * >( data );
		if ( ss->m_CurrentMessage->HasPayload() )
		{
			return;
		}
	}
	else
	{
		// payload
		ASSERT( ss->m_CurrentMessage->HasPayload() );
		payload = data;
		payloadSize = size;
	}

	// determine message type
	const Protocol::IMessage * imsg = ss->m_CurrentMessage;
	Protocol::MessageType messageType = imsg->GetType();

	switch ( messageType )
	{
		case Protocol::MSG_REQUEST_JOB:
		{
			const Protocol::MsgRequestJob * msg = static_cast< const Protocol::MsgRequestJob * >( imsg );
			Process( connection, msg ); 
			break;
		}
		case Protocol::MSG_JOB_RESULT:
		{
			const Protocol::MsgJobResult * msg = static_cast< const Protocol::MsgJobResult * >( imsg );
			Process( connection, msg, payload, payloadSize ); 
			break;
		}
		case Protocol::MSG_REQUEST_MANIFEST:
		{
			const Protocol::MsgRequestManifest * msg = static_cast< const Protocol::MsgRequestManifest * >( imsg );
			Process( connection, msg ); 
			break;
		}
		default:
		{
			// unknown message type
			ASSERT( false ); // this indicates a protocol bug
			Disconnect( connection );
			break;
		}
	}

	// free everything
	::Free( (void *)( ss->m_CurrentMessage ) );
	::Free( payload );
	ss->m_CurrentMessage = nullptr;
}

// FreeBuffer
//------------------------------------------------------------------------------
/*virtual*/ void Client::FreeBuffer( void * UNUSED( data ) )
{
	// nothing to do - memory is managed in OnReceive
}

// Process( MsgRequestJob )
//------------------------------------------------------------------------------
void Client::Process( const ConnectionInfo * connection, const Protocol::MsgRequestJob * )
{
	if ( JobQueue::IsValid() == false )
	{
		return;
	}

	ServerState * ss = (ServerState *)connection->GetUserData();
	ASSERT( ss );

	Job * job = JobQueue::Get().GetDistributableJobToProcess();
	if ( job == nullptr )
	{
		// tell the client we don't have anything right now
		// (we completed or gave away the job already)
		MutexHolder mh( ss->m_Mutex );
		Protocol::MsgNoJobAvailable msg;
		msg.Send( connection );
		return;
	}

	// send the job to the client
	MemoryStream stream;
	job->Serialize( stream );

	MutexHolder mh( ss->m_Mutex );

	ss->m_Jobs.Append( job ); // Track in-flight job

	// obtain the toolid
	uint32_t toolId = 0; // default to no tool synchronization

	// if tool is explicity specified, get the id of the tool manifest
	Node * n = job->GetNode()->CastTo< ObjectNode >()->GetCompiler();
	if ( n->GetType() == Node::COMPILER_NODE )
	{
		const ToolManifest & manifest = n->CastTo< CompilerNode >()->GetManifest();
		toolId = manifest.GetToolId();
	}

	Protocol::MsgJob msg( toolId );
	msg.Send( connection, stream );
}

// Process( MsgJobResult )
//------------------------------------------------------------------------------
void Client::Process( const ConnectionInfo * connection, const Protocol::MsgJobResult *, const void * payload, size_t payloadSize )
{
	// find server
	ServerState * ss = (ServerState *)connection->GetUserData();
	ASSERT( ss );

	ConstMemoryStream ms( payload, payloadSize );

	uint32_t jobId = 0;
	ms.Read( jobId );

	AStackString<> name;
	ms.Read( name );

	bool result = false;
	ms.Read( result );

	uint32_t buildTime;
	ms.Read( buildTime );

	// get result data (built data or errors if failed)
	uint32_t size = 0;
	ms.Read( size );
	const void * data = (const char *)ms.GetData() + ms.Tell();

	MutexHolder mh( ss->m_Mutex );
	auto iter = ss->m_Jobs.FindDeref( jobId );
	ASSERT( iter ); // got an unexpected job result - something is very wrong

	if ( result == true )
	{
		// built ok - serialize to disc

		const AString & nodeName = (*iter)->GetNode()->GetName();
		if ( Node::EnsurePathExistsForFile( nodeName ) == false )
		{
			FLOG_ERROR( "Failed to create path for '%s'", nodeName );
			result = false;
		}
		else
		{
			FileStream fs;
			if ( fs.Open( nodeName.Get(), FileStream::WRITE_ONLY ) == false )
			{
				FLOG_ERROR( "Failed to create file '%s'", nodeName );
				result = false;
			}
			else
			{
				if ( fs.WriteBuffer( data, size ) != size )
				{
					FLOG_ERROR( "Failed to write file '%s'", nodeName );
					result = false;
				}
				else
				{
					// record build time
					fs.Close();
					FileNode * f = (FileNode *)(*iter)->GetNode();
					f->m_Stamp = FileIO::GetFileLastWriteTime( nodeName );

					// record time taken to build
					f->SetLastBuildTime( buildTime );
				}
			}
		}
	}
	else
	{
		// failed - emit errors
		Node::DumpOutput( (const char *)data, size );
	}

	JobQueue::Get().FinishedProcessingJob( *iter, result, true );

	ss->m_Jobs.Erase( iter );
}

// Process( MsgRequestManifest )
//------------------------------------------------------------------------------
void Client::Process( const ConnectionInfo * connection, const Protocol::MsgRequestManifest * msg )
{
	ServerState * ss = (ServerState *)connection->GetUserData();
	ASSERT( ss );

	MutexHolder mh( ss->m_Mutex );


	// find a job associated with this client with this toolId
	const uint32_t toolId = msg->GetToolId();
	ASSERT( toolId != 0 ); // server should not request 'no sync' tool id
	const ToolManifest * manifest( nullptr );
	for ( auto it = ss->m_Jobs.Begin();
		  it != ss->m_Jobs.End();
		  ++it )
	{
		Node * n = ( *it )->GetNode()->CastTo< ObjectNode >()->GetCompiler();
		const ToolManifest & m = n->CastTo< CompilerNode >()->GetManifest();
		if ( m.GetToolId() == toolId )
		{
			// found a job with the same toolid
			manifest = &m;
			break;
		}
	}

	if ( manifest == nullptr )
	{
		// client asked for a manifest that is not valid
		ASSERT( false ); // this indicates a logic bug
		Disconnect( connection );
		return;
	}

	MemoryStream ms;
	manifest->Serialize( ms );

	// Send manifest to worker
	Protocol::MsgManifest resultMsg;
	resultMsg.Send( connection, ms );
}

// CONSTRUCTOR( ServerState )
//------------------------------------------------------------------------------
Client::ServerState::ServerState() 
	: m_CurrentMessage( nullptr ) 
	, m_Connection( nullptr )
	, m_Jobs( 16, true )
{ 
	m_DelayTimer.Start( 999.0f );
}

//------------------------------------------------------------------------------
