// VSProjectGenerator
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "VSProjectGenerator.h"

// Core
#include "Core/Math/CRC32.h"
#include "Core/Strings/AStackString.h"

// system
#include <stdarg.h> // for va_args

// CONSTRUCTOR
//------------------------------------------------------------------------------
VSProjectGenerator::VSProjectGenerator()
	: m_Files( 1024, true )
	, m_Platforms( 8, true )
	, m_Configs( 8, true )
	, m_AllowedFileExtensions( 0, true )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
VSProjectGenerator::~VSProjectGenerator()
{
}

// SetBasePath
//------------------------------------------------------------------------------
void VSProjectGenerator::SetBasePath( const AString & path )
{
	m_BasePath = path;

	// ensure slash consistency which we rely on later
	m_BasePath.Replace( '/', '\\' );

	// ensure backslash terminated
	if ( m_BasePath.EndsWith( '\\' ) == false )
	{
		m_BasePath += '\\';
	}
}

// AddFile
//------------------------------------------------------------------------------
void VSProjectGenerator::AddFile( const AString & file, bool filterByExtension )
{
	// ensure slash consistency which we rely on later
	AStackString<> fileCopy( file );
	fileCopy.Replace( '/', '\\' );

	// filtering by extension?
	size_t numAllowedFileExtensions = m_AllowedFileExtensions.GetSize();
	if ( filterByExtension && numAllowedFileExtensions )
	{
		bool keep = false;
		for ( size_t i=0; i<numAllowedFileExtensions; ++i )
		{
			if ( file.EndsWithI( m_AllowedFileExtensions[ i ] ) )
			{
				keep = true;
				break;
			}
		}
		if ( !keep )
		{
			return;
		}
	}

	ASSERT( !m_Files.Find( fileCopy ) );
	m_Files.Append( fileCopy );
}

// AddFiles
//------------------------------------------------------------------------------
void VSProjectGenerator::AddFiles( const Array< AString > & files, bool filterByExtension )
{
	auto fEnd = files.End();
	for ( auto fIt = files.Begin(); fIt!=fEnd; ++fIt )
	{
		AddFile( *fIt, filterByExtension );
	}
}

// AddPlatform
//------------------------------------------------------------------------------
void VSProjectGenerator::AddPlatform( const AString & platform )
{
	ASSERT( !m_Platforms.Find( platform ) );
	m_Platforms.Append( platform );
}

// AddConfig
//------------------------------------------------------------------------------
void VSProjectGenerator::AddConfig( const AString & config )
{
	ASSERT( !m_Configs.Find( config ) );
	m_Configs.Append( config );
}

// GenerateVCXProj
//------------------------------------------------------------------------------
const AString & VSProjectGenerator::GenerateVCXProj()
{
	ASSERT( !m_ProjectName.IsEmpty() ); // needed for valid guid generation

	// preallocate to avoid re-allocations
	m_Tmp.SetReserved( MEGABYTE );
	m_Tmp.SetLength( 0 );

	// header
	Write( "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" );
	Write( "<Project DefaultTargets=\"Build\" ToolsVersion=\"4.0\" xmlns=\"http://schemas.microsoft.com/developer/msbuild/2003\">\n" );

	// Project Configurations
	{
		Write( "  <ItemGroup Label=\"ProjectConfigurations\">\n" );
		auto cEnd = m_Configs.End();
		auto pEnd = m_Platforms.End();
		for ( auto cIt = m_Configs.Begin(); cIt!=cEnd; ++cIt )
		{
			for ( auto pIt = m_Platforms.Begin(); pIt!=pEnd; ++pIt )
			{
				Write( "    <ProjectConfiguration Include=\"%s|%s\">\n", cIt->Get(), pIt->Get() );
				Write( "      <Configuration>%s</Configuration>\n", cIt->Get() );
				Write( "      <Platform>%s</Platform>\n", pIt->Get() );
				Write( "    </ProjectConfiguration>\n" );
			}
		}
		Write( "  </ItemGroup>\n" );
	}

	// files
	{
		Write("  <ItemGroup>\n" );
		auto fEnd = m_Files.End();
		for ( auto fIt = m_Files.Begin(); fIt!=fEnd; ++fIt )
		{
			const char * fileName = fIt->BeginsWithI( m_BasePath ) ? fIt->Get() + m_BasePath.GetLength() : fIt->Get();
		    Write( "    <CustomBuild Include=\"%s\" />\n", fileName );
		}
		Write("  </ItemGroup>\n" );
	}

	// Globals
	Write( "  <PropertyGroup Label=\"Globals\">\n" );
    Write( "    <ProjectGuid>{%08x-6c94-4f93-bc2a-7f5284b7d434}</ProjectGuid>\n", CRC32::Calc( m_ProjectName ) );
    Write( "    <Keyword>MakeFileProj</Keyword>\n" );
	Write( "  </PropertyGroup>\n" );

	// Default props
	Write( "  <Import Project=\"$(VCTargetsPath)\\Microsoft.Cpp.Default.props\" />\n" );

	// Configurations
	{
		auto cEnd = m_Configs.End();
		auto pEnd = m_Platforms.End();
		for ( auto cIt = m_Configs.Begin(); cIt!=cEnd; ++cIt )
		{
			for ( auto pIt = m_Platforms.Begin(); pIt!=pEnd; ++pIt )
			{
				Write( "  <PropertyGroup Condition=\"'$(Configuration)|$(Platform)'=='%s|%s'\" Label=\"Configuration\">\n", cIt->Get(), pIt->Get() );
				Write( "    <ConfigurationType>Makefile</ConfigurationType>\n" );
				Write( "    <UseDebugLibraries>false</UseDebugLibraries>\n" );

				if ( !m_LocalDebuggerCommandArguments.IsEmpty() )
				{
					Write( "    <LocalDebuggerCommandArguments>%s</LocalDebuggerCommandArguments>\n", m_LocalDebuggerCommandArguments.Get() );
				}
				if ( !m_LocalDebuggerWorkingDirectory.IsEmpty() )
				{
					Write( "    <LocalDebuggerWorkingDirectory>%s</LocalDebuggerWorkingDirectory>\n", m_LocalDebuggerWorkingDirectory.Get() );
				}
				if ( !m_LocalDebuggerCommand.IsEmpty() )
				{
					Write( "    <LocalDebuggerCommand>%s</LocalDebuggerCommand>\n", m_LocalDebuggerCommand.Get() );
				}
				if ( !m_LocalDebuggerEnvironment.IsEmpty() )
				{
					Write( "    <LocalDebuggerEnvironment>%s</LocalDebuggerEnvironment>\n", m_LocalDebuggerEnvironment.Get() );
				}

				Write( "  </PropertyGroup>\n" );
			}
		}
	}

	// Imports
	{
		Write( "  <Import Project=\"$(VCTargetsPath)\\Microsoft.Cpp.props\" />\n" );
		Write( "  <ImportGroup Label=\"ExtensionSettings\">\n" );
		Write( "  </ImportGroup>\n" );
	}

	// Property Sheets
	{
		auto cEnd = m_Configs.End();
		auto pEnd = m_Platforms.End();
		for ( auto cIt = m_Configs.Begin(); cIt!=cEnd; ++cIt )
		{
			for ( auto pIt = m_Platforms.Begin(); pIt!=pEnd; ++pIt )
			{
				Write( "  <ImportGroup Label=\"PropertySheets\" Condition=\"'$(Configuration)|$(Platform)'=='%s|%s'\">\n", cIt->Get(), pIt->Get() );
				Write( "    <Import Project=\"$(UserRootDir)\\Microsoft.Cpp.$(Platform).user.props\" Condition=\"exists('$(UserRootDir)\\Microsoft.Cpp.$(Platform).user.props')\" Label=\"LocalAppDataPlatform\" />\n" );
				Write( "  </ImportGroup>\n" );
			}
		}
	}

	// User macros
	Write( "  <PropertyGroup Label=\"UserMacros\" />\n" );

	// Property Group
	{
		auto cEnd = m_Configs.End();
		auto pEnd = m_Platforms.End();
		for ( auto cIt = m_Configs.Begin(); cIt!=cEnd; ++cIt )
		{
			for ( auto pIt = m_Platforms.Begin(); pIt!=pEnd; ++pIt )
			{
				Write( "  <PropertyGroup Condition=\"'$(Configuration)|$(Platform)'=='%s|%s'\">\n", cIt->Get(), pIt->Get() );
				//Write( "    <NMakeOutput />\n" );
				//Write( "    <NMakePreprocessorDefinitions />\n" );
				if ( !m_BuildCommand.IsEmpty() )
				{
					Write( "    <NMakeBuildCommandLine>%s</NMakeBuildCommandLine>\n", m_BuildCommand.Get() );
				}
				if ( !m_RebuildCommand.IsEmpty() )
				{
					Write( "    <NMakeReBuildCommandLine>%s</NMakeReBuildCommandLine>\n", m_RebuildCommand.Get() );
				}
				if ( !m_CleanCommand.IsEmpty() )
				{
					Write( "    <NMakeCleanCommandLine>%s</NMakeCleanCommandLine>\n", m_CleanCommand.Get() );
				}
				if ( !m_IntDir.IsEmpty() )
				{
					Write( "    <IntDir>%s</IntDir>\n", m_IntDir.Get() );
				}
				if ( !m_OutDir.IsEmpty() )
				{
					Write( "    <OutDir>%s</OutDir>\n", m_OutDir.Get() );
				}
				Write( "  </PropertyGroup>\n" );
			}
		}
	}

	// ItemDefinition Groups
	{
		auto cEnd = m_Configs.End();
		auto pEnd = m_Platforms.End();
		for ( auto cIt = m_Configs.Begin(); cIt!=cEnd; ++cIt )
		{
			for ( auto pIt = m_Platforms.Begin(); pIt!=pEnd; ++pIt )
			{
				Write( "  <ItemDefinitionGroup Condition=\"'$(Configuration)|$(Platform)'=='%s|%s'\">\n", cIt->Get(), pIt->Get() );
				Write( "    <BuildLog>\n" );
				Write( "      <Path />\n" );
				Write( "    </BuildLog>\n" );
				Write( "  </ItemDefinitionGroup>\n" );
			}
		}
	}

	// footer
	Write("  <Import Project=\"$(VCTargetsPath)\\Microsoft.Cpp.targets\" />\n" );
	Write("  <ImportGroup Label=\"ExtensionTargets\">\n" );
	Write("  </ImportGroup>\n" );
	Write( "</Project>" ); // carriage return at end

	m_OutputVCXProj = m_Tmp;
	return m_OutputVCXProj;
}

// GenerateVCXProjFilters
//------------------------------------------------------------------------------
const AString & VSProjectGenerator::GenerateVCXProjFilters()
{
	// preallocate to avoid re-allocations
	m_Tmp.SetReserved( MEGABYTE );
	m_Tmp.SetLength( 0 );

	// header
	Write( "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" );
	Write( "<Project ToolsVersion=\"4.0\" xmlns=\"http://schemas.microsoft.com/developer/msbuild/2003\">\n" );

	// list of all folders
	Array< AString > folders( 1024, true );

	// files
	{
		Write( "  <ItemGroup>\n" );
		auto fEnd = m_Files.End();
		for ( auto fIt = m_Files.Begin(); fIt!=fEnd; ++fIt )
		{
			// get folder part, relative to base dir
			const char * begin = fIt->Get();
			const char * end = fIt->GetEnd();
			{
				if ( fIt->BeginsWithI( m_BasePath ) )
				{
					begin = fIt->Get() + m_BasePath.GetLength();
				}
				const char * lastSlash = fIt->FindLast( '\\' );
				if ( lastSlash )
				{
					end = lastSlash;
				}
			}
			AStackString<> folder;
			if ( begin < end )
			{
				folder.Assign( begin, end );
			}
			const char * fileName = fIt->BeginsWithI( m_BasePath ) ? fIt->Get() + m_BasePath.GetLength() : fIt->Get();
			Write( "    <CustomBuild Include=\"%s\">\n", fileName );
			if ( !folder.IsEmpty() )
			{
				Write( "      <Filter>%s</Filter>\n", folder.Get() );
			}
			Write( "    </CustomBuild>\n" );

			// add new folders
			if ( !folder.IsEmpty() )
			{
				for (;;)
				{
					// add this folder if unique
					bool found = false;
					for ( auto it=folders.Begin(); it!=folders.End(); ++it )
					{
						if ( it->CompareI( folder ) == 0 )
						{
							found = true;
							break;
						}
					}
					if ( !found )
					{
						folders.Append( folder );
					}

					// handle intermediate folders
					const char * lastSlash = folder.FindLast( '\\' );
					if ( lastSlash == nullptr )
					{
						break;
					}
					folder.SetLength( (uint32_t)( lastSlash - folder.Get() ) );
				}
			}
		}
		Write( "  </ItemGroup>\n" );
	}

	// folders
	{
		auto fEnd = folders.End();
		for ( auto fIt = folders.Begin(); fIt!=fEnd; ++fIt )
		{
			Write( "  <ItemGroup>\n" );
			Write( "    <Filter Include=\"%s\">\n", fIt->Get() );
			Write( "      <UniqueIdentifier>{%08x-6c94-4f93-bc2a-7f5284b7d434}</UniqueIdentifier>\n", CRC32::Calc( *fIt ) );
			Write( "    </Filter>\n" );
			Write( "  </ItemGroup>\n" );
		}
	}

	// footer
	Write( "</Project>" ); // no carriage return

	m_OutputVCXProjFilters = m_Tmp;
	return m_OutputVCXProjFilters;
}

// Write
//------------------------------------------------------------------------------
void VSProjectGenerator::Write( const char * fmtString, ... )
{
	AStackString< 1024 > tmp;

	va_list args;
	va_start(args, fmtString);
	tmp.VFormat( fmtString, args );
	va_end( args );

	// resize output buffer in large chunks to prevent re-sizing
	if ( m_Tmp.GetLength() + tmp.GetLength() > m_Tmp.GetReserved() )
	{
		m_Tmp.SetReserved( m_Tmp.GetReserved() + MEGABYTE );
	}	

	m_Tmp += tmp;
}

//------------------------------------------------------------------------------
