// Compressor
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Compressor.h"

#include "Core/Env/Assert.h"
#include "Core/Env/Types.h"
#include "Core/Math/Conversions.h"
#include "Core/Mem/Mem.h"

#include <memory.h>

// Static Data & Typedefs
//------------------------------------------------------------------------------
typedef uint32_t AccumulatorType;
const size_t AccumulatorSizeBits( sizeof( AccumulatorType ) * 8 );

//------------------------------------------------------------------------------
Compressor::Compressor()
	: m_Result( nullptr )
	, m_ResultSize( 0 )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Compressor::~Compressor()
{
	::Free( m_Result );
}

// Compress
//------------------------------------------------------------------------------
#pragma warning( push )
#pragma warning( disable : 6262 ) // static analysis warns about stack usage of predictionTable (64KB)
bool Compressor::Compress( const void * data, size_t dataSize )
{
	ASSERT( data );
	ASSERT( ( (uintptr_t)data % 4 ) == 0 ); // input must be 4 byte aligned
	ASSERT( m_Result == nullptr );

	// create and initialize prediction table
	unsigned char predictionTable[ 256 * 256 ];
	memset( predictionTable, 0, 256 * 256 );

	// allocate space for worst case ( mispredict 1 bit cost for every byte ) 
	size_t worstCaseSpace = dataSize +			// input data
							( dataSize / 8 ) +	// with no compression is a 1 bit per byte overhead
							1 +					// plus rounding
							sizeof( AccumulatorType ) +// rounding
							sizeof( AccumulatorType ); // space for decompressed size at stream start
	m_Result = (char *)::Alloc( worstCaseSpace );
	AccumulatorType * compressedData = (AccumulatorType *)m_Result;
	*compressedData = (AccumulatorType)dataSize; // store the uncompressed size
	++compressedData;
	memset( compressedData, 0, worstCaseSpace - 4 );

	// iterate through data
	const unsigned char * pos = (const unsigned char *)data;
	const unsigned char * const end = pos + dataSize;

	size_t posInBits = 0;
	size_t predictionIndex( 0 );

	do
	{
		const unsigned char c = *pos++;
		const unsigned char predictedC = predictionTable[ predictionIndex ];
		predictionTable[ predictionIndex ] = c;

		size_t bitsNum = ( c == predictedC ) ? 1 : 9;
		size_t bitMask = ( c == predictedC ) ? 0 : 0x1FF; // 1 zeroed bit / 9 bits
		size_t bitsVal = ( bitMask & ( ( c << 1 ) | 0x1 ) );

		size_t outPos = ( (size_t)compressedData + ( posInBits >> 3 ) & ~1);
		*( unsigned int *)outPos |= bitsVal << ( posInBits & 0xF );

		posInBits += bitsNum;

		predictionIndex <<= 8;
		predictionIndex += c;
		predictionIndex &= 0xFFFF;
	}
	while( pos < end );

	uint32_t size = uint32_t( posInBits >> 3 );
	size += ( ( posInBits - ( (size_t)size << 3 ) ) > 0 ) ? 1 : 0;
	size = Math::RoundUp( size, 4 );
	m_ResultSize = size + 4;

	// Result is the size rounded up
	ASSERT( m_ResultSize <= worstCaseSpace );


	return ( m_ResultSize < dataSize );
}
#pragma warning( pop )

// Decompress
//------------------------------------------------------------------------------
#pragma warning( push )
#pragma warning( disable : 6262 ) // static analysis warns about stack usage of predictionTable (64KB)
void Compressor::Decompress( const void * data )
{
	ASSERT( data );
	ASSERT( ( (uintptr_t)data % 4 ) == 0 ); // output must be 4 byte aligned
	ASSERT( m_Result == nullptr );

	// first 4 bytes contains decompression size
	AccumulatorType * compressedData = (AccumulatorType *)data;
	uint32_t uncompressedSize = (uint32_t)*compressedData;
	compressedData++;

	// allocate space for output
	m_Result = (char *)::Alloc( uncompressedSize );

	// create and initialize prediction table
	unsigned char predictionTable[ 256 * 256 ];
	memset( predictionTable, 0, 256 * 256 );

	unsigned char * uncompressedData = (unsigned char *)m_Result;
	const unsigned char * const uncompressedDataEnd( uncompressedData + uncompressedSize );

	size_t posInBits = 0;
	size_t predictionIndex( 0 );

	do
	{		
		// read 9 bits
		size_t inPos = ( (size_t)compressedData + ( posInBits >> 3 ) & ~1);
		size_t bits = *( unsigned int *)inPos;
		bits >>= ( posInBits & 0xF );
		size_t isMiss = ( bits & 0x1 );
		posInBits += 1 + ( 8 * isMiss );
		size_t charBits = ( bits >> 1 );

		unsigned char predictedC = predictionTable[ predictionIndex ];
		size_t keepMask		= ( isMiss * 0xFF );
		size_t keepMaskInv	= 0xFF & ~keepMask;
		unsigned char c = (unsigned char)( ( keepMask & charBits ) | ( keepMaskInv & predictedC ) );
		predictionTable[ predictionIndex ] = c;
				
		*uncompressedData++ = c;

		predictionIndex <<= 8;
		predictionIndex += c;
		predictionIndex &= 0xFFFF;
	}
	while ( uncompressedData != uncompressedDataEnd );

	m_ResultSize = uncompressedSize;
}
#pragma warning( pop )

//------------------------------------------------------------------------------
