// CIncludeParser
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "CIncludeParser.h"

#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"

#include "Core/Math/CRC32.h"
#include "Core/Strings/AStackString.h"
#include "Core/Tracing/Tracing.h"

#include <string.h>

//------------------------------------------------------------------------------
CIncludeParser::CIncludeParser()
: m_Includes( 0, true )
, m_NonUniqueCount( 0 )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
CIncludeParser::~CIncludeParser()
{
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseMSCL_Output( const char * compilerOutput, 
									   size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "Note: including file:", false );
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseMSCL_Preprocessed( const char * compilerOutput, 
											 size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "#line", true );
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseGCC_Preprocessed( const char * compilerOutput, 
											size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "# ", true );
}

// SwapIncludes
//------------------------------------------------------------------------------
void CIncludeParser::SwapIncludes( Array< AString > & includes )
{
	m_Includes.Swap( includes );
}

// SkipWhitespace
//------------------------------------------------------------------------------
/*static*/ void CIncludeParser::SkipWhitespace( const char * & pos, const char * end )
{
	while( pos < end )
	{
		const char c = *pos;
		if ( ( c == ' ' ) || ( c == '\t' ) || ( c == '\r' ) || ( c == '\n' ) )
		{
			++pos;
			continue;
		}
		return;
	}
}

// SkipToEndOfLine
//------------------------------------------------------------------------------
/*static*/ void CIncludeParser::SkipToEndOfLine( const char * & pos, const char * end )
{
	// find end of line
	while( pos < end )
	{
		const char c = *pos;
		if ( ( c == '\r' ) || ( c == '\n' ) )
		{
			break;
		}
		++pos;
	}
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::Parse( const char * compilerOutput, 
							size_t compilerOutputSize,
							const char * startOfLineString,
							bool quoted )
{
	m_Includes.SetCapacity( 4096 );

	// store crc of includes found so far to prevent adding duplicates
	Array< uint32_t > crcs( 4096, true );

	const char * pos = compilerOutput;
	const char * end = compilerOutput + compilerOutputSize;

	const size_t startOfLineStringLen = strlen( startOfLineString );

	uint32_t lastCRC = 0;

	while ( pos < end )
	{
		SkipWhitespace( pos, end );
		const char * lineStart = pos;
		if ( pos >= end )
		{
			break;
		}

		SkipToEndOfLine( pos, end );
		const char * lineEnd( pos );

		// is this line an include directive?
		if ( strncmp( lineStart, startOfLineString, startOfLineStringLen ) != 0 )
		{
			// not an include directive
			continue; // try next line
		}

		// skip over include directive
		lineStart += startOfLineStringLen;

		// trim start/end to just path
		if ( quoted )
		{
			// find surrounding quotes
			while ( ( lineStart < lineEnd ) && ( *lineStart != '"' ) )
			{
				lineStart++;
			}
			lineStart++;
			while ( ( lineEnd > lineStart ) && ( *lineEnd != '"' ) )
			{
				lineEnd--;
			}
			if ( ( lineEnd - lineStart ) < 1 )
			{
				return false; // caller is responsible for emitting error msg
			}
		}
		else
		{
			// advance past whitespace before path
			SkipWhitespace( lineStart, end );
		}

		// ignore special case GCC "<built-in>" and "<command line>"
		if ( *lineStart == '<' )
		{
			continue;
		}
		// ignore GCC paths
		const char lastChar( lineEnd[ -1 ] );
		if ( ( lastChar == '\\' ) || ( lastChar == '/' ) )
		{
			continue;
		}

		m_NonUniqueCount++;

		// record include
		AStackString< 256 > include( lineStart, lineEnd );
		AStackString< 256 > cleanInclude;
		NodeGraph::CleanPath( include, cleanInclude );
		uint32_t crc = CRC32::CalcLower( cleanInclude );
		if ( crc == lastCRC )
		{
			continue;
		}
		lastCRC = crc;
		if ( crcs.Find( crc ) == nullptr )
		{
			crcs.Append( crc );
			m_Includes.Append( cleanInclude );
		}

		// continue looking for next include
	}

	return true;
}

//------------------------------------------------------------------------------
