// UnityNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "UnityNode.h"
#include "DirectoryListNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Graph/VCXProjectNode.h"
#include "Tools/FBuild/FBuildCore/Helpers/VSProjectGenerator.h"

#include "Core/Env/Env.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Process/Process.h"
#include "Core/Strings/AStackString.h"

// system
#include <string.h> // for memcmp

// CONSTRUCTOR
//------------------------------------------------------------------------------
VCXProjectNode::VCXProjectNode( const AString & projectOutput,
								const AString & projectBasePath,
								const Array< DirectoryListNode * > & paths,
								const Array< AString > & pathsToExclude,
								const Array< AString > & allowedFileExtensions,
								const Array< AString > & files,
								const Array< AString > & filesToExclude,
								const Array< AString > & configs,
								const Array< AString > & platforms,
								const AString & buildCmd,
								const AString & rebuildCmd,
								const AString & cleanCmd,
								const AString & localDebuggerCommandArguments,
								const AString & localDebuggerWorkingDirectory,
								const AString & localDebuggerCommand,
								const AString & localDebuggerEnvironment )
: Node( projectOutput, Node::VCXPROJECT_NODE, Node::FLAG_NONE )
, m_ProjectBasePath( projectBasePath )
, m_PathsToExclude( pathsToExclude )
, m_AllowedFileExtensions( allowedFileExtensions )
, m_Files( files )
, m_FilesToExclude( filesToExclude )
, m_Configs( configs )
, m_Platforms( platforms )
, m_BuildCommand( buildCmd )
, m_RebuildCommand( rebuildCmd )
, m_CleanCommand( cleanCmd )
, m_LocalDebuggerCommandArguments( localDebuggerCommandArguments )
, m_LocalDebuggerWorkingDirectory( localDebuggerWorkingDirectory )
, m_LocalDebuggerCommand( localDebuggerCommand )
, m_LocalDebuggerEnvironment( localDebuggerEnvironment ) 
{
	m_LastBuildTimeMs = 100; // higher default than a file node

	// depend on the input nodes
	m_StaticDependencies.Append( ( const Array< Node * > & )paths );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
VCXProjectNode::~VCXProjectNode()
{
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult VCXProjectNode::DoBuild( Job * UNUSED( job ) )
{
	VSProjectGenerator pg;
	pg.SetBasePath( m_ProjectBasePath );
	pg.SetAllowedFileExtensions( m_AllowedFileExtensions );

	// get project file name only
	const char * p1 = m_Name.FindLast( '\\' );
	p1 = p1 ? p1 : m_Name.Get();
	AStackString<> projectName( p1 );
	pg.SetProjectName( projectName );

	// configs
	for ( auto it=m_Configs.Begin(); it!=m_Configs.End(); ++it )
	{
		pg.AddConfig( *it );
	}

	// platforms
	for ( auto it=m_Platforms.Begin(); it!=m_Platforms.End(); ++it )
	{
		pg.AddPlatform( *it );
	}

	// files from directory listings
	Array< FileIO::FileInfo * > files( 1024, true );
	GetFiles( files );
	for ( auto it=files.Begin(); it!=files.End(); ++it )
	{
		const AString & fileName = ( *it )->m_Name;
		AddFile( pg, fileName );
	}

	// files explicitly listed
	for ( auto it=m_Files.Begin(); it!=m_Files.End(); ++it )
	{
		const AString & fileName = ( *it );
		pg.AddFile( fileName, false ); // don't filter by extension
	}	

	// commands
	pg.SetBuildCommand( m_BuildCommand );
	pg.SetRebuildCommand( m_RebuildCommand );
	pg.SetCleanCommand( m_CleanCommand );

	// debugger
	pg.SetLocalDebuggerCommandArguments( m_LocalDebuggerCommandArguments );
	pg.SetLocalDebuggerWorkingDirectory( m_LocalDebuggerWorkingDirectory );
	pg.SetLocalDebuggerCommand( m_LocalDebuggerCommand );
	pg.SetLocalDebuggerEnvironment( m_LocalDebuggerEnvironment );

	// .vcxproj
	const AString & project = pg.GenerateVCXProj();
	if ( Save( project, m_Name ) == false )
	{
		return NODE_RESULT_FAILED; // Save will have emitted an error
	}

	// .vcxproj.filters
	const AString & filters = pg.GenerateVCXProjFilters();
	AStackString<> filterFile( m_Name );
	filterFile += ".filters";
	if ( Save( filters, filterFile ) == false )
	{
		return NODE_RESULT_FAILED; // Save will have emitted an error
	}

	return NODE_RESULT_OK;
}

// AddFile
//------------------------------------------------------------------------------
void VCXProjectNode::AddFile( VSProjectGenerator & pg, const AString & fileName ) const
{
	auto end = m_FilesToExclude.End();
	for( auto it=m_FilesToExclude.Begin(); it!=end; ++it )
	{
		if ( fileName.EndsWithI( *it ) )
		{
			return; // file is ignored
		}
	}

	pg.AddFile( fileName, true );
}

// Save
//------------------------------------------------------------------------------
bool VCXProjectNode::Save( const AString & content, const AString & fileName ) const
{
	bool needToWrite = false;

	FileStream old;
	if ( FBuild::Get().GetOptions().m_ForceCleanBuild )
	{
		needToWrite = true;
	}
	else if ( old.Open( fileName.Get(), FileStream::READ_ONLY ) == false )
	{
		needToWrite = true;
	}
	else
	{
		// files differ in size?
		size_t oldFileSize = (size_t)old.GetFileSize();
		if ( oldFileSize != content.GetLength() )
		{
			needToWrite = true;
		}
		else
		{
			// check content
			AutoPtr< char > mem( ( char *)::Alloc( oldFileSize ) );
			if ( old.Read( mem.Get(), oldFileSize ) != oldFileSize )
			{
				FLOG_ERROR( "VCXProject - Failed to read '%s'", fileName.Get() );
				return false;
			}

			// compare content
			if ( memcmp( mem.Get(), content.Get(), oldFileSize ) != 0 )
			{
				needToWrite = true;
			}
		}

		// ensure we are closed, so we can open again for write if needed
		old.Close();
	}

	// only save if missing or ner
	if ( needToWrite == false )
	{
		return true; // nothing to do.
	}

	FLOG_BUILD( "VCXProj: %s\n", fileName.Get() );

	// actually write
	FileStream f;
	if ( !f.Open( fileName.Get(), FileStream::WRITE_ONLY ) )
	{
		FLOG_ERROR( "VCXProject - Failed to open '%s' for write (error: %u)", fileName.Get(), Env::GetLastErr() );
		return false;
	}
	if ( f.Write( content.Get(), content.GetLength() ) != content.GetLength() )
	{
		FLOG_ERROR( "VCXProject - Error writing to '%s' (error: %u)", fileName.Get(), Env::GetLastErr() );
		return false;
	}
	f.Close();

	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * VCXProjectNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( AStackString<>,	projectBasePath );
	NODE_LOAD_DEPS( 1,			staticDeps );
	NODE_LOAD( Array< AString >, pathsToExclude );
	NODE_LOAD( Array< AString >, allowedFileExtensions );
	NODE_LOAD( Array< AString >, files );
	NODE_LOAD( Array< AString >, filesToExclude );
	NODE_LOAD( Array< AString >, configs );
	NODE_LOAD( Array< AString >, platforms );
	NODE_LOAD( AStackString<>,	buildCmd );
	NODE_LOAD( AStackString<>,	rebuildCmd );
	NODE_LOAD( AStackString<>,	cleanCmd );
	NODE_LOAD( AStackString<>,	localDebuggerCommandArguments );
	NODE_LOAD( AStackString<>,	localDebuggerWorkingDirectory );
	NODE_LOAD( AStackString<>,	localDebuggerCommand );
	NODE_LOAD( AStackString<>,	localDebuggerEnvironment );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	VCXProjectNode * n = ng.CreateVCXProjectNode( name,
								 projectBasePath,
								 reinterpret_cast< Array< DirectoryListNode * > & >( staticDeps ), // all static deps are DirectoryListNode
								 pathsToExclude,
								 allowedFileExtensions,
								 files,
								 filesToExclude,
								 configs,
								 platforms,
								 buildCmd,
								 rebuildCmd,
								 cleanCmd,
								 localDebuggerCommandArguments,
								 localDebuggerWorkingDirectory,
								 localDebuggerCommand,
								 localDebuggerEnvironment );
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool VCXProjectNode::Save( IOStream & stream, bool remote ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_ProjectBasePath );
	NODE_SAVE_DEPS( m_StaticDependencies );
	NODE_SAVE( m_PathsToExclude );
	NODE_SAVE( m_AllowedFileExtensions );
	NODE_SAVE( m_Files );
	NODE_SAVE( m_FilesToExclude );
	NODE_SAVE( m_Configs );
	NODE_SAVE( m_Platforms );
	NODE_SAVE( m_BuildCommand );
	NODE_SAVE( m_RebuildCommand );
	NODE_SAVE( m_CleanCommand );
	NODE_SAVE( m_LocalDebuggerCommandArguments );
	NODE_SAVE( m_LocalDebuggerWorkingDirectory );
	NODE_SAVE( m_LocalDebuggerCommand );
	NODE_SAVE( m_LocalDebuggerEnvironment );
	return true;
}

// GetFiles
//------------------------------------------------------------------------------
void VCXProjectNode::GetFiles( Array< FileIO::FileInfo * > & files ) const
{
	// find all the directory lists
	auto sEnd = m_StaticDependencies.End();
	for ( auto sIt = m_StaticDependencies.Begin(); sIt != sEnd; ++sIt )
	{
		DirectoryListNode * dirNode = ( *sIt )->CastTo< DirectoryListNode >();
		auto filesEnd = dirNode->GetFiles().End();

		// filter files in the dir list
		for ( auto filesIt = dirNode->GetFiles().Begin(); filesIt != filesEnd; ++filesIt )
		{
			bool keep = true;

			// filter excluded files
/*			auto fit = m_FilesToExclude.Begin();
			auto fend = m_FilesToExclude.End();
			for ( ; fit != fend; ++fit )
			{
				if ( filesIt->m_Name.EndsWithI( *fit ) )
				{
					keep = false;
					break;
				}
			}*/

			// filter excluded directories
			if ( keep )
			{
				auto pit = m_PathsToExclude.Begin();
				auto pend = m_PathsToExclude.End();
				for ( ; pit != pend; ++pit )
				{
					if ( filesIt->m_Name.BeginsWithI( *pit ) )
					{
						keep = false;
						break;
					}
				}
			}

			if ( keep )
			{
				files.Append( filesIt );
			}
		}
	}
}

//------------------------------------------------------------------------------
