// ObjectNode.h - a node that builds a single object from a source file
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_GRAPH_OBJECTNODE_H
#define FBUILD_GRAPH_OBJECTNODE_H

// Includes
//------------------------------------------------------------------------------
#include "FileNode.h"
#include "Core/Containers/Array.h"
#include "Core/Containers/AutoPtr.h"
#include "Core/Env/Assert.h"
#include "Core/Process/Process.h"

// Forward Declarations
//------------------------------------------------------------------------------

// ObjectNode
//------------------------------------------------------------------------------
class ObjectNode : public FileNode
{
public:
	explicit ObjectNode( const AString & objectName,
						 Node * inputNode,
						 Node * compilerNode,
						 const AString & compilerArgs,
						 Node * precompiledHeader,
						 uint32_t flags,
						 const Array< Node * > & compilerForceUsing );
	virtual ~ObjectNode();

	static inline Node::Type GetType() { return Node::OBJECT_NODE; }


	enum Flags
	{
		FLAG_CAN_BE_CACHED		=	0x01,
		FLAG_CAN_BE_DISTRIBUTED	=	0x02,
		FLAG_USING_PCH			=	0x04,
		FLAG_GCC				=	0x10,
		FLAG_MSVC				=	0x20,
		FLAG_CREATING_PCH		=   0x40,
		FLAG_SNC				=	0x80,
		FLAG_USING_CLR			=   0x100,
		FLAG_CLANG				=	0x200,
	};
	static uint32_t DetermineFlags( const AString & compiler, const AString & args );

	inline bool IsCreatingPCH() const { return GetFlag( FLAG_CREATING_PCH ); }

	virtual bool Save( IOStream & stream, bool remote ) const;
	static Node * Load( IOStream & stream, bool remote );

	inline Node * GetCompiler() const { return m_StaticDependencies[ 0 ]; }
	inline Node * GetSourceFile() const { return m_StaticDependencies[ 1 ]; }
	inline Node * GetPrecompiledHeaderCPPFile() const { ASSERT( GetFlag( FLAG_CREATING_PCH ) ); return m_StaticDependencies[ 1 ]; }

	const char * GetObjExtension() const;
private:
	virtual BuildResult DoBuild( Job * job );
	virtual BuildResult DoBuild2( Job * job );
	virtual bool Finalize();

	BuildResult DoBuildMSCL_NoCache();
	BuildResult DoBuildWithPreProcessor( Job * job );
	BuildResult DoBuildWithPreProcessor2( Job * job );
	BuildResult DoBuildOther();

	bool ProcessIncludesMSCL( const char * output, uint32_t outputSize, bool preprocessed );
	bool ProcessIncludesWithPreProcessor( Job * job );

	void GetCacheName( const void * memOut, size_t memOutSize,
					   AString & cacheName ) const;
	bool RetrieveFromCache( const AString & cacheFile );
	void WriteToCache( const AString & cacheFile );

	static void DumpOutput( const char * data, uint32_t dataSize, const AString & name );

	void EmitCompilationMessage( const AString & fullArgs ) const;

	void BuildFullArgs( AString & fullArgs, bool preProcess ) const;
	void ExpandTokenList( const Array< Node * > & nodes, AString & fullArgs, const AString & pre, const AString & post ) const;
	bool BuildPreprocessedOutput( const AString & fullArgs, Job * job ) const;
	bool WriteTmpFile( Job * job, AString & tmpFileName ) const;
	bool BuildFinalOutput( Job * job, const AString & fullArgs ) const;

	inline bool GetFlag( uint32_t flag ) const { return ( ( m_Flags & flag ) != 0 ); }

	class CompileHelper
	{
	public:
		CompileHelper( bool handleOutput = true );
		~CompileHelper();

		// start compilation
		bool SpawnCompiler( const AString & name, const Node * compiler, const AString & fullArgs, bool useResponseFile );

		// determine overall result
		inline int						GetResult() const { return m_Result; }

		// access output/error
		inline const AutoPtr< char > &	GetOut() const { return m_Out; }
		inline uint32_t					GetOutSize() const { return m_OutSize; }
		inline const AutoPtr< char > &	GetErr() const { return m_Err; }
		inline uint32_t					GetErrSize() const { return m_ErrSize; }

	private:
		bool			m_HandleOutput;
		Process			m_Process;
		AutoPtr< char > m_Out;
		uint32_t		m_OutSize;
		AutoPtr< char > m_Err;
		uint32_t		m_ErrSize;
		int				m_Result;
	};

	Array< AString > m_Includes;
	uint32_t m_Flags;
	AString m_CompilerArgs;
	AString m_ObjExtensionOverride;
	Array< Node * > m_CompilerForceUsing;
};

//------------------------------------------------------------------------------
#endif // FBUILD_GRAPH_OBJECTNODE_H
