// ObjectNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "ObjectNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Helpers/CIncludeParser.h"
#include "Tools/FBuild/FBuildCore/Helpers/Compressor.h"
#include "Tools/FBuild/FBuildCore/Helpers/ResponseFile.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/Job.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/WorkerThread.h"

#include "Core/Env/Env.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Math/CRC32.h"
#include "Core/Process/Process.h"
#include "Core/Process/Thread.h"
#include "Core/Tracing/Tracing.h"
#include "Core/Strings/AStackString.h"

#include <string.h>
#include <windows.h> // for FILETIME etc


// CONSTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::ObjectNode( const AString & objectName,
					    Node * inputNode,
					    Node * compilerNode,
						const AString & compilerArgs,
						Node * precompiledHeader,
						uint32_t flags,
						const Array< Node * > & compilerForceUsing )
: FileNode( objectName, Node::FLAG_NONE )
, m_Includes( 0, true )
, m_Flags( flags )
, m_CompilerArgs( compilerArgs )
, m_CompilerForceUsing( compilerForceUsing )
{
	m_StaticDependencies.SetCapacity( 3 );

	ASSERT( compilerNode );
	m_StaticDependencies.Append( compilerNode );

	ASSERT( inputNode );
	m_StaticDependencies.Append( inputNode );

	if ( precompiledHeader )
	{
		m_StaticDependencies.Append( precompiledHeader );
	}

	m_StaticDependencies.Append( compilerForceUsing );

	m_Type = OBJECT_NODE;
	m_LastBuildTimeMs = 5000; // higher default than a file node
}

// DESTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::~ObjectNode()
{
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuild( Job * job )
{
	// delete previous file
	if ( FileIO::FileExists( GetName().Get() ) )
	{
		if ( FileIO::FileDelete( GetName().Get() ) == false )
		{
			FLOG_ERROR( "Failed to delete file before build '%s'", GetName().Get() );
			return NODE_RESULT_FAILED;
		}
	}
	if ( GetFlag( FLAG_MSVC ) && GetFlag( FLAG_CREATING_PCH ) )
	{
		AStackString<> pchObj( GetName() );
		pchObj += GetObjExtension();
		if ( FileIO::FileExists( pchObj.Get() ) )
		{
			if ( FileIO::FileDelete( pchObj.Get() ) == false )
			{
				FLOG_ERROR( "Failed to delete file before build '%s'", GetName().Get() );
				return NODE_RESULT_FAILED;
			}
		}
	}

	bool useCache = GetFlag( FLAG_CAN_BE_CACHED ) &&
					( FBuild::Get().GetOptions().m_UseCacheRead ||
					  FBuild::Get().GetOptions().m_UseCacheWrite );
	bool useDist = GetFlag( FLAG_CAN_BE_DISTRIBUTED ) && FBuild::Get().GetOptions().m_AllowDistributed;
	bool usePreProcessor = ( useCache || useDist || GetFlag( FLAG_GCC ) || GetFlag( FLAG_SNC ) || GetFlag( FLAG_CLANG ) );

    // TODO:B Smarter logic to avoid use of preprocessor when
    // cache hit is unlikely (i.e. locally modified files)

	if ( usePreProcessor )
	{
		return DoBuildWithPreProcessor( job );
	}

	if ( GetFlag( FLAG_MSVC ) )
	{
		return DoBuildMSCL_NoCache();
	}

	return DoBuildOther();
}

// DoBuild_Remote
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuild2( Job * job )
{
	return DoBuildWithPreProcessor2( job );
}

// Finalize
//------------------------------------------------------------------------------
/*virtual*/ bool ObjectNode::Finalize()
{
	ASSERT( Thread::IsMainThread() );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// convert includes to nodes
	m_DynamicDependencies.Clear();
	m_DynamicDependencies.SetCapacity( m_Includes.GetSize() );
	for ( Array< AString >::ConstIter it = m_Includes.Begin();
			it != m_Includes.End();
			it++ )
	{
		Node * fn = FBuild::Get().GetDependencyGraph().FindNode( *it );
		if ( fn == nullptr )
		{
			fn = ng.CreateFileNode( *it );
		}
		else if ( fn->IsAFile() == false )
		{
			FLOG_ERROR( "'%s' is not a FileNode (type: %s)", fn->GetName().Get(), fn->GetTypeName() );
			return false;
		}
		m_DynamicDependencies.Append( fn );
	}

	return true;
}

// DoBuildMSCL_NoCache
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuildMSCL_NoCache()
{
	// Format compiler args string
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, false );
	fullArgs += " /showIncludes"; // we'll extract dependency information from this

	EmitCompilationMessage( fullArgs );

//  TODO:C Use this
//	if ( BuildFinalOutput( fullArgs ) == false )
//	{
//		return NODE_RESULT_FAILED; // BuildFinalOutput will have emitted error
//	}

	// spawn the process
	CompileHelper ch;
	if ( !ch.SpawnCompiler( GetName(), GetCompiler(), fullArgs, true ) ) // use response file for MSVC
	{
		return NODE_RESULT_FAILED; // SpawnCompiler has logged error
	}

	// compiled ok, try to extract includes
	if ( ProcessIncludesMSCL( ch.GetOut().Get(), ch.GetOutSize(), false) == false )
	{
		return NODE_RESULT_FAILED; // ProcessIncludesMSCL will have emitted an error
	}

	// record new file time
	m_Stamp = FileIO::GetFileLastWriteTime( m_Name );

	return NODE_RESULT_OK;
}

// DoBuildWithPreProcessor
//------------------------------------------------------------------------------
Node::BuildResult ObjectNode::DoBuildWithPreProcessor( Job * job )
{
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, true );

	if ( BuildPreprocessedOutput( fullArgs, job ) == false )
	{
		return NODE_RESULT_FAILED; // BuildPreprocessedOutput will have emitted an error
	}

	// preprocessed ok, try to extract includes
	if ( ProcessIncludesWithPreProcessor( job ) == false )
	{
		return NODE_RESULT_FAILED; // ProcessIncludesWithPreProcessor will have emitted an error
	}

	// calculate the cache entry lookup
	AStackString<> cacheFile;
	if ( FBuild::Get().GetOptions().m_UseCacheRead ||
		 FBuild::Get().GetOptions().m_UseCacheWrite )
	{
		GetCacheName( job->GetData(), job->GetDataSize(), cacheFile );
	}

	// try to get from cache
	if ( RetrieveFromCache( cacheFile ) )
	{
		return NODE_RESULT_OK_CACHE;
	}

    // TODO:B Smarter logic to abandon cacheing if precompiled header could be used
    // locally (or if in read-only cache mode)

	// can we do the rest of the work remotely?
	if ( GetFlag( FLAG_CAN_BE_DISTRIBUTED ) &&
		 FBuild::Get().GetOptions().m_AllowDistributed &&
		 JobQueue::Get().GetDistributableJobsMemUsage() < ( 512 * MEGABYTE ) )
	{
		// compress job data
		Compressor c;
		c.Compress( job->GetData(), job->GetDataSize() );
		size_t compressedSize = c.GetResultSize();
		job->OwnData( c.ReleaseResult(), compressedSize, true );

		// yes... re-queue for secondary build
		return NODE_RESULT_NEED_SECOND_BUILD_PASS;
	}

	// can't do the work remotely, so do it right now
	Node::BuildResult result = DoBuildWithPreProcessor2( job );
	if ( result != Node::NODE_RESULT_OK )
	{
		return result;
	}

	// record new file time
	m_Stamp = FileIO::GetFileLastWriteTime( m_Name );

	if ( m_Stamp && FBuild::Get().GetOptions().m_UseCacheWrite )
	{
		WriteToCache( cacheFile );
	}

	return Node::NODE_RESULT_OK;
}

// DoBuildWithPreProcessor2
//------------------------------------------------------------------------------
Node::BuildResult ObjectNode::DoBuildWithPreProcessor2( Job * job )
{
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, false );

	bool usePreProcessedOutput = true;

	// should never use preprocessor if using CLR
	ASSERT( GetFlag( FLAG_USING_CLR ) == false );
	
	if ( usePreProcessedOutput )
	{
		AStackString<> tmpFileName;
		if ( WriteTmpFile( job, tmpFileName ) == false )
		{
			return NODE_RESULT_FAILED; // WriteTmpFile will have emitted an error
		}

		// use preprocesed output path as input
		Node * sourceFile = GetSourceFile();
		fullArgs.Replace( sourceFile->GetName().Get(), tmpFileName.Get() );

		// remove pch related flags ('Yu' and 'Fp')
		if ( GetFlag( FLAG_USING_PCH ) )
		{
			Array< AString > args;
			fullArgs.Tokenize( args );
			fullArgs.Clear();
			auto iter = args.Begin();
			const auto end = args.End();
			for ( ; iter != end; ++iter )
			{
				if ( iter->BeginsWith( "/Yu" ) ||
					 iter->BeginsWith( "/Fp" ) )
				{
					continue;
				}
				if ( iter != args.Begin() )
				{
					fullArgs += ' ';
				}
				fullArgs += *iter;
			}
		}

	}

	if ( BuildFinalOutput( job, fullArgs ) == false )
	{
		return NODE_RESULT_FAILED; // BuildFinalOutput will have emitted error
	}
	return NODE_RESULT_OK;
}

// DoBuildOther
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuildOther()
{
	// Format compiler args string
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, false );

	EmitCompilationMessage( fullArgs );

	// spawn the process
	CompileHelper ch;
	if ( !ch.SpawnCompiler( GetName(), GetCompiler(), fullArgs, false ) ) // don't use response file
	{
		return NODE_RESULT_FAILED; // compile has logged error
	}

	// record new file time
	m_Stamp = FileIO::GetFileLastWriteTime( m_Name );

	return NODE_RESULT_OK;
}

// ProcessIncludesMSCL
//------------------------------------------------------------------------------
bool ObjectNode::ProcessIncludesMSCL( const char * output, uint32_t outputSize, bool preprocessed )
{
	Timer t;

	{
		ASSERT( output && outputSize );

		CIncludeParser parser;
		bool result = preprocessed ? parser.ParseMSCL_Preprocessed( output, outputSize )
								   : parser.ParseMSCL_Output( output, outputSize );
		if ( result == false )
		{
			FLOG_ERROR( "Failed to process includes for '%s'", GetName().Get() );
			return false;
		}

		// record that we have a list of includes
		// (we need a flag because we can't use the array size
		// as a determinator, because the file might not include anything)
		m_Includes.Clear();
		parser.SwapIncludes( m_Includes );
	}

	FLOG_INFO( "Process Includes:\n - File: %s\n - Time: %u ms\n - Num : %u", m_Name.Get(), uint32_t( t.GetElapsedMS() ), uint32_t( m_Includes.GetSize() ) );

	return true;
}

// ProcessIncludesWithPreProcessor
//------------------------------------------------------------------------------
bool ObjectNode::ProcessIncludesWithPreProcessor( Job * job )
{
	Timer t;

	{
		const char  * output = (char *)job->GetData();
		const size_t outputSize = job->GetDataSize();

		ASSERT( output && outputSize );

		CIncludeParser parser;
		bool result = GetFlag( FLAG_MSVC ) ? parser.ParseMSCL_Preprocessed( output, outputSize )
										   : parser.ParseGCC_Preprocessed( output, outputSize );
		if ( result == false )
		{
			FLOG_ERROR( "Failed to process includes for '%s'", GetName().Get() );
			return false;
		}

		// record that we have a list of includes
		// (we need a flag because we can't use the array size
		// as a determinator, because the file might not include anything)
		m_Includes.Clear();
		parser.SwapIncludes( m_Includes );
	}

	FLOG_INFO( "Process Includes:\n - File: %s\n - Time: %u ms\n - Num : %u", m_Name.Get(), uint32_t( t.GetElapsedMS() ), uint32_t( m_Includes.GetSize() ) );

	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * ObjectNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD_DEPS( 3,			staticDeps );
	NODE_LOAD_DEPS( 0,			dynamicDeps );
	NODE_LOAD( uint32_t,		flags );
	NODE_LOAD( AStackString<>,	compilerArgs );
	NODE_LOAD( AStackString<>,	objExtensionOverride );
	NODE_LOAD_DEPS( 0,			compilerForceUsing )

	// we are making inferences from the size of the staticDeps
	// ensure we catch if those asumptions break
	size_t numStaticDepsExcludingForceUsing = staticDeps.GetSize() - compilerForceUsing.GetSize();
	ASSERT( ( numStaticDepsExcludingForceUsing == 2 ) ||
 			( numStaticDepsExcludingForceUsing == 3 ) );

	ASSERT( staticDeps.GetSize() >= 2 );
	Node * compiler = staticDeps[ 0 ];
	Node * staticDepNode = staticDeps[ 1 ];

	Node * precompiledHeader = nullptr;
	if ( numStaticDepsExcludingForceUsing == 3 )
	{
		precompiledHeader = staticDeps[ 2 ];
	}

	Node * on = nullptr;
	if ( remote )
	{
		AStackString<> dummyName( "$$dummyName$$" );
		on = new ObjectNode( dummyName, staticDepNode, compiler, compilerArgs, precompiledHeader, flags, compilerForceUsing );
	}
	else
	{
		NodeGraph & ng = FBuild::Get().GetDependencyGraph();
		on = ng.CreateObjectNode( name, staticDepNode, compiler, compilerArgs, precompiledHeader, flags, compilerForceUsing );
	}

	ObjectNode * objNode = on->CastTo< ObjectNode >();
	objNode->m_DynamicDependencies.Swap( dynamicDeps );
	objNode->m_ObjExtensionOverride = objExtensionOverride;

	return objNode;
}

// DetermineFlags
//------------------------------------------------------------------------------
/*static*/ uint32_t ObjectNode::DetermineFlags( const AString & compiler, const AString & args )
{
	uint32_t flags = 0;

	// Compiler Type
	if ( compiler.EndsWithI( "\\cl.exe" ) ||
		 compiler.EndsWithI( "\\cl" ) )
	{
		flags |= ObjectNode::FLAG_MSVC;
	}
	else if ( compiler.EndsWithI( "gcc.exe" ) || 
			  compiler.EndsWithI( "gcc" ) ||
			  compiler.EndsWithI( "g++.exe" ) ||
			  compiler.EndsWithI( "g++" ) )
	{
		flags |= ObjectNode::FLAG_GCC;
		flags |= ObjectNode::FLAG_CAN_BE_CACHED;
		flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;

		// TODO:B GCC arguments checks for PCH files
	}
	else if ( compiler.EndsWithI( "\\ps3ppusnc.exe" ) ||
			  compiler.EndsWithI( "\\ps3ppusnc" ) )
	{
		flags |= ObjectNode::FLAG_SNC;
		flags |= ObjectNode::FLAG_CAN_BE_CACHED;
		flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;

		// TODO:B SNC arguments checks for PCH files
	}
	else if ( compiler.EndsWithI( "clang++.exe" ) ||
			  compiler.EndsWithI( "clang++" ) )
	{
		flags |= ObjectNode::FLAG_CLANG;
		flags |= ObjectNode::FLAG_CAN_BE_CACHED;
		flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;
	}

	// Check MS compiler options
	if ( flags & ObjectNode::FLAG_MSVC )
	{
		bool usingZ7DebugFormat = false;
		bool usingCLR = false;
		bool usingWinRT = false;

		Array< AString > tokens;
		args.Tokenize( tokens );
		const auto end = tokens.End();
		for ( auto it = tokens.Begin(); it != end; ++it )
		{
			const AString & token = *it;

			if ( token == "/Z7" )
			{
				usingZ7DebugFormat = true;
			}
			else if ( token == "/clr" )
			{
				usingCLR = true;
				flags |= ObjectNode::FLAG_USING_CLR;
			}
			else if ( token == "/ZW" )
			{
				usingWinRT = true;
			}
			else if ( token.BeginsWith( "/Yu" ) )
			{
				flags |= ObjectNode::FLAG_USING_PCH;
			}
			else if ( token.BeginsWith( "/Yc" ) )
			{
				flags |= ObjectNode::FLAG_CREATING_PCH;
			}
		}

		// 1) clr code cannot be distributed due to a compiler bug where the preprocessed using
		// statements are truncated
		// 2) code consuming the windows runtime cannot be cached due to preprocessing weirdness
		// 3) distribution requires Z7 format
		// 4) pch files are machine specific
		if ( usingZ7DebugFormat && !usingWinRT && !usingCLR && !( flags & ObjectNode::FLAG_CREATING_PCH ) )
		{
			flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;
			flags |= ObjectNode::FLAG_CAN_BE_CACHED;
		}
	}

	return flags;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool ObjectNode::Save( IOStream & stream, bool remote ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE_DEPS( m_StaticDependencies );
	NODE_SAVE_DEPS( m_DynamicDependencies );
	NODE_SAVE( m_Flags );
	NODE_SAVE( m_CompilerArgs );
	NODE_SAVE( m_ObjExtensionOverride );
	NODE_SAVE_DEPS( m_CompilerForceUsing )

	return true;
}

// GetObjExtension
//------------------------------------------------------------------------------
const char * ObjectNode::GetObjExtension() const
{
	if ( m_ObjExtensionOverride.IsEmpty() )
	{
		return ".obj";
	}
	return m_ObjExtensionOverride.Get();
}

// DumpOutput
//------------------------------------------------------------------------------
/*static*/ void ObjectNode::DumpOutput( const char * data, uint32_t dataSize, const AString & name )
{
	if ( ( data != nullptr ) && ( dataSize > 0 ) )
	{
		Array< AString > exclusions( 2, false );
		exclusions.Append( AString( "Note: including file:" ) );
		exclusions.Append( AString( "#line" ) );

		AStackString<> msg;
		msg.Format( "FAILED: %s\n", name.Get() );

		AutoPtr< char > mem( (char *)Alloc( dataSize + msg.GetLength() ) );
		memcpy( mem.Get(), msg.Get(), msg.GetLength() );
		memcpy( mem.Get() + msg.GetLength(), data, dataSize );

		Node::DumpOutput( mem.Get(), dataSize + msg.GetLength(), &exclusions );
	}
}

// GetCacheName
//------------------------------------------------------------------------------
void ObjectNode::GetCacheName( const void * memOut, size_t memOutSize,
										  AString & cacheName ) const
{
	Timer t;

	// hash the pre-processed intput data
	uint32_t a = CRC32::Calc( memOut, memOutSize );

	// hash the build "environment"
	uint32_t b = CRC32::Start();
	b = CRC32::Update( b, m_CompilerArgs.Get(), m_CompilerArgs.GetLength() );
	// TODO:B Exclude preprocessor control defines (the preprocessed input has considered those already)
	// TODO:B - Include compiler version/compiler hash
	b = CRC32::Stop( b );

	FBuild::Get().GetCacheFileName( a, b, cacheName );

	FLOG_INFO( "Cache hash: %u ms - %u kb '%s'\n", 
				uint32_t( t.GetElapsedMS() ), 
				uint32_t( memOutSize / KILOBYTE ), 
				cacheName.Get() );
}

// RetrieveFromCache
//------------------------------------------------------------------------------
bool ObjectNode::RetrieveFromCache( const AString & cacheFileName )
{
	if ( FBuild::Get().GetOptions().m_UseCacheRead == false )
	{
		return false;
	}

	Timer t;
	FileStream cacheFile;
	if ( cacheFile.Open( cacheFileName.Get(), FileStream::READ_ONLY ) )
	{
		const size_t cacheFileSize = (size_t)cacheFile.GetFileSize();
		AutoPtr< char > mem( (char *)::Alloc( cacheFileSize ) );
		if ( cacheFile.Read( mem.Get(), cacheFileSize ) == cacheFileSize )
		{
			// we expect a cache file header indicating compression status
			const uint32_t headerVal = *( (uint32_t *)mem.Get() );
			ASSERT( ( headerVal == 'OBJ0' ) || ( headerVal == 'OBJ1' ) );

			FileStream objFile;
			if ( !objFile.Open( m_Name.Get(), FileStream::WRITE_ONLY ) )
			{
				FLOG_ERROR( "Failed to open local file during cache retrieval '%s'", m_Name.Get() );
				return false;
			}

			// header contains compressed state
			const bool compressed = ( headerVal == 'OBJ1' );
			const void * data = ( mem.Get() + sizeof( uint32_t ) );
			size_t dataSize = cacheFileSize - sizeof( uint32_t );
			Compressor c;
			if ( compressed )
			{
				c.Decompress( data );
				data = c.GetResult();
				dataSize = c.GetResultSize();
			}

			if ( objFile.Write( data, dataSize ) != dataSize )
			{
				FLOG_ERROR( "Failed to write to local file during cache retrieval '%s'", m_Name.Get() );
				return false;
			}

			// Get current "system time" and convert to "file time"
			SYSTEMTIME st;
			FILETIME ft;
			GetSystemTime( &st );
			if ( FALSE == SystemTimeToFileTime( &st, &ft ) )
			{
				FLOG_ERROR( "Failed to convert file time after cache hit '%s' (%u)", m_Name.Get(), Env::GetLastErr() );
				return false;
			}
			uint64_t fileTimeNow = ( (uint64_t)ft.dwLowDateTime | ( (uint64_t)ft.dwHighDateTime << 32 ) );
	
			// set the time on the local file
			if ( objFile.SetLastWriteTime( fileTimeNow ) == false )
			{
				FLOG_ERROR( "Failed to set timestamp on file after cache hit '%s' (%u)", m_Name.Get(), Env::GetLastErr() );
				return false;
			}

			objFile.Close();

			FileIO::WorkAroundForWindowsFilePermissionProblem( m_Name );

			// the file time we set and local file system might have different 
			// granularity for timekeeping, so we need to update with the actual time written
			m_Stamp = FileIO::GetFileLastWriteTime( m_Name );

			FLOG_INFO( "Cache hit: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
			SetStatFlag( Node::STATS_CACHE_HIT );
			return true;
		}
	}

	FLOG_INFO( "Cache miss: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
	SetStatFlag( Node::STATS_CACHE_MISS );
	return false;
}

// WriteToCache
//------------------------------------------------------------------------------
void ObjectNode::WriteToCache( const AString & cacheFileName )
{
	ASSERT( !cacheFileName.IsEmpty() );

	Timer t;

	// make sure the cache output path exists
	AStackString<> cachePath( cacheFileName.Get(), cacheFileName.FindLast( '\\' ) );
	if ( !FileIO::EnsurePathExists( cachePath ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
		return;
	}

	// open the compiled object which we want to store to the cache
	FileStream objFile;
	if ( !objFile.Open( m_Name.Get(), FileStream::READ_ONLY ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
		return;
	}

	// open output cache (tmp) file
	AStackString<> cacheFileTmpName( cacheFileName );
	cacheFileTmpName += ".tmp";
	FileStream cacheTmpFile;
	if( !cacheTmpFile.Open( cacheFileTmpName.Get(), FileStream::WRITE_ONLY ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
		return;
	}

	// read obj all into memory
	const size_t objFileSize = (size_t)objFile.GetFileSize();
	AutoPtr< char > mem( (char *)::Alloc( objFileSize ) );
	if ( objFile.Read( mem.Get(), objFileSize ) != objFileSize )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
		return;
	}

	// try to compress
	Compressor c;
	const bool compressed = c.Compress( mem.Get(), objFileSize );
	const void * data = compressed ? c.GetResult() : mem.Get();
	const size_t dataSize = compressed ? c.GetResultSize() : objFileSize;

	// cache file contains a header to indicate compression
	const uint32_t headerVal = compressed ? 'OBJ1' : 'OBJ0';

	// write header and data
	bool cacheTmpWriteOk = cacheTmpFile.Write( headerVal ) && ( cacheTmpFile.Write( data, dataSize ) == dataSize );

	cacheTmpFile.Close();

	if ( !cacheTmpWriteOk )
	{
		// failed to write to cache tmp file
		FileIO::FileDelete( cacheFileTmpName.Get() ); // try to cleanup failure
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
		return;
	}

	// rename tmp file to real file
	if ( FileIO::FileMove( cacheFileTmpName, cacheFileName ) == false )
	{
		// try to delete (possibly) existing file
		FileIO::FileDelete( cacheFileName.Get() );

		// try rename again
		if ( FileIO::FileMove( cacheFileTmpName, cacheFileName ) == false )
		{
			// problem renaming file
			FileIO::FileDelete( cacheFileTmpName.Get() ); // try to cleanup tmp file
			FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
			return;
		}
	}

	// cache store complete
	FLOG_INFO( "Cache store: %u ms '%s'\n", uint32_t( t.GetElapsedMS() ), cacheFileName.Get() );
	SetStatFlag( Node::STATS_CACHE_STORE );
}

// EmitCompilationMessage
//------------------------------------------------------------------------------
void ObjectNode::EmitCompilationMessage( const AString & fullArgs ) const
{
	// print basic or detailed output, depending on options
	// we combine everything into one string to ensure it is contiguous in
	// the output
	AStackString<> output;
	output += "Obj: ";
	output += GetName();
	output += '\n';
	if ( FLog::ShowInfo() )
	{
		output += GetCompiler()->GetName();
		output += ' ';
		output += fullArgs;
		output += '\n';
	}
	FLOG_BUILD( "%s", output.Get() );
}

// BuildFullArgs
//------------------------------------------------------------------------------
void ObjectNode::BuildFullArgs( AString & fullArgs, bool preProcess ) const
{
	Array< AString > tokens( 1024, true );
	m_CompilerArgs.Tokenize( tokens );
	fullArgs.Clear();

	bool skipNextToken = false;

	auto const end = tokens.End();
	for ( auto it = tokens.Begin(); it!=end; ++it )
	{
		if ( skipNextToken )
		{
			skipNextToken = false;
			continue;
		}

		const AString & token = *it;

		// -o removal for preprocessor
		if ( preProcess )
		{
			if ( GetFlag( FLAG_GCC ) || GetFlag( FLAG_SNC ) || GetFlag( FLAG_CLANG ) )
			{
				if ( token.BeginsWith( "-o" ) )
				{
					// if -o is by itself, the next token is the path
					if ( token == "-o" )
					{
						// "-o fileName" format
						skipNextToken = true; // skip next as well as this one
					}
					else
					{
						// "-ofileName" format
					}
					continue; // skip this token in both cases
				}

				if ( token == "-c" )
				{
					continue; // remove -c (compile) option when using preprocessor only
				}
			}
		}

		// remove includes (-I) for second pass with Clang
		if ( !preProcess && GetFlag( FLAG_CLANG ) )
		{
			if ( token.BeginsWith( "-I" ) )
			{
				if ( token == "-I" )
				{
					// "-I include" format
					skipNextToken = true;
				}
				else
				{
					// "-Iinclude" format
				}
				continue; // skip this token in both cases
			}
		}

		// %1 -> InputFile
		const char * found = token.Find( "%1" );
		if ( found )
		{
			fullArgs += AStackString<>( token.Get(), found );
			fullArgs += GetSourceFile()->GetName();
			fullArgs += AStackString<>( found + 2, token.GetEnd() );
			fullArgs += ' ';
			continue;
		}

		// %2 -> OutputFile
		found = token.Find( "%2" );
		if ( found )
		{
			fullArgs += AStackString<>( token.Get(), found );
			fullArgs += m_Name;
			fullArgs += AStackString<>( found + 2, token.GetEnd() );
			fullArgs += ' ';
			continue;
		}

		// %3 -> PrecompiledHeader Obj
		if ( GetFlag( FLAG_MSVC ) )
		{
			found = token.Find( "%3" );
			if ( found )
			{
				// handle /Option:%3 -> /Option:A
				fullArgs += AStackString<>( token.Get(), found );
				fullArgs += m_Name;
				fullArgs += GetObjExtension(); // convert 'PrecompiledHeader.pch' to 'PrecompiledHeader.pch.obj'
				fullArgs += AStackString<>( found + 2, token.GetEnd() );
				fullArgs += ' ';
				continue;
			}
		}

		// %4 -> CompilerForceUsing list
		if ( GetFlag( FLAG_MSVC ) )
		{
			found = token.Find( "%4" );
			if ( found )
			{
				AStackString<> pre( token.Get(), found );
				AStackString<> post( found + 2, token.GetEnd() );
				ExpandTokenList( m_CompilerForceUsing, fullArgs, pre, post );
				fullArgs += ' ';
				continue;
			}
		}
				
		// untouched token
		fullArgs += token;
		fullArgs += ' ';
	}

	if ( preProcess )
	{
		if ( GetFlag( FLAG_MSVC ) )
		{
			fullArgs += "/E"; // run pre-processor only
		}
		else
		{
			ASSERT( GetFlag( FLAG_GCC ) || GetFlag( FLAG_SNC ) || GetFlag( FLAG_CLANG ) );
			fullArgs += "-E"; // run pre-processor only
		}
	}
}

// ExpandTokenList
//------------------------------------------------------------------------------
void ObjectNode::ExpandTokenList( const Array< Node * > & nodes, AString & fullArgs, const AString & pre, const AString & post ) const
{
	const auto end = nodes.End();
	for ( auto it = nodes.Begin(); it != end; ++it )
	{
		Node * n = *it;

		fullArgs += pre;
		fullArgs += n->GetName();
		fullArgs += post;
		fullArgs += ' ';
	}
}

// BuildPreprocessedOutput
//------------------------------------------------------------------------------
bool ObjectNode::BuildPreprocessedOutput( const AString & fullArgs, Job * job ) const
{
	EmitCompilationMessage( fullArgs );

	// spawn the process
	CompileHelper ch( false ); // don't handle output (we'll do that)
	const bool useResponseFile = GetFlag( FLAG_MSVC ) || GetFlag( FLAG_GCC ) || GetFlag( FLAG_SNC ) || GetFlag( FLAG_CLANG );
	if ( !ch.SpawnCompiler( GetName(), GetCompiler(), fullArgs, useResponseFile ) )
	{
		// only output errors in failure case
		// (as preprocessed output goes to stdout, normal logging is pushed to
		// stderr, and we don't want to see that unless there is a problem)
		if ( ch.GetResult() != 0 )
		{
			DumpOutput( ch.GetErr().Get(), ch.GetErrSize(), GetName() );
		}

		return false; // SpawnCompiler will have emitted error
	}

	// take a copy of the output because ReadAllData uses huge buffers to avoid re-sizing
	char * memCopy = (char *)::Alloc( ch.GetOutSize() );
	memcpy( memCopy, ch.GetOut().Get(), ch.GetOutSize() );

	job->OwnData( memCopy, ch.GetOutSize() );

	return true;
}

// WriteTmpFile
//------------------------------------------------------------------------------
bool ObjectNode::WriteTmpFile( Job * job, AString & tmpFileName ) const
{
	ASSERT( job->GetData() && job->GetDataSize() );

	Node * sourceFile = GetSourceFile();

	FileStream tmpFile;
	const char * tmpFileExt = sourceFile->GetName().FindLast( '.' ) + 1;

	if ( GetFlag( FLAG_GCC ) )
	{
		// GCC requires preprocessed output to be named a certain way
		// SNC & MSVC like the extension left alone
		// C code (.c) -> .i (NOTE: lower case only)
		// C++ code (.C, .cc, .cp, .cpp, .cxx) -> .ii
		tmpFileExt = ( strcmp( tmpFileExt, "c" ) == 0 ) ? "i" : "ii";
	}

	void const * dataToWrite = job->GetData();
	size_t dataToWriteSize = job->GetDataSize();

	// handle compressed data
	Compressor c; // scoped here so we can access decompression buffer
	if ( job->IsDataCompressed() )
	{
		c.Decompress( dataToWrite );
		dataToWrite = c.GetResult();
		dataToWriteSize = c.GetResultSize();
	}

	WorkerThread::CreateTempFilePath( tmpFileExt, tmpFileName );
	if ( WorkerThread::CreateTempFile( tmpFileName, tmpFile ) == false ) 
	{
		FLOG_ERROR( "Failed to create temp file '%s' to build '%s' (error %u)", tmpFileName.Get(), GetName().Get(), Env::GetLastErr );
		return NODE_RESULT_FAILED;
	}
	if ( tmpFile.Write( dataToWrite, dataToWriteSize ) != dataToWriteSize )
	{
		FLOG_ERROR( "Failed to write to temp file '%s' to build '%s' (error %u)", tmpFileName.Get(), GetName().Get(), Env::GetLastErr );
		return NODE_RESULT_FAILED;
	}
	tmpFile.Close();

	FileIO::WorkAroundForWindowsFilePermissionProblem( tmpFileName );

	return true;
}

// BuildFinalOutput
//------------------------------------------------------------------------------
bool ObjectNode::BuildFinalOutput( Job * job, const AString & fullArgs ) const
{
	// spawn the process
	CompileHelper ch;
	const bool useResponseFile = GetFlag( FLAG_MSVC ) || GetFlag( FLAG_GCC ) || GetFlag( FLAG_SNC ) || GetFlag( FLAG_CLANG );
	if ( !ch.SpawnCompiler( GetName(), GetCompiler(), fullArgs, useResponseFile ) )
	{
		// did spawn fail, or did we spawn and fail to compile?
		if ( ch.GetResult() != 0 )
		{
			// failed to compile

			// for remote jobs, we must serialize the errors to return with the job
			if ( job->IsLocal() == false )
			{
				AutoPtr< char > mem( (char *)::Alloc( ch.GetOutSize() + ch.GetErrSize() ) );
				memcpy( mem.Get(), ch.GetOut().Get(), ch.GetOutSize() );
				memcpy( mem.Get() + ch.GetOutSize(), ch.GetErr().Get(), ch.GetErrSize() );
				job->OwnData( mem.Release(), ( ch.GetOutSize() + ch.GetErrSize() ) );
			}
		}

		return false; // compile has logged error
	}

	return true;
}

// CompileHelper::CONSTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::CompileHelper::CompileHelper( bool handleOutput )
	: m_HandleOutput( handleOutput )
	, m_OutSize( 0 )
	, m_ErrSize( 0 )
	, m_Result( 0 )
{
}

// CompileHelper::DESTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::CompileHelper::~CompileHelper()
{
}

// CompilHelper::SpawnCompiler
//------------------------------------------------------------------------------
bool ObjectNode::CompileHelper::SpawnCompiler( const AString & name, 
											   const Node * compiler,
											   const AString & fullArgs,
											   bool useResponseFile )
{
	// using response file?
	ResponseFile rf;
	AStackString<> responseFileArgs;
	if ( useResponseFile )
	{
		// write args to response file
		if ( !rf.Create( fullArgs ) )
		{
			return false; // Create will have emitted error
		}

		// override args to use response file
		responseFileArgs.Format( "@\"%s\"", rf.GetResponseFilePath().Get() );
	}

	// spawn the process
	if ( false == m_Process.Spawn( compiler->GetName().Get(), 
								   useResponseFile ? responseFileArgs.Get() : fullArgs.Get(),
								   nullptr, // use exe launch dir as the working dir 
								   FBuild::IsValid() ? FBuild::Get().GetEnvironmentString() : nullptr ) )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", name.Get() );
		return false;
	}

	// capture all of the stdout and stderr
	m_Process.ReadAllData( m_Out, &m_OutSize, m_Err, &m_ErrSize );

	// Get result
	ASSERT( !m_Process.IsRunning() );
	m_Result = m_Process.WaitForExit();

	// output any errors (even if succeeded, there might be warnings)
	if ( m_HandleOutput && m_Err.Get() )
	{ 
		DumpOutput( m_Err.Get(), m_ErrSize, name );
	}

	// failed?
	if ( m_Result != 0 )
	{
		// output 'stdout' which may contain errors for some compilers
		if ( m_HandleOutput )
		{
			DumpOutput( m_Out.Get(), m_OutSize, name );
		}

		FLOG_ERROR( "Failed to build Object (error %i) '%s'", m_Result, name.Get() );

		return false;
	}

	return true;
}

//------------------------------------------------------------------------------
