// NodeGraph.h - interface to the dependency graph
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_GRAPH_NODEGRAPH_H
#define FBUILD_GRAPH_NODEGRAPH_H

// Includes
//------------------------------------------------------------------------------
#include "Core/Containers/Array.h"
#include "Core/Strings/AString.h"
#include "Core/Time/Timer.h"

// Forward Declaration
//------------------------------------------------------------------------------
class AliasNode;
class AString;
class CompilerNode;
class CopyNode;
class CSNode;
class DirectoryListNode;
class DLLNode;
class ExeNode;
class ExecNode;
class FileNode;
class IOStream;
class LibraryNode;
class LinkerNode;
class Node;
class ObjectListNode;
class ObjectNode;
class TestNode;
class UnityNode;
class VCXProjectNode;

// NodeGraphHeader
//------------------------------------------------------------------------------
class NodeGraphHeader
{
public:
	inline explicit NodeGraphHeader()
	{
		m_Identifier[ 0 ] = 'N';
		m_Identifier[ 1 ] = 'G';
		m_Identifier[ 2 ] = 'D';
		m_Version = NODE_GRAPH_CURRENT_VERSION;
	}
	inline ~NodeGraphHeader() {}

	enum { NODE_GRAPH_CURRENT_VERSION = 22 };

	bool IsValid() const
	{
		return ( ( m_Identifier[ 0 ] == 'N' ) &&
				 ( m_Identifier[ 1 ] == 'G' ) &&
				 ( m_Identifier[ 2 ] == 'D' ) );
	}
	bool IsCompatibleVersion() const { return m_Version == NODE_GRAPH_CURRENT_VERSION; }
private:
	char		m_Identifier[ 3 ];
	uint8_t		m_Version;
};

// NodeGraph
//------------------------------------------------------------------------------
class NodeGraph
{
public:
	explicit NodeGraph();
	~NodeGraph();

	bool Initialize( const char * bffFile, const char * nodeGraphDBFile );

	bool Load( const char * nodeGraphDBFile, bool & needReparsing );
	bool Load( IOStream & stream, bool & needReparsing );
	bool Save( IOStream & stream ) const;

	// access existing nodes
	Node * FindNode( const AString & nodeName ) const;
	Node * GetNodeByIndex( uint32_t index ) const;

	// create new nodes
	CopyNode * CreateCopyNode( const AString & dstFileName, 
							   Node * sourceFile );
	ExecNode * CreateExecNode( const AString & dstFileName, 
							   FileNode * sourceFile, 
							   FileNode * executable, 
							   const AString & arguments, 
							   const AString & workingDir,
							   int32_t expectedReturnCode );
	FileNode * CreateFileNode( const AString & fileName, bool cleanPath = true );
	DirectoryListNode * CreateDirectoryListNode( const AString & name,
												 const AString & path,
												 const AString & wildCard,
												 bool recursive,
												 const AString & excludePath );
	LibraryNode *	CreateLibraryNode( const AString & libraryName,
									   Array< Node * > & inputNode,
									   FileNode * compilerNode,
									   const AString & compilerArgs,
									   const AString & compilerOutputPath,
									   const AString & linker,
									   const AString & linkerArgs,
									   ObjectNode * precompiledHeader,
									   const Array< Node * > & compilerForceUsing,
									   const Array< Node * > & preBuildDependencies,
									   const Array< Node * > & additionalInputs );
	ObjectNode *	CreateObjectNode( const AString & objectName,
									  Node * inputNode,
									  Node * compilerNode,
									  const AString & compilerArgs,
									  Node * precompiledHeader,
									  uint32_t flags,
									  const Array< Node * > & compilerForceUsing );
	AliasNode *		CreateAliasNode( const AString & aliasName,
									 const Array< Node * > & targets );
	DLLNode *		CreateDLLNode( const AString & linkerOutputName,
								   const Array< Node * > & inputLibraries,
								   const Array< Node * > & otherLibraries,
								   const AString & linker,
								   const AString & linkerArgs,
								   uint32_t flags,
								   const Array< Node * > & assemblyResources,
								   const AString & importLibName );
	ExeNode *		CreateExeNode( const AString & linkerOutputName,
								   const Array< Node * > & inputLibraries,
								   const Array< Node * > & otherLibraries,
								   const AString & linker,
								   const AString & linkerArgs,
								   uint32_t flags,
								   const Array< Node * > & assemblyResources );
	UnityNode *	CreateUnityNode( const AString & unityName,
								 const Array< DirectoryListNode * > & dirNodes,
								 const Array< AString > & files,
								 const AString & outputPath,
								 const AString & outputPattern,
								 uint32_t numUnityFilesToCreate,
								 const AString & precompiledHeader,
								 const Array< AString > & pathsToExclude,
								 const Array< AString > & filesToExclude,
								 bool isolateWritableFiles );

	CSNode * CreateCSNode( const AString & compilerOutput,
						   const Array< Node * > & inputNodes,
						   const AString & compiler,
						   const AString & compilerOptions,
						   const Array< Node * > & extraRefs );
	TestNode * CreateTestNode( const AString & testOutput,
							   FileNode * testExecutable,
							   const AString & arguments,
							   const AString & workingDir );
	CompilerNode * CreateCompilerNode( const AString & executable,
									   const Array< Node * > & extraFiles );
	VCXProjectNode * CreateVCXProjectNode( const AString & projectOutput,
										   const AString & projectBasePath,
										   const Array< DirectoryListNode * > & paths,
										   const Array< AString > & pathsToExclude,
										   const Array< AString > & allowedFileExtensions,
										   const Array< AString > & files,
										   const Array< AString > & filesToExclude,
										   const Array< AString > & configs,
										   const Array< AString > & platforms,
										   const AString & buildCmd,
										   const AString & rebuildCmd,
										   const AString & cleanCmd,
										   const AString & localDebuggerCommandArguments,
										   const AString & localDebuggerWorkingDirectory,
										   const AString & localDebuggerCommand,
										   const AString & localDebuggerEnvironment );
	ObjectListNode * CreateObjectListNode( const AString & listName,
							 const Array< Node * > & inputNodes,
							 FileNode * compiler,
							 const AString & compilerArgs,
							 const AString & compilerOutputPath,
							 ObjectNode * precompiledHeader,
							 const Array< Node * > & compilerForceUsing,
							 const Array< Node * > & preBuildDependencies );

	void DoBuildPass( Node * nodeToBuild );

	static void CleanPath( const AString & name, AString & fullPath );

	// as BFF files are encountered during parsing, we track them
	void AddUsedFile( const AString & fileName, uint64_t timeStamp );
	bool IsOneUseFile( const AString & fileName ) const;
	void SetCurrentFileAsOneUse();
private:
	friend class FBuild;

	void AddNode( Node * node );

	static void BuildRecurse( Node * nodeToBuild );
	static bool CheckDependencies( Node * nodeToBuild, const Array< Node * > & dependencies );
	static void UpdateBuildStatusRecurse( const Node * node, uint32_t & nodesBuiltTime, 
													  uint32_t & totalNodeTime,
													  uint32_t & totalNodes );

	Node * FindNodeInternal( const AString & fullPath ) const;

	struct UsedFile;
	bool ReadHeaderAndUsedFiles( IOStream & nodeGraphStream, Array< UsedFile > & files, bool & compatibleDB ) const;
	uint32_t GetLibEnvVarHash() const;

	// load/save helpers
	static bool SaveRecurse( IOStream & stream, Node * node, Array< bool > & savedNodeFlags );
	bool LoadNode( IOStream & stream );

	enum { NODEMAP_TABLE_SIZE = 65536 };
	Node *			m_NodeMap[ NODEMAP_TABLE_SIZE ];
	Array< Node * > m_AllNodes;
	uint32_t		m_NextNodeIndex;

	Timer m_Timer;

	// each file used in the generation of the node graph is tracked
	struct UsedFile
	{
		explicit UsedFile( const AString & fileName, uint64_t timeStamp ) : m_FileName( fileName ), m_TimeStamp( timeStamp ), m_Once( false ) {}
		AString		m_FileName;
		uint64_t	m_TimeStamp;
		bool		m_Once;
	};
	Array< UsedFile > m_UsedFiles;
};

//------------------------------------------------------------------------------
#endif // FBUILD_GRAPH_NODEGRAPH_H
