// Node.cpp - base interface for dependency graph nodes
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "NodeGraph.h"

#include "Tools/FBuild/FBuildCore/BFF/BFFParser.h"
#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h"

#include "AliasNode.h"
#include "CompilerNode.h"
#include "CopyNode.h"
#include "CSNode.h"
#include "DirectoryListNode.h"
#include "DLLNode.h"
#include "ExeNode.h"
#include "ExecNode.h"
#include "FileNode.h"
#include "LibraryNode.h"
#include "ObjectListNode.h"
#include "ObjectNode.h"
#include "TestNode.h"
#include "UnityNode.h"
#include "VCXProjectNode.h"

#include "Core/Containers/AutoPtr.h"
#include "Core/Env/Env.h"
#include "Core/FileIO/ConstMemoryStream.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Math/CRC32.h"
#include "Core/Math/Murmur3.h"
#include "Core/Process/Thread.h"
#include "Core/Strings/AStackString.h"
#include "Core/Tracing/Tracing.h"

#include <string.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
NodeGraph::NodeGraph()
: m_AllNodes( 1024, true )
, m_NextNodeIndex( 0 )
, m_UsedFiles( 16, true )
{
	memset( m_NodeMap, 0, sizeof( Node * ) * NODEMAP_TABLE_SIZE );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
NodeGraph::~NodeGraph()
{
	Array< Node * >::Iter i = m_AllNodes.Begin();
	Array< Node * >::Iter end = m_AllNodes.End();
	for ( ; i != end; ++i )
	{
		delete ( *i );
	}
}

// Setup
//------------------------------------------------------------------------------
bool NodeGraph::Initialize( const char * bffFile,
							const char * nodeGraphDBFile )
{
	ASSERT( bffFile ); // must be supplied (or left as default)

	ASSERT( m_UsedFiles.IsEmpty() ); // NodeGraph cannot be recycled

	bool needReparsing = false;
	bool ok = Load( nodeGraphDBFile, needReparsing );
	if ( !ok )
	{
		FLOG_ERROR( "Error reading BFF '%s' (corrupt?)", nodeGraphDBFile );
		return false;
	}

	// has something changed that requires us to re-parse the BFFs?
	if ( needReparsing )
	{
		// a change in bff means includes may have changed, so we'll build the list from scratch
		m_UsedFiles.Clear();

		// open the configuration file
		FLOG_INFO( "Loading BFF '%s'", bffFile );
		FileStream bffStream;
		if ( bffStream.Open( bffFile ) == false )
		{
			// missing bff is a fatal problem
			FLOG_ERROR( "Failed to open BFF '%s'", bffFile );
			return false;
		}
		const uint64_t rootBFFTimeStamp = FileIO::GetFileLastWriteTime( AStackString<>( bffFile ) );

		// read entire config into memory
		uint32_t size = (uint32_t)bffStream.GetFileSize();
		AutoPtr< char > data( (char *)::Alloc( size + 1 ) ); // extra byte for null character sentinel
		if ( bffStream.Read( data.Get(), size ) != size )
		{
			FLOG_ERROR( "Error reading BFF '%s'", bffFile );
			return false;
		}

		// re-parse the BFF from scratch, clean build will result
		BFFParser bffParser;
		data.Get()[ size ] = '\0'; // data passed to parser must be NULL terminated
		return bffParser.Parse( data.Get(), size, bffFile, rootBFFTimeStamp ); // pass size excluding sentinel
	}

	return true;
}

// Load
//------------------------------------------------------------------------------
bool NodeGraph::Load( const char * nodeGraphDBFile, bool & needReparsing )
{
	FileStream fs;
	if ( fs.Open( nodeGraphDBFile, FileStream::READ_ONLY ) == false )	
	{
		FLOG_INFO( "BFF file '%s' missing or unopenable (clean build will result).", nodeGraphDBFile );
		needReparsing = true;
		return true; // not opening the file (could be missing) is not an error
	}

	size_t size = (size_t)fs.GetFileSize();
	AutoPtr< void > mem( ::Alloc( size ) );
	if ( fs.Read( mem.Get(), size ) != size )
	{
		return false; // error reading DB
	}

	// read into memory and load from there
	ConstMemoryStream nodeGraphStream( mem.Get(), size );
	if ( !Load( nodeGraphStream, needReparsing ) )
	{
		FLOG_ERROR( "Database is corrupt." )
		return false; // error reading DB
	}

	return true;
}

// Load
//------------------------------------------------------------------------------
bool NodeGraph::Load( IOStream & stream, bool & needReparsing )
{
	bool compatibleDB = true;
	if ( ReadHeaderAndUsedFiles( stream, m_UsedFiles, compatibleDB ) == false )
	{
		return false; // read error
	}

	// old DB version?
	if ( !compatibleDB )
	{
		FLOG_WARN( "Database version has changed (clean build will occur)." );
		needReparsing = true;
		return true;
	}

	// check if any files used have changed
	for ( size_t i=0; i<m_UsedFiles.GetSize(); ++i )
	{
		const AString & fileName = m_UsedFiles[ i ].m_FileName;
		const uint64_t timeStamp = FileIO::GetFileLastWriteTime( fileName );
		if ( timeStamp != m_UsedFiles[ i ].m_TimeStamp )
		{
			FLOG_WARN( "BFF file '%s' has changed (reparsing will occur).", fileName.Get() );
			needReparsing = true;
			return true;
		}
	}

	// TODO:C The serialization of these settings doesn't really belong here (not part of node graph)
	{
		// cachepath
		AStackString<> cachePath;
		if ( stream.Read( cachePath ) == false )
		{
			return false;
		}
		FBuild::Get().SetCachePath( cachePath );

		// environment
		uint32_t envStringSize = 0;
		if ( stream.Read( envStringSize ) == false )
		{
			return false;
		}
		if ( envStringSize > 0 )
		{
			AutoPtr< char > envString( (char *)::Alloc( envStringSize ) );
			if ( stream.Read( envString.Get(), envStringSize ) == false )
			{
				return false;
			}
			AStackString<> libEnvVar;
			if ( stream.Read( libEnvVar ) == false )
			{
				return false;
			}
			FBuild::Get().SetEnvironmentString( envString.Get(), envStringSize, libEnvVar );
		}

		// check if 'LIB' env variable has changed
		uint32_t libEnvVarHashInDB( 0 );
		if ( stream.Read( libEnvVarHashInDB ) == false )
		{
			return false;
		}
		else
		{
			const uint32_t libEnvVarHash = GetLibEnvVarHash();
			if ( libEnvVarHashInDB != libEnvVarHash )
			{
				// make sure the user knows why some things might re-build
				FLOG_WARN( "'LIB' Environment variable has changed - BFF will be re-parsed\n" );
				needReparsing = true;
				return true;
			}
		}

		// worker list
		Array< AString > workerList( 0, true );
		if ( stream.Read( workerList ) == false )
		{
			return false;
		}
		FBuild::Get().SetWorkerList( workerList );
	}

	ASSERT( m_AllNodes.GetSize() == 0 );

	// Read nodes
	uint32_t numNodes;
	if ( stream.Read( numNodes ) == false )
	{
		return false; // error
	}

	m_AllNodes.SetSize( numNodes );
	memset( m_AllNodes.Begin(), 0, numNodes * sizeof( Node * ) );
	for ( uint32_t i=0; i<numNodes; ++i )
	{
		if ( LoadNode( stream ) == false )
		{
			return false; // error
		}
	}

	// sanity check loading
	for ( size_t i=0; i<numNodes; ++i )
	{
		ASSERT( m_AllNodes[ i ] ); // each node was loaded
		ASSERT( m_AllNodes[ i ]->GetIndex() == i ); // index was correctly persisted
	}

	return true;
}

// LoadNode
//------------------------------------------------------------------------------
bool NodeGraph::LoadNode( IOStream & stream )
{
	// load index
	uint32_t nodeIndex( INVALID_NODE_INDEX );
	if ( stream.Read( nodeIndex ) == false )
	{
		return false;
	}

	// sanity check loading (each node saved once)
	ASSERT( m_AllNodes[ nodeIndex ] == nullptr );
	m_NextNodeIndex = nodeIndex;

	// load specifics (create node)
	Node * n = Node::Load( stream, false );
	if ( n == nullptr )
	{
		return false;
	}

	// sanity check node index was correctly restored
	ASSERT( m_AllNodes[ nodeIndex ] == n );
	ASSERT( n->GetIndex() == nodeIndex );

	// load build time
	uint32_t lastTimeToBuild;
	if ( stream.Read( lastTimeToBuild ) == false )
	{
		return false;
	}
	n->SetLastBuildTime( lastTimeToBuild );

	return true;
}

// Save
//------------------------------------------------------------------------------
bool NodeGraph::Save( IOStream & stream ) const
{
	// write header and version
	NodeGraphHeader header;
	if ( stream.Write( (const void *)&header, sizeof( header ) ) == false )
	{
		return false;
	}

	// write used file
	uint32_t numUsedFiles = (uint32_t)m_UsedFiles.GetSize();
	if ( !stream.Write( numUsedFiles ) )
	{
		return false;
	}
	for ( uint32_t i=0; i<numUsedFiles; ++i )
	{
		const AString & fileName = m_UsedFiles[ i ].m_FileName;
		uint32_t fileNameLen( fileName.GetLength() );
		if ( !stream.Write( fileNameLen ) )
		{
			return false;
		}
		if ( stream.Write( fileName.Get(), fileNameLen ) != fileNameLen )
		{
			return false;
		}
		uint64_t timeStamp( m_UsedFiles[ i ].m_TimeStamp );
		if ( !stream.Write( timeStamp ) )
		{
			return false;
		}
	}

	// TODO:C The serialization of these settings doesn't really belong here (not part of node graph)
	{
		// cache path
		if ( stream.Write( FBuild::Get().GetCachePath() ) == false )
		{
			return false;
		}

		// environment
		const uint32_t envStringSize = FBuild::Get().GetEnvironmentStringSize();
		if ( stream.Write( envStringSize ) == false )
		{
			return false;
		}
		if ( envStringSize > 0 )
		{
			const char * envString = FBuild::Get().GetEnvironmentString();
			if (  stream.Write( envString, envStringSize ) == false )
			{
				return false;
			}

			AStackString<> libEnvVar;
			FBuild::Get().GetLibEnvVar( libEnvVar );
			if ( stream.Write( libEnvVar ) == false )
			{
				return false;
			}
		}

		// 'LIB' env var hash
		const uint32_t libEnvVarHash = GetLibEnvVarHash();
		if ( !stream.Write( libEnvVarHash ) )
		{
			return false;
		}

		// worker list
		const Array< AString > & workerList = FBuild::Get().GetWorkerList();
		if ( stream.Write( workerList ) == false )
		{
			return false;
		}
	}

	// Write nodes
	size_t numNodes = m_AllNodes.GetSize();
	if ( stream.Write( (uint32_t)numNodes ) == false )
	{
		return false;
	}

	// save recursively
	Array< bool > savedNodeFlags( numNodes, false );
	savedNodeFlags.SetSize( numNodes );
	memset( savedNodeFlags.Begin(), 0, numNodes );
	for ( size_t i=0; i<numNodes; ++i )
	{
		if ( SaveRecurse( stream, m_AllNodes[ i ], savedNodeFlags ) == false )
		{
			return false;
		}
	}

	// sanity check saving
	for ( size_t i=0; i<numNodes; ++i )
	{
		ASSERT( savedNodeFlags[ i ] == true ); // each node was saved
	}

	return true;
}

// SaveRecurse
//------------------------------------------------------------------------------
/*static*/ bool NodeGraph::SaveRecurse( IOStream & stream, Node * node, Array< bool > & savedNodeFlags )
{
	// ignore any already saved nodes
	uint32_t nodeIndex = node->GetIndex();
	ASSERT( nodeIndex != INVALID_NODE_INDEX );
	if ( savedNodeFlags[ nodeIndex ] )
	{
		return true;
	}

	// static deps
	{
		const Array< Node * > & staticDeps = node->GetStaticDependencies();
		auto end = staticDeps.End();
		for ( auto it = staticDeps.Begin(); it != end; ++it )
		{
			Node * n = *it;
			if ( SaveRecurse( stream, n, savedNodeFlags ) == false )
			{
				return false;
			}
		}
	}

	// dynamic deps
	{
		const Array< Node * > & dynamicDeps = node->GetDynamicDependencies();
		auto end = dynamicDeps.End();
		for ( auto it = dynamicDeps.Begin(); it != end; ++it )
		{
			Node * n = *it;
			if ( SaveRecurse( stream, n, savedNodeFlags ) == false )
			{
				return false;
			}
		}
	}

	// save this node
	ASSERT( savedNodeFlags[ nodeIndex ] == false ); // sanity check recursion

	// save index
	if ( stream.Write( nodeIndex ) == false )
	{
		return false;
	}

	// save node specific data
	if ( Node::Save( stream, node, false ) == false )
	{
		return false;
	}

	// save build time
	uint32_t lastBuildTime = node->GetLastBuildTime();
	if ( stream.Write( lastBuildTime ) == false )
	{
		return false;
	}

	savedNodeFlags[ nodeIndex ] = true; // mark as saved
	return true;
}

// FindNode (AString &)
//------------------------------------------------------------------------------
Node * NodeGraph::FindNode( const AString & nodeName ) const
{
	// try to find node 'as is'
	Node * n = FindNodeInternal( nodeName );
	if ( n )
	{
		return n;
	}

	// the expanding to a full path
	AStackString< 1024 > fullPath;
	CleanPath( nodeName, fullPath );
	return FindNodeInternal( fullPath );
}

// GetNodeByIndex
//------------------------------------------------------------------------------
Node * NodeGraph::GetNodeByIndex( uint32_t index ) const
{
	Node * n = m_AllNodes[ index ];
	ASSERT( n );
	return n;
}

// CreateCopyNode
//------------------------------------------------------------------------------
CopyNode * NodeGraph::CreateCopyNode( const AString & dstFileName, Node * sourceFile )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( sourceFile->IsAFile() );

	AStackString< 1024 > fullPathDst;
	CleanPath( dstFileName, fullPathDst );

	CopyNode * node = new CopyNode( fullPathDst, (FileNode *)sourceFile );
	AddNode( node );
	return node;
}

// CreateCopyNode
//------------------------------------------------------------------------------
ExecNode * NodeGraph::CreateExecNode( const AString & dstFileName, 
									  FileNode * sourceFile, 
									  FileNode * executable, 
									  const AString & arguments,									  
									  const AString & workingDir,
									  int32_t expectedReturnCode )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 512 > fullPath;
	CleanPath( dstFileName, fullPath );

	ExecNode * node = new ExecNode( fullPath, sourceFile, executable, arguments, workingDir, expectedReturnCode );
	AddNode( node );
	return node;
}

// CreateFileNode
//------------------------------------------------------------------------------
FileNode * NodeGraph::CreateFileNode( const AString & fileName, bool cleanPath )
{
	ASSERT( Thread::IsMainThread() );

	FileNode * node;

	if ( cleanPath )
	{
		AStackString< 512 > fullPath;
		CleanPath( fileName, fullPath );
		node = new FileNode( fullPath );
	}
	else
	{
		node = new FileNode( fileName );
	}

	AddNode( node );
	return node;
}

// CreateDirectoryListNode
//------------------------------------------------------------------------------
DirectoryListNode * NodeGraph::CreateDirectoryListNode( const AString & name,
													    const AString & path,
													    const AString & wildCard,
													    bool recursive,
													    const AString & excludePath )
{
	ASSERT( Thread::IsMainThread() );

	// NOTE: DirectoryListNode assumes valid values from here
	// and will assert as such (so we don't check here)

	DirectoryListNode * node = new DirectoryListNode( name, path, wildCard, recursive, excludePath );
	AddNode( node );
	return node;
}

// CreateLibraryNode
//------------------------------------------------------------------------------
LibraryNode * NodeGraph::CreateLibraryNode( const AString & libraryName,
										    Array< Node * > & inputNodes,
											FileNode * compiler,
											const AString & compilerArgs,
											const AString & compilerOutputPath,
											const AString & linker,
											const AString & linkerArgs,
											ObjectNode * precompiledHeader,
											const Array< Node * > & compilerForceUsing,
											const Array< Node * > & preBuildDependencies,
											const Array< Node * > & additionalInputs )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( libraryName, fullPath );

	LibraryNode * node = new LibraryNode( fullPath, inputNodes, compiler, compilerArgs, compilerOutputPath, linker, linkerArgs, precompiledHeader, compilerForceUsing, preBuildDependencies, additionalInputs );
	AddNode( node );
	return node;
}

// CreateObjectNode
//------------------------------------------------------------------------------
ObjectNode * NodeGraph::CreateObjectNode( const AString & objectName,
										  Node * inputNode,
										  Node * compilerNode,
										  const AString & compilerArgs,
										  Node * precompiledHeader,
										  uint32_t flags,
										  const Array< Node * > & compilerForceUsing )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 512 > fullPath;
	CleanPath( objectName, fullPath );

	ObjectNode * node = new ObjectNode( fullPath, inputNode, compilerNode, compilerArgs, precompiledHeader, flags, compilerForceUsing );
	AddNode( node );
	return node;
}

// CreateAliasNode
//------------------------------------------------------------------------------
AliasNode * NodeGraph::CreateAliasNode( const AString & aliasName,
										const Array< Node * > & targets )
{
	ASSERT( Thread::IsMainThread() );

	AliasNode * node = new AliasNode( aliasName, targets );
	AddNode( node );
	return node;
}

// CreateDLLNode
//------------------------------------------------------------------------------
DLLNode * NodeGraph::CreateDLLNode( const AString & linkerOutputName,
									const Array< Node * > & inputLibraries,
								    const Array< Node * > & otherLibraries,
									const AString & linker,
									const AString & linkerArgs,
									uint32_t flags,
									const Array< Node * > & assemblyResources,
									const AString & importLibName )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( inputLibraries.IsEmpty() == false );

	AStackString< 1024 > fullPath;
	CleanPath( linkerOutputName, fullPath );

	DLLNode * node = new DLLNode( fullPath, 
								  inputLibraries,
								  otherLibraries,
								  linker,
								  linkerArgs,
								  flags,
								  assemblyResources,
								  importLibName );
	AddNode( node );
	return node;
}

// CreateExeNode
//------------------------------------------------------------------------------
ExeNode * NodeGraph::CreateExeNode( const AString & linkerOutputName,
									const Array< Node * > & inputLibraries,
								    const Array< Node * > & otherLibraries,
									const AString & linker,
									const AString & linkerArgs,
									uint32_t flags,
									const Array< Node * > & assemblyResources )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( inputLibraries.IsEmpty() == false );

	AStackString< 1024 > fullPath;
	CleanPath( linkerOutputName, fullPath );

	ExeNode * node = new ExeNode( fullPath, 
								  inputLibraries,
								  otherLibraries,
								  linker,
								  linkerArgs,
								  flags,
								  assemblyResources );
	AddNode( node );
	return node;
}

// CreateUnityNode
//------------------------------------------------------------------------------
UnityNode *	NodeGraph::CreateUnityNode( const AString & unityName,
										const Array< DirectoryListNode * > & dirNodes,
										const Array< AString > & files,
										const AString & outputPath,
										const AString & outputPattern,
										uint32_t numUnityFilesToCreate,
										const AString & precompiledHeader,
										const Array< AString > & pathsToExclude,
										const Array< AString > & filesToExclude,
										bool isolateWritableFiles )
{
	ASSERT( Thread::IsMainThread() );

	UnityNode * node = new UnityNode( unityName,
									  dirNodes,
									  files,
									  outputPath,
									  outputPattern,
									  numUnityFilesToCreate,
									  precompiledHeader,
									  pathsToExclude,
									  filesToExclude,
									  isolateWritableFiles );
	AddNode( node );
	return node;
}

CSNode * NodeGraph::CreateCSNode( const AString & compilerOutput,
								  const Array< Node * > & inputNodes,
								  const AString & compiler,
								  const AString & compilerOptions,
								  const Array< Node * > & extraRefs )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( inputNodes.IsEmpty() == false );

	AStackString< 1024 > fullPath;
	CleanPath( compilerOutput, fullPath );

	CSNode * node = new CSNode( fullPath,
								inputNodes,
								compiler,
								compilerOptions,
								extraRefs );
	AddNode( node );
	return node;
}

// CreateTestNode
//------------------------------------------------------------------------------
TestNode * NodeGraph::CreateTestNode( const AString & testOutput,
									  FileNode * testExecutable,
									  const AString & arguments,
									  const AString & workingDir )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( testOutput, fullPath );

	TestNode * node = new TestNode( fullPath,
									testExecutable,
									arguments,
									workingDir );
	AddNode( node );
	return node;
}

// CreateCompilerNode
//------------------------------------------------------------------------------
CompilerNode * NodeGraph::CreateCompilerNode( const AString & executable,
											  const Array< Node * > & extraFiles )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( executable, fullPath );

	CompilerNode * node = new CompilerNode( fullPath, extraFiles );
	AddNode( node );
	return node;
}

// CreateVCXProjectNode
//------------------------------------------------------------------------------
VCXProjectNode * NodeGraph::CreateVCXProjectNode( const AString & projectOutput,
												  const AString & projectBasePath,
												  const Array< DirectoryListNode * > & paths,
												  const Array< AString > & pathsToExclude,
												  const Array< AString > & allowedFileExtensions,
												  const Array< AString > & files,
												  const Array< AString > & filesToExclude,
												  const Array< AString > & configs,
												  const Array< AString > & platforms,
												  const AString & buildCmd,
												  const AString & rebuildCmd,
												  const AString & cleanCmd,
												  const AString & localDebuggerCommandArguments,
												  const AString & localDebuggerWorkingDirectory,
												  const AString & localDebuggerCommand,
												  const AString & localDebuggerEnvironment )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( projectOutput, fullPath );

	VCXProjectNode * node = new VCXProjectNode( fullPath,
												projectBasePath,
												paths,
												pathsToExclude,
												allowedFileExtensions,
												files,
												filesToExclude,
												configs,
												platforms,
												buildCmd,
												rebuildCmd,
												cleanCmd,
												localDebuggerCommandArguments,
												localDebuggerWorkingDirectory,
												localDebuggerCommand,
												localDebuggerEnvironment );
	AddNode( node );
	return node;
}

// CreateObjectListNode
//------------------------------------------------------------------------------
ObjectListNode * NodeGraph::CreateObjectListNode( const AString & listName,
												  const Array< Node * > & inputNodes,
												  FileNode * compiler,
												  const AString & compilerArgs,
												  const AString & compilerOutputPath,
												  ObjectNode * precompiledHeader,
												  const Array< Node * > & compilerForceUsing,
												  const Array< Node * > & preBuildDependencies )
{
	ASSERT( Thread::IsMainThread() );

	ObjectListNode * node = new ObjectListNode( listName,
												inputNodes,
												compiler,
												compilerArgs,
												compilerOutputPath,
												precompiledHeader,
												compilerForceUsing,
												preBuildDependencies );
	AddNode( node );
	return node;

}

// AddNode
//------------------------------------------------------------------------------
void NodeGraph::AddNode( Node * node )
{
	ASSERT( Thread::IsMainThread() );

	ASSERT( node );

	ASSERT( FindNodeInternal( node->GetName() ) == nullptr ); // node name must be unique

	// track in NodeMap
	const uint32_t crc = CRC32::CalcLower( node->GetName() );
	const size_t key = ( crc & 0xFFFF );
	node->m_Next = m_NodeMap[ key ];
	m_NodeMap[ key ] = node;

	// add to regular list
	if ( m_NextNodeIndex == m_AllNodes.GetSize() )
	{
		// normal addition of nodes to the end
		m_AllNodes.Append( node );
	}
	else
	{
		// inserting nodes during database load at specific indices
		ASSERT( m_AllNodes[ m_NextNodeIndex ] == nullptr );
		m_AllNodes[ m_NextNodeIndex ] = node;
	}

	// set index on node
	node->SetIndex( m_NextNodeIndex );
	m_NextNodeIndex = (uint32_t)m_AllNodes.GetSize();
}

// Build
//------------------------------------------------------------------------------
void NodeGraph::DoBuildPass( Node * nodeToBuild )
{
	// do a build pass
	if ( nodeToBuild->GetState() != Node::BUILDING )
	{
		BuildRecurse( nodeToBuild );
	}
}

// BuildRecurse
//------------------------------------------------------------------------------
/*static*/ void NodeGraph::BuildRecurse( Node * nodeToBuild )
{
	ASSERT( nodeToBuild );

	// already building, or queued to build?
	ASSERT( nodeToBuild->GetState() != Node::BUILDING )

	// check pre-build dependencies
	if ( nodeToBuild->GetState() == Node::NOT_PROCESSED )
	{
		// all static deps done?
		bool allDependenciesUpToDate = CheckDependencies( nodeToBuild, nodeToBuild->GetPreBuildDependencies() );
		if ( allDependenciesUpToDate == false )
		{
			return; // not ready or failed
		}

		nodeToBuild->SetState( Node::PRE_DEPS_READY );
	}

	ASSERT( ( nodeToBuild->GetState() == Node::PRE_DEPS_READY ) ||
			( nodeToBuild->GetState() == Node::STATIC_DEPS_READY ) ||
			( nodeToBuild->GetState() == Node::DYNAMIC_DEPS_DONE ) );

	// test static dependencies first
	if ( nodeToBuild->GetState() == Node::PRE_DEPS_READY )
	{
		// all static deps done?
		bool allDependenciesUpToDate = CheckDependencies( nodeToBuild, nodeToBuild->GetStaticDependencies() );
		if ( allDependenciesUpToDate == false )
		{
			return; // not ready or failed
		}

		nodeToBuild->SetState( Node::STATIC_DEPS_READY );
	}

	ASSERT( ( nodeToBuild->GetState() == Node::STATIC_DEPS_READY ) ||
			( nodeToBuild->GetState() == Node::DYNAMIC_DEPS_DONE ) );

	if ( nodeToBuild->GetState() != Node::DYNAMIC_DEPS_DONE )
	{
		// static deps ready, update dynamic deps
		bool forceClean = FBuild::Get().GetOptions().m_ForceCleanBuild;
		if ( nodeToBuild->DoDynamicDependencies( forceClean ) == false )
		{
			nodeToBuild->SetState( Node::FAILED );
			return;		
		}

		nodeToBuild->SetState( Node::DYNAMIC_DEPS_DONE );
	}

	ASSERT( nodeToBuild->GetState() == Node::DYNAMIC_DEPS_DONE );

	// dynamic deps
	{
		// all static deps done?
		bool allDependenciesUpToDate = CheckDependencies( nodeToBuild, nodeToBuild->GetDynamicDependencies() );
		if ( allDependenciesUpToDate == false )
		{
			return; // not ready or failed
		}
	}

	// dependencies are uptodate, so node can now tell us if it needs
	// building
	bool forceClean = FBuild::Get().GetOptions().m_ForceCleanBuild;
	nodeToBuild->SetStatFlag( Node::STATS_PROCESSED );
	if ( nodeToBuild->DetermineNeedToBuild( forceClean ) )
	{
		JobQueue::Get().QueueJob( nodeToBuild );
	}
	else
	{
		nodeToBuild->SetState( Node::UP_TO_DATE );
	}
}

// CheckDependencies
//------------------------------------------------------------------------------
/*static*/ bool NodeGraph::CheckDependencies( Node * nodeToBuild, const Array< Node * > & dependencies )
{
	bool allDependenciesUpToDate = true;

	Array< Node * >::Iter i = dependencies.Begin();
	Array< Node * >::Iter end = dependencies.End();
	for ( ; i < end; ++i )
	{
		Node::State state = ( *i )->GetState();

		// recurse into nodes which have not been processed yet
		if ( state < Node::BUILDING )
		{
			BuildRecurse( *i );
		}

		// dependency is uptodate, nothing more to be done
		state = ( *i )->GetState();
		if ( state == Node::UP_TO_DATE )
		{
			continue;
		}

		allDependenciesUpToDate = false;

		// dependency failed?
		if ( state == Node::FAILED )
		{
			// propogate failure state to this node
			nodeToBuild->SetState( Node::FAILED );
			break;
		}

		// keep trying to progress other nodes...
	}

	return allDependenciesUpToDate;
}

// CleanPath
//------------------------------------------------------------------------------
/*static*/ void NodeGraph::CleanPath( const AString & name, AString & fullPath )
{
	ASSERT( &name != &fullPath );

	char * dst;

	//  - path should be fully qualified
	bool isFullPath = ( ( name.GetLength() > 1 ) && ( name[1] == ':' ) );
	if ( !isFullPath )
	{
		// make a full path by prepending working dir
		const AString & workingDir = FBuild::Get().GetWorkingDir();

		// we're making the assumption that we don't need to clean the workingDir
		ASSERT( workingDir.Find( '/' ) == nullptr );
		ASSERT( workingDir.Find( "\\\\" ) == nullptr );

		// build the start of the path
		fullPath = workingDir;
		fullPath += '\\';

		// concatenate
		uint32_t len = fullPath.GetLength();

		// make sure the dest will be big enough for the extra stuff
		fullPath.SetLength( fullPath.GetLength() + name.GetLength() ); 

		// set the output (which maybe a newly allocated ptr)
		dst = fullPath.Get() + len;
	}
	else
	{
		// make sure the dest will be big enough
		fullPath.SetLength( name.GetLength() ); 

		// copy from the start
		dst = fullPath.Get();
	}

	// the untrusted part of the path we need to copy/fix
	const char * src = name.Get();
	const char * const srcEnd = name.GetEnd();

	// clean slashes
	char lastChar = '\\'; // consider first item to follow a path (so "..\file.dat" works)
	while ( src < srcEnd )
	{
		const char thisChar = *src;

		// hit a slash?
		if ( ( thisChar == '\\' ) || ( thisChar == '/' ) )
		{
			// write it the correct way
			*dst = '\\';
			dst++;

			// skip until non-slashes
			while ( ( *src == '\\' ) || ( *src == '/' ) )
			{
				src++;
			}
			lastChar = '\\';
			continue;
		}
		else if ( thisChar == '.' )
		{
			if ( lastChar == '\\' ) // fixed up slash, so we only need to check backslash
			{
				// check for \.\ (or \./)
				char nextChar = *( src + 1 );
				if ( ( nextChar == '\\' ) || ( nextChar == '/' ) )
				{
					src++; // skip . and slashes
					while ( ( *src == '\\') || ( *src == '/' ) )
					{
						++src;
					}
					continue; // leave lastChar as-is, since we added nothing
				}

				// check for \..\ (or \../)
				if ( nextChar == '.' )
				{
					nextChar = *( src + 2 );
					if ( ( nextChar == '\\' ) || ( nextChar == '/' ) )
					{
						src+=2; // skip .. and slashes
						while ( ( *src == '\\') || ( *src == '/' ) )
						{
							++src;
						}

						if ( dst > fullPath.Get() + 3 )
						{
							--dst; // remove slash

							// remove one level of path (but never past first slash ( e.g. leave at least c:\ ))
							while ( dst > fullPath.Get() + 3 )
							{
								--dst;
								if ( *dst == '\\' ) // only need to check for cleaned slashes
								{
									++dst; // keep this slash
									break;
								}
							}
						}

						continue;
					}
				}
			}
		}

		// write non-slash character
		*dst++ = *src++;
		lastChar = thisChar;
	}

	// correct length of destination
	fullPath.SetLength( (uint16_t)( dst - fullPath.Get() ) );
	ASSERT( AString::StrLen( fullPath.Get() ) == fullPath.GetLength() );

	// sanity checks
	ASSERT( fullPath.Find( '/' ) == nullptr );
	ASSERT( fullPath.Find( "\\\\" ) == nullptr );
}

// AddUsedFile
//------------------------------------------------------------------------------
void NodeGraph::AddUsedFile( const AString & fileName, uint64_t timeStamp )
{
	const size_t numFiles = m_UsedFiles.GetSize();
	for ( size_t i=0 ;i<numFiles; ++i )
	{
		if ( m_UsedFiles[ i ].m_FileName.CompareI( fileName ) == 0 )
		{
			ASSERT( m_UsedFiles[ i ].m_Once == false ); // should not be trying to add a file a second time
			return;
		}
	}
	m_UsedFiles.Append( UsedFile( fileName, timeStamp ) );
}

// IsOneUseFile
//------------------------------------------------------------------------------
bool NodeGraph::IsOneUseFile( const AString & fileName ) const
{
	const size_t numFiles = m_UsedFiles.GetSize();
	ASSERT( numFiles ); // shouldn't be called if there are no files
	for ( size_t i=0 ;i<numFiles; ++i )
	{
		if ( m_UsedFiles[ i ].m_FileName.CompareI( fileName ) == 0 )
		{
			return m_UsedFiles[ i ].m_Once;
		}
	}

	// file never seen, so it can be included multiple time initially
	// (if we hit a #once during parsing, we'll flag the file then)
	return false;
}

// SetCurrentFileAsOneUse
//------------------------------------------------------------------------------
void NodeGraph::SetCurrentFileAsOneUse()
{
	ASSERT( !m_UsedFiles.IsEmpty() );
	m_UsedFiles[ m_UsedFiles.GetSize() - 1 ].m_Once = true;
}

// FindNodeInternal
//------------------------------------------------------------------------------
Node * NodeGraph::FindNodeInternal( const AString & fullPath ) const
{
	ASSERT( Thread::IsMainThread() );

	const uint32_t crc = CRC32::CalcLower( fullPath );
	const size_t key = ( crc & 0xFFFF );

	Node * n = m_NodeMap[ key ];
	while ( n )
	{
		if ( n->GetNameCRC() == crc )
		{
			if ( n->GetName().CompareI( fullPath ) == 0 )
			{
				return n;
			}
		}
		n = n->m_Next;
	}
	return nullptr;
}

// UpdateBuildStatusRecurse
//------------------------------------------------------------------------------
/*static*/ void NodeGraph::UpdateBuildStatusRecurse( const Node * node, uint32_t & nodesBuiltTime, 
															 uint32_t & totalNodeTime,
															 uint32_t & totalNodes )
{
	totalNodeTime += node->GetLastBuildTime();

	if ( node->GetState() == Node::UP_TO_DATE )
	{
		nodesBuiltTime += node->GetLastBuildTime();
	}

	const Array< Node * > & staticDeps = node->GetStaticDependencies();
	totalNodes += (uint32_t)staticDeps.GetSize();
	for ( Array< Node * >::Iter i = staticDeps.Begin();
		i != staticDeps.End();
		i++ )
	{
		UpdateBuildStatusRecurse( *i, nodesBuiltTime, totalNodeTime, totalNodes );
	}

	const Array< Node * > & dynamicDeps = node->GetDynamicDependencies();
	totalNodes += (uint32_t)dynamicDeps.GetSize();
	for ( Array< Node * >::Iter i = dynamicDeps.Begin();
		i != dynamicDeps.End();
		i++ )
	{
		UpdateBuildStatusRecurse( *i, nodesBuiltTime, totalNodeTime, totalNodes );
	}
}

// ReadHeaderAndUsedFiles
//------------------------------------------------------------------------------
bool NodeGraph::ReadHeaderAndUsedFiles( IOStream & nodeGraphStream, Array< UsedFile > & files, bool & compatibleDB ) const
{
	// check for a valid header
	NodeGraphHeader ngh;
	if ( ( nodeGraphStream.Read( &ngh, sizeof( ngh ) ) != sizeof( ngh ) ) ||
		 ( ngh.IsValid() == false ) )
	{
		return false;
	}

	// check if version is loadable
	if ( ngh.IsCompatibleVersion() == false )
	{
		compatibleDB = false;
		return true;
	}

	uint32_t numFiles = 0;
	if ( !nodeGraphStream.Read( numFiles ) )
	{
		return false;
	}

	for ( uint32_t i=0; i<numFiles; ++i )
	{
		uint32_t fileNameLen( 0 );
		if ( !nodeGraphStream.Read( fileNameLen ) )
		{
			return false;
		}
		AStackString<> fileName;
		fileName.SetLength( fileNameLen ); // handles null terminate
		if ( nodeGraphStream.Read( fileName.Get(), fileNameLen ) != fileNameLen )
		{
			return false;
		}
		uint64_t timeStamp;
		if ( !nodeGraphStream.Read( timeStamp ) )
		{
			return false;
		}

		files.Append( UsedFile( fileName, timeStamp ) );
	}

	return true;
}

// GetLibEnvVarHash
//------------------------------------------------------------------------------
uint32_t NodeGraph::GetLibEnvVarHash() const
{
	// if env var doesn't exist, or is empty
	AStackString<> libVar;
	FBuild::Get().GetLibEnvVar( libVar );
	if ( libVar.IsEmpty() )
	{
		return 0; // return 0 (rather than hash of empty string)
	}

	return Murmur3::Calc32( libVar );
}

//------------------------------------------------------------------------------
