// Node.cpp - base interface for dependency graph nodes
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Node.h"
#include "FileNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/Graph/AliasNode.h"
#include "Tools/FBuild/FBuildCore/Graph/CompilerNode.h"
#include "Tools/FBuild/FBuildCore/Graph/CopyNode.h"
#include "Tools/FBuild/FBuildCore/Graph/CSNode.h"
#include "Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/DLLNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ExeNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ExecNode.h"
#include "Tools/FBuild/FBuildCore/Graph/FileNode.h"
#include "Tools/FBuild/FBuildCore/Graph/LibraryNode.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeProxy.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectNode.h"
#include "Tools/FBuild/FBuildCore/Graph/TestNode.h"
#include "Tools/FBuild/FBuildCore/Graph/UnityNode.h"
#include "Tools/FBuild/FBuildCore/Graph/VCXProjectNode.h"

#include "Core/Containers/Array.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/IOStream.h"
#include "Core/Math/CRC32.h"
#include "Core/Strings/AStackString.h"

// Static Data
//------------------------------------------------------------------------------
/*static*/ const char * const Node::s_NodeTypeNames[] = 
{
	"Proxy",
	"Copy",
	"Directory",
	"Exec",
	"File",
	"Library",
	"Object",
	"Alias",
	"Exe",
	"Unity",
	"C#",
	"Test",
	"Compiler",
	"DLL",
	"VCXProj",
	"ObjectList"
};

// CONSTRUCTOR
//------------------------------------------------------------------------------
Node::Node( const AString & name, Type type, uint32_t controlFlags )
	: m_State( NOT_PROCESSED )
	, m_ControlFlags( controlFlags )
	, m_StatsFlags( 0 )
	, m_Stamp( 0 )
	, m_Type( type )
	, m_Next( nullptr )
	, m_LastBuildTimeMs( 0 )
	, m_ProcessingTime( 0 )
	, m_Name( name )
	, m_Index( INVALID_NODE_INDEX )
{
	m_NameCRC = CRC32::CalcLower( name );

	// Compile time check to ensure name vector is in sync
	CTASSERT( sizeof( s_NodeTypeNames ) / sizeof(const char *) == NUM_NODE_TYPES );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Node::~Node()
{
}

// DoDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ bool Node::DoDynamicDependencies( bool )
{
	return true;
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
bool Node::DetermineNeedToBuild( bool forceClean ) const
{
	if ( forceClean )
	{
		return true;
	}

	// if we don't have a stamp, we are building for the first time
	// (or we're a node that is built every time)
	if ( m_Stamp == 0 )
	{
		// don't output for file nodes, which are always built
		if ( GetType() != Node::FILE_NODE )
		{
			FLOG_INFO( "Need to build '%s' (first time)", GetName().Get() );
		}
		return true;
	}

	if ( IsAFile() )
	{
		uint64_t lastWriteTime = FileIO::GetFileLastWriteTime( m_Name );

		if ( lastWriteTime == 0 )
		{
			// file is missing on disk
			FLOG_INFO( "Need to build '%s' (missing)", GetName().Get() );
			return true;
		}

		if ( lastWriteTime != m_Stamp )
		{
			// on disk file doesn't match our file
			// (modified by some external process)
			FLOG_INFO( "Need to build '%s' (externally modified - stamp = %llu, disk = %llu)", GetName().Get(), m_Stamp, lastWriteTime );
			return true;
		}
	}

	// static deps
	const Array< Node * > & staticDeps = GetStaticDependencies();
	for ( Array< Node * >::ConstIter it = staticDeps.Begin();
		  it != staticDeps.End();
		  it++ )
	{
		Node * n = *it;

		// ignore directories - the derived node should extract what it needs in DoDynamicDependencies
		if ( n->GetType() == Node::DIRECTORY_LIST_NODE )
		{
			continue;
		}

		// ignore unity nodes - the derived node should extract what it needs in DoDynamicDependencies
		if ( n->GetType() == Node::UNITY_NODE )
		{
			continue;
		}

		if ( n->GetType() == Node::UNITY_NODE )
		{
			continue;
		}

		// we're about to compare stamps, so we should be a file (or a file list)
		ASSERT( n->IsAFile() || ( n->GetType() == Node::OBJECT_LIST_NODE ) );

		if ( n->GetStamp() == 0 )
		{
			// file missing - this may be ok, but node needs to build to find out
			FLOG_INFO( "Need to build '%s' (dep missing: '%s')", GetName().Get(), n->GetName().Get() );
			return true;
		}

		if ( n->GetStamp() > m_Stamp )
		{
			// file is newer than us
			FLOG_INFO( "Need to build '%s' (dep is newer: '%s' this = %llu, dep = %llu)", GetName().Get(), n->GetName().Get(), m_Stamp, n->GetStamp() );
			return true;
		}
	}

	// dynamic deps
	const Array< Node * > & dynamicDeps = GetDynamicDependencies();
	for ( Array< Node * >::ConstIter it = dynamicDeps.Begin();
		  it != dynamicDeps.End();
		  it++ )
	{
		Node * n = *it;

		// we're about to compare stamps, so we should be a file (or a file list)
		ASSERT( n->IsAFile() || ( n->GetType() == Node::OBJECT_LIST_NODE ) );

		// should be a file
		if ( n->GetStamp() == 0 )
		{
			// file missing - this may be ok, but node needs to build to find out
			FLOG_INFO( "Need to build '%s' (dep missing: '%s')", GetName().Get(), n->GetName().Get() );
			return true;
		}

		if ( n->GetStamp() > m_Stamp )
		{
			// file is newer than us
			FLOG_INFO( "Need to build '%s' (dep is newer: '%s' this = %llu, dep = %llu)", GetName().Get(), n->GetName().Get(), m_Stamp, n->GetStamp() );
			return true;
		}
	}


	// nothing needs building
	FLOG_INFO( "Up-To-Date '%s'", GetName().Get() );
	return false;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult Node::DoBuild( Job * UNUSED( job ) )
{
	ASSERT( false ); // Derived class is missing implementation
	return Node::NODE_RESULT_FAILED;
}

// DoBuild2
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult Node::DoBuild2( Job * UNUSED( job ) )
{
	ASSERT( false ); // Derived class is missing implementation
	return Node::NODE_RESULT_FAILED;
}

// Finalize
//------------------------------------------------------------------------------
/*virtual*/ bool Node::Finalize()
{
	// most nodes have nothing to do
	return true;
}

//------------------------------------------------------------------------------
bool Node::SaveDepArray( IOStream & fileStream, const Array< Node * > & depArray, bool remote ) const
{
	size_t numDeps = depArray.GetSize();
	if ( fileStream.Write( (uint32_t)numDeps ) == false )
	{
		return false;
	}
	for ( uint32_t i=0; i<numDeps; ++i )
	{
		if ( remote )
		{
			if ( fileStream.Write( depArray[ i ]->GetName() ) == false )
			{
				return false;
			}
		}
		else
		{
			uint32_t index = depArray[ i ]->GetIndex();
			if ( fileStream.Write( index ) == false )
			{
				return false;
			}
		}
	}
	return true;
}

//------------------------------------------------------------------------------
/*static*/ bool Node::LoadDepArray( IOStream & fileStream, Array< Node * > & deps, bool remote )
{
	uint32_t numDeps;
	if ( fileStream.Read( numDeps ) == false )
	{
		return false;
	}
	if ( deps.GetCapacity() < deps.GetSize() + numDeps )
	{
		deps.SetCapacity( deps.GetSize() + numDeps );
	}
	for ( uint32_t i=0; i<numDeps; ++i )
	{
		Node * n = nullptr;
		if ( remote )
		{
			AStackString< 512 > depName;
			if ( fileStream.Read( depName ) == false )
			{
				return false;
			}
			n = new NodeProxy( depName );
		}
		else
		{
			NodeGraph & ng = FBuild::Get().GetDependencyGraph();
			uint32_t index( INVALID_NODE_INDEX );
			if ( fileStream.Read( index ) == false )
			{
				return false;
			}
			n = ng.GetNodeByIndex( index );
			ASSERT( n );
		}
		deps.Append( n );
	}
	return true;
}

// SaveNode
//------------------------------------------------------------------------------
bool Node::SaveNode( IOStream & fileStream, const Node * node ) const
{
	// for null pointer, write an empty string
	if ( node == nullptr )
	{
		if ( fileStream.Write( AString::GetEmpty() ) == false )
		{
			return false;
		}
	}
	else
	{
		// for valid nodes, write the node name
		if ( fileStream.Write( node->GetName() ) == false )
		{
			return false;
		}
	}

	return true;
}

// LoadNode
//------------------------------------------------------------------------------
/*static*/ bool Node::LoadNode( IOStream & stream, Node * & node )
{
	// read the name of the node
	AStackString< 512 > nodeName;
	if ( stream.Read( nodeName ) == false )
	{
		return false;
	}

	// empty name means the pointer was null, which is supported
	if ( nodeName.IsEmpty() )
	{
		node = nullptr;
		return true;
	}

	// find the node by name - this should never fail
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * n = ng.FindNode( nodeName );
	if ( n == nullptr )
	{
		return false;
	}
	node = n;

	return true;
}

// LoadNode (FileNode)
//------------------------------------------------------------------------------
/*static*/ bool Node::LoadNode( IOStream & stream, FileNode * & fileNode )
{
	Node * node;
	if ( !LoadNode( stream, node ) )
	{
		return false;
	}
	if ( !node->IsAFile() )
	{
		return false;
	}
	fileNode = node->CastTo< FileNode >();
	return ( fileNode != nullptr );
}

// EnsurePathExistsForFile
//------------------------------------------------------------------------------
/*static*/ bool Node::EnsurePathExistsForFile( const AString & name )
{
	const char * lastSlash = name.FindLast( '\\' );
	ASSERT( lastSlash ); // should be guaranteed to be a full path
	AStackString<> pathOnly( name.Get(), lastSlash );
	if ( FileIO::EnsurePathExists( pathOnly ) == false )
	{
		FLOG_ERROR( "Failed to create path '%s'", pathOnly.Get() );
		return false;
	}
	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * Node::Load( IOStream & stream, bool remote )
{
	// read type
	uint32_t nodeType;
	if ( stream.Read( nodeType ) == false )
	{
		return nullptr;
	}

	// read stamp (but not for file nodes)
	uint64_t stamp( 0 );
	if ( nodeType != Node::FILE_NODE )
	{
		if ( stream.Read( stamp ) == false )
		{
			return false;
		}
	}

	// read contents
	Node * n = nullptr;
	switch ( (Node::Type)nodeType )
	{
		case Node::PROXY_NODE:			ASSERT( false );								break;
		case Node::COPY_NODE:			n = CopyNode::Load( stream, remote );			break;
		case Node::DIRECTORY_LIST_NODE: n = DirectoryListNode::Load( stream, remote );	break;
		case Node::EXEC_NODE:			n = ExecNode::Load( stream, remote );			break;
		case Node::FILE_NODE:			n = FileNode::Load( stream, remote );			break;
		case Node::LIBRARY_NODE:		n = LibraryNode::Load( stream, remote );		break;
		case Node::OBJECT_NODE:			n = ObjectNode::Load( stream, remote );			break;
		case Node::ALIAS_NODE:			n = AliasNode::Load( stream, remote );			break;
		case Node::EXE_NODE:			n = ExeNode::Load( stream, remote );			break;
		case Node::CS_NODE:				n = CSNode::Load( stream, remote );				break;
		case Node::UNITY_NODE:			n = UnityNode::Load( stream, remote );			break;
		case Node::TEST_NODE:			n = TestNode::Load( stream, remote );			break;
		case Node::COMPILER_NODE:		n = CompilerNode::Load( stream, remote );		break;
		case Node::DLL_NODE:			n = DLLNode::Load( stream, remote );			break;
		case Node::VCXPROJECT_NODE:		n = VCXProjectNode::Load( stream, remote );		break;
		case Node::OBJECT_LIST_NODE:	n = ObjectListNode::Load( stream, remote );		break;
		case Node::NUM_NODE_TYPES:		ASSERT( false );								break;
	}

	ASSERT( n );
	if ( n )
	{
		// set stamp
		n->m_Stamp = stamp;
	}

	return n;
}

// 
//------------------------------------------------------------------------------
/*static*/ bool Node::Save( IOStream & stream, const Node * node, bool remote )
{
	ASSERT( node );

	// save type
	uint32_t nodeType = (uint32_t)node->GetType();
	if ( stream.Write( nodeType ) == false )
	{
		return false;
	}

	// save stamp (but not for file nodes)
	if ( nodeType != Node::FILE_NODE )
	{
		uint64_t stamp = node->GetStamp();
		if ( stream.Write( stamp ) == false )
		{
			return false;
		}
	}

	// save contents
	return node->Save( stream, remote );
}

// ReplaceDummyName
//------------------------------------------------------------------------------
void Node::ReplaceDummyName( const AString & newName )
{
	ASSERT( m_Name == "$$dummyName$$" );
	m_Name = newName;
}

// DumpOutput
//------------------------------------------------------------------------------
/*static*/ void Node::DumpOutput( const char * data, 
								  uint32_t dataSize,
								  const Array< AString > * exclusions )
{
	if ( ( data == nullptr ) || ( dataSize == 0 ) )
	{
		return;
	}

	// preallocate a large buffer
	AString buffer( 1024 * 1024 );

	const char * end = data + dataSize;
	while( data < end )
	{
		// find the limits of the current line
		const char * lineStart = data;
		const char * lineEnd = lineStart;
		while ( lineEnd < end )
		{
			if ( ( *lineEnd == '\r' ) || ( *lineEnd == '\n' ) )
			{
				break;
			}
			lineEnd++;
		}
		if ( lineStart != lineEnd ) // ignore empty
		{
			// make a copy of the line to output
			AStackString< 1024 > copy( lineStart, lineEnd );

			// skip this line?
			bool skip = false;
			if ( exclusions )
			{
				auto iter = exclusions->Begin();
				auto endIter = exclusions->End();
				while ( iter != endIter )
				{
					if ( copy.BeginsWith( *iter ) )
					{
						skip = true;
						break;
					}
					iter++;
				}
			}
			if ( !skip )
			{
				copy.Replace( "\n", "" );
				copy.Replace( "\r", "" );
				copy += '\n';
				buffer += copy;
			}
		}
		data = ( lineEnd + 1 );
	}

	// print everything at once
	FLOG_ERROR_DIRECT( buffer.Get() );
}

//------------------------------------------------------------------------------
