// LinkerNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "LinkerNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/Graph/CopyNode.h"
#include "Tools/FBuild/FBuildCore/Graph/DLLNode.h"
#include "Tools/FBuild/FBuildCore/Graph/FileNode.h"
#include "Tools/FBuild/FBuildCore/Graph/LibraryNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectNode.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Helpers/ResponseFile.h"

#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Process/Process.h"
#include "Core/Strings/AStackString.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
LinkerNode::LinkerNode( const AString & linkerOutputName,
						 const Array< Node * > & inputLibraries,
						 const Array< Node * > & otherLibraries,
						 const AString & linker,
						 const AString & linkerArgs,
						 uint32_t flags,
						 const Array< Node * > & assemblyResources )
: FileNode( linkerOutputName, Node::FLAG_NONE )
, m_Flags( flags )
, m_AssemblyResources( assemblyResources )
, m_OtherLibraries( otherLibraries )
{
	m_LastBuildTimeMs = 20000;

	// depend on everything we'll link together
	ASSERT( inputLibraries.IsEmpty() == false );
	m_StaticDependencies.SetCapacity( inputLibraries.GetSize() + assemblyResources.GetSize() + otherLibraries.GetSize() );
	m_StaticDependencies.Append( inputLibraries );
	m_StaticDependencies.Append( assemblyResources );
	m_StaticDependencies.Append( otherLibraries );	

	// store options we'll need to use dynamically
	m_Linker = linker; // TODO:C This should be a node
	m_LinkerArgs = linkerArgs;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
LinkerNode::~LinkerNode()
{
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult LinkerNode::DoBuild( Job * UNUSED( job ) )
{
	// need building...
	FLOG_BUILD( "%s: %s\n", GetDLLOrExe(), GetName().Get() );

	// Format compiler args string
	AStackString< 4 * KILOBYTE > fullArgs;
	GetFullArgs( fullArgs );

	// use the exe launch dir as the working dir
	const char * workingDir = nullptr;

	const char * environment = FBuild::Get().GetEnvironmentString();

	FLOG_INFO( "%s %s\n", m_Linker.Get(), fullArgs.Get() );

	// handle response file
	ResponseFile rf;
	AStackString<> responseFileArgs;
	const bool useResponseFile = GetFlag( LINK_FLAG_MSVC ) || GetFlag( LINK_FLAG_GCC ) || GetFlag( LINK_FLAG_SNC );
	if ( useResponseFile )
	{
		// write args to response file
		if ( !rf.Create( fullArgs ) )
		{
			return NODE_RESULT_FAILED; // Create will have emitted error
		}

		// override args to use response file
		responseFileArgs.Format( "@\"%s\"", rf.GetResponseFilePath().Get() );
	}

	// we retry if linker crashes
	uint32_t attempt( 0 );

	for (;;)
	{
		++attempt;

		// spawn the process
		Process p;
		bool spawnOK = p.Spawn( m_Linker.Get(),
								useResponseFile ? responseFileArgs.Get() : fullArgs.Get(),
								workingDir,
								environment );

		if ( !spawnOK )
		{
			FLOG_ERROR( "Failed to spawn process for %s creation for '%s'", GetDLLOrExe(), GetName().Get() );
			return NODE_RESULT_FAILED;
		}

		// capture all of the stdout and stderr
		AutoPtr< char > memOut;
		AutoPtr< char > memErr;
		uint32_t memOutSize = 0;
		uint32_t memErrSize = 0;
		p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

		ASSERT( !p.IsRunning() );
		// Get result
		int result = p.WaitForExit();

		if ( result != 0 )
		{
			if ( memOut.Get() ) { FLOG_ERROR_DIRECT( memOut.Get() ); }
			if ( memErr.Get() ) { FLOG_ERROR_DIRECT( memErr.Get() ); }
		}

		// did the executable fail?
		if ( result != 0 )
		{
			// did the linker have an ICE (LNK1000)?
			if ( GetFlag( LINK_FLAG_MSVC ) && ( result == 1000 ) && ( attempt == 1 ) )
			{
				FLOG_WARN( "FBuild: Warning: Linker crashed (LNK1000), retrying '%s'", GetName().Get() );
				continue; // try again
			}

			// some other (genuine) linker failure
			FLOG_ERROR( "Failed to build %s (error %i) '%s'", GetDLLOrExe(), result, GetName().Get() );
			return NODE_RESULT_FAILED;
		}
		else
		{	
			break; // success!
		}
	}


	// record time stamp for next time
	m_Stamp = FileIO::GetFileLastWriteTime( m_Name );
	ASSERT( m_Stamp );

	return NODE_RESULT_OK;
}

// GetFullArgs
//------------------------------------------------------------------------------
void LinkerNode::GetFullArgs( AString & fullArgs ) const
{
	// split into tokens
	Array< AString > tokens( 1024, true );
	m_LinkerArgs.Tokenize( tokens );

	auto const end = tokens.End();
	for ( auto it = tokens.Begin(); it!=end; ++it )
	{
		const AString & token = *it;

		// %1 -> InputFiles
		const char * found = token.Find( "%1" );
		if ( found )
		{
			AStackString<> pre( token.Get(), found );
			AStackString<> post( found + 2, token.GetEnd() );
			GetInputFiles( fullArgs, pre, post );
			fullArgs += ' ';
			continue;
		}

		// %2 -> OutputFile
		found = token.Find( "%2" );
		if ( found )
		{
			fullArgs += AStackString<>( token.Get(), found );
			fullArgs += m_Name;
			fullArgs += AStackString<>( found + 2, token.GetEnd() );
			fullArgs += ' ';
			continue;
		}

		// %3 -> AssemblyResources
		if ( GetFlag( LINK_FLAG_MSVC ) )
		{
			found = token.Find( "%3" );
			if ( found )
			{
				AStackString<> pre( token.Get(), found );
				AStackString<> post( found + 2, token.GetEnd() );
				GetAssemblyResourceFiles( fullArgs, pre, post );
				fullArgs += ' ';
				continue;
			}
		}

		// untouched token
		fullArgs += token;
		fullArgs += ' ';
	}
}

// GetInputFiles
//------------------------------------------------------------------------------
void LinkerNode::GetInputFiles( AString & fullArgs, const AString & pre, const AString & post ) const
{
	// (exlude assembly resources from inputs)
	const auto end = m_StaticDependencies.End() - ( m_AssemblyResources.GetSize() + m_OtherLibraries.GetSize() );
	for ( Array< Node * >::Iter i = m_StaticDependencies.Begin();
		  i != end;
		  i++ )
	{
		Node * n( *i );
		GetInputFiles( n, fullArgs, pre, post );
	}
}

// GetInputFiles
//------------------------------------------------------------------------------
void LinkerNode::GetInputFiles( Node * n, AString & fullArgs, const AString & pre, const AString & post ) const
{
	if ( n->GetType() == Node::LIBRARY_NODE )
	{
		bool linkObjectsInsteadOfLibs = GetFlag( LINK_OBJECTS );

		if ( linkObjectsInsteadOfLibs )
		{
			LibraryNode * ln = n->CastTo< LibraryNode >();
			ln->GetInputFiles( fullArgs, pre, post );
		}
		else
		{
			// not building a DLL, so link the lib directly
			fullArgs += pre;
			fullArgs += n->GetName();
			fullArgs += post;
		}
	}
	else if ( n->GetType() == Node::OBJECT_LIST_NODE )
	{
		ObjectListNode * ol = n->CastTo< ObjectListNode >();
		ol->GetInputFiles( fullArgs, pre, post );
	}
	else if ( n->GetType() == Node::DLL_NODE )
	{
		// for a DLL, link to the import library
		DLLNode * dllNode = n->CastTo< DLLNode >();
		AStackString<> importLibName;
		dllNode->GetImportLibName( importLibName );
		fullArgs += pre;
		fullArgs += importLibName;
		fullArgs += post;
	}
	else if ( n->GetType() == Node::COPY_NODE )
	{
		CopyNode * copyNode = n->CastTo< CopyNode >();
		Node * srcNode = copyNode->GetSourceNode();
		GetInputFiles( srcNode, fullArgs, pre, post );
	}
	else
	{
		// link anything else directly
		fullArgs += pre;
		fullArgs += n->GetName();
		fullArgs += post;
	}

	fullArgs += ' ';
}

// GetAssemblyResourceFiles
//------------------------------------------------------------------------------
void LinkerNode::GetAssemblyResourceFiles( AString & fullArgs, const AString & pre, const AString & post ) const
{
	const auto end = m_AssemblyResources.End();
	for ( Array< Node * >::Iter i = m_AssemblyResources.Begin();
		  i != end;
		  i++ )
	{
		Node * n( *i );

		if ( n->GetType() == Node::OBJECT_LIST_NODE )
		{
			ObjectListNode * oln = n->CastTo< ObjectListNode >();
			oln->GetInputFiles( fullArgs, pre, post );
			continue;
		}

		if ( n->GetType() == Node::LIBRARY_NODE )
		{
			LibraryNode * ln = n->CastTo< LibraryNode >();
			ln->GetInputFiles( fullArgs, pre, post );
			continue;
		}

		fullArgs += pre;
		fullArgs += n->GetName();
		fullArgs += post;
		fullArgs += ' ';
	}
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool LinkerNode::Save( IOStream & stream, bool remote ) const
{
	// we want to save the static deps excluding the 
	Array< Node * > staticDeps( m_StaticDependencies );
	staticDeps.SetSize( m_StaticDependencies.GetSize() - ( m_AssemblyResources.GetSize() + m_OtherLibraries.GetSize() ) );

	NODE_SAVE( m_Name );
	NODE_SAVE( m_Linker );
	NODE_SAVE( m_LinkerArgs );
	NODE_SAVE_DEPS( staticDeps );
	NODE_SAVE( m_Flags );
	NODE_SAVE_DEPS( m_AssemblyResources );
	NODE_SAVE_DEPS( m_OtherLibraries );
	return true;
}

//------------------------------------------------------------------------------
