// DirectoryListNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "DirectoryListNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"

#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Strings/AStackString.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
DirectoryListNode::DirectoryListNode( const AString & name,
									  const AString & path,
									  const AString & wildcard,
								      bool recursive,
									  const AString & excludePath )
: Node( name, Node::DIRECTORY_LIST_NODE, ControlFlag::FLAG_NONE )
	, m_Path( path )
	, m_WildCard( wildcard )
	, m_ExcludePath( excludePath )
	, m_Recursive( recursive )
	, m_Files( 4096, true )
{
	// ensure name is correctly formatted
	//   path|wildcard|recursive|[excludePath]
	ASSERT( name.BeginsWith( path ) );
	ASSERT( name[ path.GetLength() ] == '|' );
	ASSERT( name.Find( wildcard.Get() ) == name.Get() + path.GetLength() + 1 );
	ASSERT( ( recursive && name.Find( "|true|" ) ) || 
			( !recursive && name.Find( "|false|" ) ) );
	ASSERT( name.EndsWith( excludePath.Get() ) );

	// paths must have trailing slash
	ASSERT( path.EndsWith( '\\' ) );

	// make sure exclusion path has trailing slash if provided
	ASSERT( excludePath.IsEmpty() ||
			excludePath.EndsWith( '\\' ) );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
DirectoryListNode::~DirectoryListNode()
{
}

// FormatName
//------------------------------------------------------------------------------
/*static*/ void DirectoryListNode::FormatName( const AString & path,
											   const AString & pattern,
											   bool recursive,
											   const AString & excludePath,
											   AString & result )
{
	ASSERT( path.EndsWith( '\\' ) );
	ASSERT( pattern.IsEmpty() == false );
	ASSERT( excludePath.IsEmpty() || excludePath.EndsWith( '\\' ) );

	result.Format( "%s|%s|%s|%s", path.Get(),
								  pattern.Get(),
								  recursive ? "true" : "false",
								  excludePath.Get() );
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult DirectoryListNode::DoBuild( Job * UNUSED( job ) )
{
	// NOTE: The DirectoryListNode makes no assumptions about whether no files
	// is an error or not.  That's up to the dependent nodes to decide.

	if ( m_ExcludePath.IsEmpty() )
	{
		FileIO::GetFilesEx( m_Path, m_WildCard, m_Recursive, &m_Files );
	}
	else
	{
		Array< FileIO::FileInfo > files( 4096, true );
		FileIO::GetFilesEx( m_Path, m_WildCard, m_Recursive, &files );

		m_Files.SetCapacity( files.GetSize() );

		// filter exclusions
		for ( Array< FileIO::FileInfo >::Iter it = files.Begin();
			  it != files.End();
			  it++ )
		{
			if ( it->m_Name.BeginsWithI( m_ExcludePath ) )
			{
				continue;
			}
			m_Files.Append( *it );
		}
	}

	if ( FLog::ShowInfo() )
	{
		const size_t numFiles = m_Files.GetSize();
		FLOG_INFO( "Dir: '%s' (found %u files)\n", 
							m_Name.Get(), 
							(uint32_t)numFiles);
		for ( size_t i=0; i<numFiles; ++i )
		{
			FLOG_INFO( " - %s\n", m_Files[ i ].m_Name.Get() );
		}
	}

	return NODE_RESULT_OK;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * DirectoryListNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( AStackString<>,	path );
	NODE_LOAD( AStackString<>,	wildCard );
	NODE_LOAD( AStackString<>,	excludePath );
	NODE_LOAD( bool,			recursive );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * n = ng.CreateDirectoryListNode( name, path, wildCard, recursive, excludePath );
	ASSERT( n );
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool DirectoryListNode::Save( IOStream & stream, bool UNUSED( remote ) ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_Path );
	NODE_SAVE( m_WildCard );
	NODE_SAVE( m_ExcludePath );
	NODE_SAVE( m_Recursive );
	return true;
}

//------------------------------------------------------------------------------
