// FileNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "CopyNode.h"

#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/FLog.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"

#include "Core/Env/Env.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/Strings/AStackString.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
CopyNode::CopyNode( const AString & dstFileName,
					FileNode * sourceFile )
: FileNode( dstFileName, ControlFlag::FLAG_NONE )
, m_SourceFile( sourceFile )
{
	ASSERT( sourceFile );
	ASSERT( sourceFile->IsAFile() );
	m_StaticDependencies.Append( m_SourceFile );
	m_Type = Node::COPY_NODE;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
CopyNode::~CopyNode()
{
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult CopyNode::DoBuild( Job * UNUSED( job ) )
{
	EmitCopyMessage();

	// copy the file
	if ( FileIO::FileCopy( m_SourceFile->GetName().Get(), m_Name.Get() ) == false )
	{
		FLOG_ERROR( "Copy failed (error %i) '%s'", Env::GetLastErr(), GetName().Get() );
		return NODE_RESULT_FAILED; // copy failed
	}

	if ( FileIO::SetReadOnly( m_Name.Get(), false ) == false )
	{
		FLOG_ERROR( "Copy read-only flag set failed (error %i) '%s'", Env::GetLastErr(), GetName().Get() );
		return NODE_RESULT_FAILED; // failed to remove read-only
	}

	// update the file's "last modified" time to be equal to the src file
	m_Stamp = m_SourceFile->GetStamp();
	ASSERT( m_Stamp );
	if ( FileIO::SetFileLastWriteTime( m_Name, m_Stamp ) == false )
	{
		FLOG_ERROR( "Copy set last write time failed (error %i) '%s'", Env::GetLastErr(), GetName().Get() );
		m_Stamp = 0;
		return NODE_RESULT_FAILED; // failed to set the time
	}

	return NODE_RESULT_OK;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * CopyNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	fileName);
	NODE_LOAD( AStackString<>,	sourceFile );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * srcNode = ng.FindNode( sourceFile );
	ASSERT( srcNode ); // load/save logic should ensure the src was saved first
	ASSERT( srcNode->IsAFile() );
	CopyNode * n = ng.CreateCopyNode( fileName, (FileNode *)srcNode );
	ASSERT( n );
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool CopyNode::Save( IOStream & stream, bool UNUSED( remote ) ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_SourceFile->GetName() );
	return true;
}

// EmitCompilationMessage
//------------------------------------------------------------------------------
void CopyNode::EmitCopyMessage() const
{
	// we combine everything into one string to ensure it is contiguous in
	// the output
	AStackString<> output;
	output += "Copy: ";
	output += m_StaticDependencies[ 0 ]->GetName();
	output += " -> ";
	output += GetName();
	output += '\n';
	FLOG_BUILD( "%s", output.Get() );
}

//------------------------------------------------------------------------------
