// FBuild.cpp - The main FBuild interface class
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_FBUILD_H
#define FBUILD_FBUILD_H

// Includes
//------------------------------------------------------------------------------
#include "Helpers/FBuildStats.h"

#include "Core\Containers\Array.h"
#include "Core\Containers\Singleton.h"
#include "Core\Strings\AString.h"
#include "Core\Time\Timer.h"

// Forward Declarations
//------------------------------------------------------------------------------
class Client;
class FileStream;
class JobQueue;
class Node;
class NodeGraph;

// FBuildOptions
//------------------------------------------------------------------------------
struct FBuildOptions
{
public:
	FBuildOptions();

	void SetWorkingDir( const AString & path );
	inline const AString & GetWorkingDir() const { return m_WorkingDir; }

	bool m_ForceCleanBuild;
	bool m_UseCacheRead;
	bool m_UseCacheWrite;
	bool m_ShowInfo;
	bool m_ShowErrors;
	bool m_ShowProgress;
	bool m_AllowDistributed;
	bool m_ShowSummary;
	bool m_ShowBuildTime;
	bool m_GenerateReport;
	uint32_t m_NumWorkerThreads;
	AString m_ConfigFile;

private:
	AString m_WorkingDir;
};


// FBuild
//------------------------------------------------------------------------------
class FBuild : public Singleton< FBuild >
{
public:
	explicit FBuild( const FBuildOptions & options = FBuildOptions() );
	~FBuild();

	// initialize the dependency graph, using the BFF config file
	// OR a previously saved NodeGraph DB (if available/matching the BFF)
	bool Initialize( const char * nodeGraphDBFile = nullptr );

	// build a target
	bool Build( const AString & target );
	bool Build( Node * nodeToBuild );

	// after a build we can store progress/parsed rules for next time
	bool SaveDependencyGraph( const char * nodeGraphDBFile = nullptr ) const;

	const FBuildOptions & GetOptions() const { return m_Options; }
	NodeGraph & GetDependencyGraph() const { return *m_DependencyGraph; }
	
	const AString & GetWorkingDir() const { return m_Options.GetWorkingDir(); }

	static const char * GetDependencyGraphFileName();
	static const char * GetDefaultBFFFileName();

	const AString & GetCachePath() const { return m_CachePath; }
	void SetCachePath( const AString & path );

	void GetCacheFileName( uint32_t keyA, uint32_t keyB,
						   AString & path ) const;

	void SetWorkerList( const Array< AString > & workers )		{ m_WorkerList = workers; }
	const Array< AString > & GetWorkerList() const { return m_WorkerList; }

	void SetEnvironmentString( const char * envString, uint32_t size, const AString & libEnvVar );
	inline const char * GetEnvironmentString() const			{ return m_EnvironmentString; }
	inline uint32_t		GetEnvironmentStringSize() const		{ return m_EnvironmentStringSize; }

	void GetLibEnvVar( AString & libEnvVar ) const;

	// stats - read access
	const FBuildStats & GetStats() const	{ return m_BuildStats; }
	// stats - write access
	FBuildStats & GetStatsMutable()			{ return m_BuildStats; }

	// attempt to cleanly stop the build
	static inline void SetStopBuild() { s_StopBuild = true; }
	static inline bool GetStopBuild() { return s_StopBuild; }

private:
	void UpdateBuildStatus( const Node * node, bool forceUpdate );

	static bool s_StopBuild;

	NodeGraph * m_DependencyGraph;
	JobQueue * m_JobQueue;
	Client * m_Client; // manage connections to worker servers

	AString m_CachePath;

	Timer m_Timer;
	float m_LastProgressOutputTime;
	float m_LastProgressCalcTime;
	float m_SmoothedProgressCurrent;
	float m_SmoothedProgressTarget;

	FBuildStats m_BuildStats;

	FBuildOptions m_Options;

	Array< AString > m_WorkerList;

	AString m_OldWorkingDir;

	// a double-null terminated string
	char *		m_EnvironmentString;
	uint32_t	m_EnvironmentStringSize; // size excluding last null
	AString		m_LibEnvVar; // LIB= value
};

//------------------------------------------------------------------------------
#endif // FBUILD_FBUILD_H
