// FBuild - the main application
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FBuild.h"

#include "FLog.h"
#include "BFF/BFFParser.h"
#include "BFF/Functions/Function.h"
#include "Graph/Node.h"
#include "Graph/NodeGraph.h"
#include "Helpers/Report.h"
#include "Protocol/Client.h"
#include "Protocol/Protocol.h"
#include "WorkerPool/JobQueue.h"
#include "WorkerPool/WorkerThread.h"

#include "Core/Env/Assert.h"
#include "Core/Env/Env.h"
#include "Core/Env/Types.h"
#include "Core/FileIO/FileIO.h"
#include "Core/FileIO/FileStream.h"
#include "Core/FileIO/MemoryStream.h"
#include "Core/Profile/Profile.h"
#include "Core/Strings/AStackString.h"
#include "Core/Tracing/Tracing.h"

#include <stdio.h>
#include <time.h>

//#define DEBUG_CRT_MEMORY_USAGE // Uncomment this for (very slow) detailed mem checks
#ifdef DEBUG_CRT_MEMORY_USAGE
	#include <crtdbg.h>
#endif

// Static
//------------------------------------------------------------------------------
/*static*/ bool FBuild::s_StopBuild( false );

// CONSTRUCTOR - FBuildOptions
//------------------------------------------------------------------------------
FBuildOptions::FBuildOptions()
: m_ForceCleanBuild( false )
, m_UseCacheRead( false )
, m_UseCacheWrite( false )
, m_ShowInfo( false )
, m_ShowProgress( false )
, m_ShowErrors( true )
, m_AllowDistributed( false )
, m_ShowSummary( false )
, m_ShowBuildTime( true )
, m_GenerateReport( false )
{
#ifdef DEBUG
	//m_ShowInfo = true; // uncomment this to enable spam when debugging
#endif

	// Default to NUMBER_OF_PROCESSORS
	m_NumWorkerThreads = Env::GetNumProcessors();
}

// FBuildOptions::SetWorkingDir
//------------------------------------------------------------------------------
void FBuildOptions::SetWorkingDir( const AString & path )
{
	m_WorkingDir = path;
	if ( m_WorkingDir.IsEmpty() )
	{
		return;
	}

	// clean path

	// slashed must face the right way
	m_WorkingDir.Replace( '/', '\\' );

	// eliminate redundant slash pairs
	while ( m_WorkingDir.Replace( "\\\\", "\\" ) );

	// no trailing slash
	if ( m_WorkingDir.EndsWith( '\\' ) )
	{
		m_WorkingDir.SetLength( m_WorkingDir.GetLength() - 1 );
	}

	#if defined( __WINDOWS__ )
		// so C:\ and c:\ are treated the same on Windows, for better cache hits
		// make the drive letter always uppercase
		if ( ( m_WorkingDir.GetLength() >= 2 ) &&
			 ( m_WorkingDir[ 1 ] == ':' ) &&
			 ( m_WorkingDir[ 0 ] >= 'a' ) &&
			 ( m_WorkingDir[ 0 ] <= 'z' ) )
		{
			m_WorkingDir[ 0 ] = ( 'A' + ( m_WorkingDir[ 0 ] - 'a' ) );
		}
	#endif
}

// CONSTRUCTOR - FBuild
//------------------------------------------------------------------------------
FBuild::FBuild( const FBuildOptions & options )
	: m_LastProgressOutputTime( 0.0f )
	, m_LastProgressCalcTime( 0.0f )
	, m_SmoothedProgressCurrent( 0.0f )
	, m_SmoothedProgressTarget( 0.0f )
	, m_Client( nullptr )
	, m_WorkerList( 0, true )
	, m_EnvironmentString( nullptr )
	, m_EnvironmentStringSize( 0 )
{
	#ifdef DEBUG_CRT_MEMORY_USAGE
		_CrtSetDbgFlag( _CRTDBG_ALLOC_MEM_DF | 
						_CRTDBG_CHECK_ALWAYS_DF | //_CRTDBG_CHECK_EVERY_16_DF |
						_CRTDBG_CHECK_CRT_DF | 
						_CRTDBG_DELAY_FREE_MEM_DF | 
						_CRTDBG_LEAK_CHECK_DF );
	#endif
	m_DependencyGraph = new NodeGraph();

	// store all user provided options
	m_Options = options;

	// track the old working dir to restore if modified (mainly for unit tests)
	VERIFY( FileIO::GetCurrentDir( m_OldWorkingDir ) );

	// use specified working dir, or the current dir if not provided
	if ( options.GetWorkingDir().IsEmpty() )
	{
		m_Options.SetWorkingDir( m_OldWorkingDir );
	}

	// poke options where required
	FLog::SetShowInfo( m_Options.m_ShowInfo );
	FLog::SetShowErrors( m_Options.m_ShowErrors );
	FLog::SetShowProgress( m_Options.m_ShowProgress );

	Function::Create();
}

// DESTRUCTOR
//------------------------------------------------------------------------------
FBuild::~FBuild()
{
	Function::Destroy();

	delete m_DependencyGraph;
	delete m_Client;
	::Free( m_EnvironmentString );

	// restore the old working dir to restore
	ASSERT( !m_OldWorkingDir.IsEmpty() );
	if ( !FileIO::SetCurrentDir( m_OldWorkingDir ) )
	{
		FLOG_ERROR( "Failed to restore working dir: '%s' (error: %u)", m_OldWorkingDir.Get(), Env::GetLastErr() );
	}
}

// Initialize
//------------------------------------------------------------------------------
bool FBuild::Initialize( const char * nodeGraphDBFile )
{
	// handle working dir
	if ( !FileIO::SetCurrentDir( m_Options.GetWorkingDir() ) )
	{
		FLOG_ERROR( "Failed to set working dir: '%s' (error: %u)", m_Options.GetWorkingDir().Get(), Env::GetLastErr() );
		return false;
	}

	const char * bffFile = m_Options.m_ConfigFile.IsEmpty() ? GetDefaultBFFFileName()
														    : m_Options.m_ConfigFile.Get();
	if ( m_DependencyGraph->Initialize( bffFile, nodeGraphDBFile ) == false )
	{
		return false;
	}

	// if the cache is enabled, make sure the path is set and accessible
	if ( m_Options.m_UseCacheRead || m_Options.m_UseCacheWrite )
	{
		if ( m_CachePath.IsEmpty() )
		{
			FLOG_WARN( "CachePath not set - Caching disabled" );
			m_Options.m_UseCacheRead = false;
			m_Options.m_UseCacheWrite = false;
		}
		else if ( FileIO::EnsurePathExists( m_CachePath ) == false )
		{
			FLOG_WARN( "Cache inaccessible - Caching disabled (%s)", m_CachePath.Get() );
			m_Options.m_UseCacheRead = false;
			m_Options.m_UseCacheWrite = false;
		}
	}

	//
	// create the connection management system if we might need it
	if ( m_Options.m_AllowDistributed )
	{
		if ( m_WorkerList.IsEmpty() )
		{
			FLOG_WARN( "No workers configured - Distributed compilation disabled" );
			m_Options.m_AllowDistributed = false;
		}
		else
		{
			m_Client = new Client( m_WorkerList );
		}
	}

	return true;
}

// Build
//------------------------------------------------------------------------------
bool FBuild::Build( const AString & target )
{
	PROFILE_FUNCTION

	ASSERT( !target.IsEmpty() );

	// get the node being requested (search for exact match, to find aliases etc first)
	Node * node = m_DependencyGraph->FindNodeInternal( target );
	if ( node == nullptr )
	{
		// failed to find the node, try looking for a fully pathed equivalent
		node = m_DependencyGraph->FindNode( target );
	}

	if ( node == nullptr )
	{
		FLOG_ERROR( "Unknown build target '%s'", target.Get() );
		return false;
	}

	// build it!
	return Build( node );
}

// SaveDependencyGraph
//------------------------------------------------------------------------------
bool FBuild::SaveDependencyGraph( const char * nodeGraphDBFile ) const
{
	nodeGraphDBFile = nodeGraphDBFile ? nodeGraphDBFile : GetDependencyGraphFileName();

	FLOG_INFO( "Saving DepGraph '%s'", nodeGraphDBFile );

	Timer t;

	// try to open the file
	FileStream fileStream;
	if ( fileStream.Open( nodeGraphDBFile, FileStream::WRITE_ONLY ) == false )
	{
		// failing to open the dep graph for saving is a serious problem
		FLOG_ERROR( "Failed to open DepGraph for saving '%s'", nodeGraphDBFile );
		return false;
	}

	// serialize into memory first
	MemoryStream memoryStream( 32 * 1024 * 1024, 8 * 1024 * 1024 );
	if ( m_DependencyGraph->Save( memoryStream ) )
	{
		// write in-memory serialized data to disk
		if ( fileStream.Write( memoryStream.GetData(), memoryStream.GetSize() ) == memoryStream.GetSize() )
		{
			FLOG_INFO( "Saving DepGraph Complete in %2.3fs", t.GetElapsed() );
			return true;
		}
	}

	FLOG_ERROR( "Saving DepGraph FAILED!" );
	return false;
}

// Build
//------------------------------------------------------------------------------
bool FBuild::Build( Node * nodeToBuild )
{
	PROFILE_FUNCTION

	ASSERT( nodeToBuild );

	// create worker threads
	m_JobQueue = new JobQueue( m_Options.m_NumWorkerThreads );

	m_Timer.Start();
	m_LastProgressOutputTime = 0.0f;
	m_LastProgressCalcTime = 0.0f;
	m_SmoothedProgressCurrent = 0.0f;
	m_SmoothedProgressTarget = 0.0f;
	FLog::StartBuild();

	bool stopping( false );

	// keep doing build passes until completed/failed
	for ( ;; )
	{
		bool complete = ( nodeToBuild->GetState() == Node::UP_TO_DATE ) ||
						( nodeToBuild->GetState() == Node::FAILED );

		if ( s_StopBuild || complete )
		{
			if ( stopping == false ) 
			{
				// free the network distribution system (if there is one)
				delete m_Client;
				m_Client = nullptr;

				// wait for workers to exit.  Can still be building even though we've failed:
				//  - only 1 failed node propagating up to root while others are not yet complete
				//  - aborted build, so workers can be incomplete
				m_JobQueue->SignalStopWorkers();
				stopping = true;
			}
		}

		if ( !stopping )
		{
			// do a sweep of the graph to create more jobs
			m_DependencyGraph->DoBuildPass( nodeToBuild );
		}

		if ( m_Options.m_NumWorkerThreads == 0 )
		{
			// no local threads - do build directly
			WorkerThread::Update();
		}

		// process completed jobs
		m_JobQueue->FinalizeCompletedJobs();

		// completely stopped?
		if ( stopping && m_JobQueue->HaveWorkersStopped() )
		{
			break;
		}

		Sleep( 16 );

		// update progress
		UpdateBuildStatus( nodeToBuild, false );

		PROFILE_SYNCHRONIZE
	}

	delete m_JobQueue;
	m_JobQueue = nullptr;

	// force one more update to show completion time
	UpdateBuildStatus( nodeToBuild, true );

	FLog::StopBuild();

	// TODO:C Move this into BuildStats
	float timeTaken = m_Timer.GetElapsed();
	m_BuildStats.m_TotalBuildTime = timeTaken;

	m_BuildStats.OnBuildStop( nodeToBuild );

	bool buildOK = ( nodeToBuild->GetState() == Node::UP_TO_DATE );

	// final line of output - status of build
	if ( m_Options.m_ShowBuildTime )
	{
		uint32_t minutes = uint32_t( timeTaken / 60.0f );
		timeTaken -= ( minutes * 60.0f );
		float seconds = timeTaken;
		const char * msgString = buildOK ? "Build complete for '%s' in %um %05.3fs\n" 
										 : "Build FAILED for '%s' in %um %05.3fs\n";
		FLOG_BUILD( msgString, nodeToBuild->GetName().Get(), minutes, seconds );
	}

	return buildOK;
}

// SetEnvironmentString
//------------------------------------------------------------------------------
void FBuild::SetEnvironmentString( const char * envString, uint32_t size, const AString & libEnvVar )
{
	::Free( m_EnvironmentString );
	m_EnvironmentString = (char *)::Alloc( size + 1 );
	m_EnvironmentStringSize = size;
	AString::Copy( envString, m_EnvironmentString, size );
	m_LibEnvVar = libEnvVar;
}

// GetLibEnvVar
//------------------------------------------------------------------------------
void FBuild::GetLibEnvVar( AString & value ) const
{
	// has environment been overridden in BFF?
	if ( m_EnvironmentString )
	{
		// use overridden LIB path (which maybe empty)
		value = m_LibEnvVar;
	}
	else
	{
		// use real environment LIB path
		Env::GetEnvVariable( "LIB", value );
	}
}

// UpdateBuildStatus
//------------------------------------------------------------------------------
void FBuild::UpdateBuildStatus( const Node * node, bool forceUpdate )
{
	PROFILE_FUNCTION

	if ( FBuild::Get().GetOptions().m_ShowProgress == false ) 
	{
		return;
	}

	const float OUTPUT_FREQUENCY( 1.0f );
	const float CALC_FREQUENCY( 5.0f );

	float timeNow = m_Timer.GetElapsed();

	bool doUpdate = ( ( timeNow - m_LastProgressOutputTime ) >= OUTPUT_FREQUENCY );	
	doUpdate |= forceUpdate;

	if ( doUpdate == false )
	{
		return;
	}

	// recalculate progress estimate?
	if ( ( timeNow - m_LastProgressCalcTime ) >= CALC_FREQUENCY )
	{
		FBuildStats & bs = m_BuildStats;
		bs.m_NodeTimeProgressms = 0;
		bs.m_NodeTimeTotalms = 0;
		bs.m_NumNodesInTarget = 0;
		m_DependencyGraph->UpdateBuildStatusRecurse( node, bs.m_NodeTimeProgressms, bs.m_NodeTimeTotalms, bs.m_NumNodesInTarget );
		m_LastProgressCalcTime = timeNow;

		// calculate percentage
		float doneRatio = (float)( (double)bs.m_NodeTimeProgressms / (double)bs.m_NodeTimeTotalms );

		// don't allow it to reach 100% (handles rounding inaccuracies)
		float donePerc = Math::Min< float >( doneRatio * 100.0f, 99.9f );

		// don't allow progress to go backwards
		m_SmoothedProgressTarget = Math::Max< float >( donePerc, m_SmoothedProgressTarget );
	}

	m_SmoothedProgressCurrent = ( 0.5f * m_SmoothedProgressCurrent ) + ( m_SmoothedProgressTarget * 0.5f );

	// get nodes counts;
	uint32_t numJobs = 0;
	uint32_t numJobsActive = 0;
	uint32_t numJobsDist = 0;
	uint32_t numJobsDistActive = 0;
	if ( JobQueue::IsValid() )
	{
		JobQueue::Get().GetJobStats( numJobs, numJobsActive, numJobsDist, numJobsDistActive );
	}

	FLog::OutputProgress( timeNow, m_SmoothedProgressCurrent, numJobs, numJobsActive, numJobsDist, numJobsDistActive );
	m_LastProgressOutputTime = timeNow;
}

// GetDependencyGraphFileName
//------------------------------------------------------------------------------
/*static*/ const char * FBuild::GetDependencyGraphFileName()
{
	return "fbuild.fdb";
}

// GetDefaultBFFFileName
//------------------------------------------------------------------------------
/*static*/ const char * FBuild::GetDefaultBFFFileName()
{
	return "fbuild.bff";
}

// SetCachePath
//------------------------------------------------------------------------------
void FBuild::SetCachePath( const AString & path )
{ 
	m_CachePath = path;
	if ( m_CachePath.EndsWith( '\\' ) == false )
	{
		m_CachePath += '\\';
	}
}

// GetCacheFileName
//------------------------------------------------------------------------------
void FBuild::GetCacheFileName( uint32_t keyA, uint32_t keyB,
						   AString & path ) const
{
	// cache version - bump if cache format is changed
	static const int cacheVersion( 2 );

	const AString & cacheRoot = GetCachePath();
	ASSERT( cacheRoot.IsEmpty() == false );

	// cache is split into sub-folders so there aren't so many files in one dir
	// in order to avoid SMB network filesystem slowdowns
	uint8_t byte1 = uint8_t( keyA & 0x000000FF );
	uint8_t byte2 = uint8_t( ( keyA & 0x0000FF00 ) >> 8 );

	// format example: N:\\fbuild.cache\\32\\DE\\2377DE32_FED872A1.2 
	path.Format( "%s\\%02X\\%02X\\%08X_%08X.%u", cacheRoot.Get(), byte1, byte2, keyA, keyB, cacheVersion );
}

//------------------------------------------------------------------------------
