// FunctionVCXProject
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
// FBuild
#include "FunctionVCXProject.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFVariable.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/VCXProjectNode.h"

// Core
#include "Core/Strings/AStackString.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionVCXProject::FunctionVCXProject()
: Function( "VCXProject" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionVCXProject::AcceptsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionVCXProject::Commit( const BFFIterator & funcStartIter ) const
{
	// required
	const BFFVariable * projectOutputV;
	if ( !GetString( funcStartIter, projectOutputV,		".ProjectOutput", true ) )
	{
		return false;
	}

	// optional inputs
	Array< AString > inputPaths;
	Array< AString > inputPathsExclude;
	if ( !GetStrings( funcStartIter, inputPaths,		".ProjectInputPaths", false ) ||
		 !GetStrings( funcStartIter, inputPathsExclude,	".ProjectInputPathsExclude", false ) )
	{
		return false;
	}

	// optional (with defaults)
	Array< AString > configs( 8, true );
	Array< AString > platforms( 8, true );	
	if ( !GetStrings( funcStartIter, configs,	".ProjectConfigs", false ) ||
 		 !GetStrings( funcStartIter, platforms, ".ProjectPlatforms", false ) )
	{
		return false;
	}
	if ( configs.IsEmpty() )
	{
		configs.Append( AString( "Debug" ) );
		configs.Append( AString( "Release" ) );
	}
	if ( platforms.IsEmpty() )
	{
		platforms.Append( AString( "Win32" ) );
		platforms.Append( AString( "x64" ) );
	}
	const BFFVariable * buildCmd;
	const BFFVariable * rebuildCmd;
	const BFFVariable * cleanCmd;
	if ( !GetString( funcStartIter, buildCmd,	".ProjectBuildCommand", false ) ||
		 !GetString( funcStartIter, rebuildCmd, ".ProjectRebuildCommand", false ) ||
		 !GetString( funcStartIter, cleanCmd,	".ProjectCleanCommand", false ) )
	{
		return false;
	}
	const BFFVariable * basePathV;
	if ( !GetString( funcStartIter, basePathV,	".ProjectBasePath", false ) )
	{
		return false;
	}
	AStackString<> basePath;
	if ( basePathV )
	{
		NodeGraph::CleanPath( basePathV->GetString(), basePath );
		if ( basePath.EndsWith( '\\' ) == false )
		{
			basePath += '\\';
		}
	}
	Array< AString > allowedFileExtensions( 8, true );
	if ( !GetStrings( funcStartIter, allowedFileExtensions, ".ProjectAllowedFileExtensions", false ) )
	{
		return true;
	}
	if ( allowedFileExtensions.IsEmpty() )
	{
		const char * extensions[] = { ".cpp", ".hpp", ".cxx",".hxx",".c",".h",".cc",".hh",
									  ".cp",".hp",".cs",".inl",".bff",".rc",".resx", 
									  nullptr };
		AStackString<> tmp;
		const char ** item = extensions;
		while ( *item )
		{
			tmp.Assign( *item );
			allowedFileExtensions.Append( tmp );
			++item;
		}
	}
	Array< AString > files( 8, true );
	Array< AString > filesToExclude( 8, true );	
	if ( !GetStrings( funcStartIter, files,				".ProjectFiles", false ) ||
 		 !GetStrings( funcStartIter, filesToExclude,	".ProjectFilesToExclude", false ) )
	{
		return false;
	}

	// debugger options
	const BFFVariable * localDebuggerCommandArguments;
	const BFFVariable * localDebuggerWorkingDirectory;
	const BFFVariable * localDebuggerCommand;
	const BFFVariable * localDebuggerEnvironment;
	if ( !GetString( funcStartIter, localDebuggerCommandArguments,	".LocalDebuggerCommandArguments", false ) ||
		 !GetString( funcStartIter, localDebuggerWorkingDirectory,	".LocalDebuggerWorkingDirectory", false ) ||
		 !GetString( funcStartIter, localDebuggerCommand,			".LocalDebuggerCommand", false ) ||
		 !GetString( funcStartIter, localDebuggerEnvironment,		".LocalDebuggerEnvironment", false ) )
	{
		return false;
	}

	// path cleaning
	CleanFolderPaths( inputPaths );			// input paths
	CleanFolderPaths( inputPathsExclude );	// exclude paths
	CleanFilePaths( files );				// explicit files

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// create all of the DirectoryListNodes we need
	AStackString<> inputPattern( "*" );
	Array< DirectoryListNode * > dirNodes( inputPaths.GetSize() );
	for ( auto it = inputPaths.Begin(); it != inputPaths.End(); ++it )
	{
		const AString & path = *it;

		// get node for the dir we depend on
		AStackString<> dirNodeName;
		DirectoryListNode::FormatName( path, 
									   inputPattern,
									   true,
									   AString::GetEmpty(),
									   dirNodeName );
		Node * dirNode = ng.FindNode( dirNodeName );
		if ( dirNode == nullptr )
		{
			dirNode = ng.CreateDirectoryListNode( dirNodeName,
												  path,
												  inputPattern,
												  true,
												  AString::GetEmpty() );
		}
		else if ( dirNode->GetType() != Node::DIRECTORY_LIST_NODE )
		{
			Error::Error_1102_UnexpectedType( funcStartIter, this, "ProjectInputPaths", dirNode->GetName(), dirNode->GetType(), Node::DIRECTORY_LIST_NODE );
			return false;
		}
		dirNodes.Append( dirNode->CastTo< DirectoryListNode >() );
	}

	// Check for existing node
	if ( ng.FindNode( projectOutputV->GetString() ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, projectOutputV->GetString() );
		return false;
	}

	VCXProjectNode * pn = ng.CreateVCXProjectNode( projectOutputV->GetString(),
												   basePath,
												   dirNodes,
												   inputPathsExclude,
												   allowedFileExtensions,
												   files,
												   filesToExclude,
												   configs,
												   platforms,
												   buildCmd ? buildCmd->GetString() : AString::GetEmpty(),
												   rebuildCmd ? rebuildCmd->GetString() : AString::GetEmpty(),
												   cleanCmd ? cleanCmd->GetString() : AString::GetEmpty(),
												   localDebuggerCommandArguments ? localDebuggerCommandArguments->GetString() : AString::GetEmpty(),
												   localDebuggerWorkingDirectory ? localDebuggerWorkingDirectory->GetString() : AString::GetEmpty(),
												   localDebuggerCommand ? localDebuggerCommand->GetString() : AString::GetEmpty(),
												   localDebuggerEnvironment ? localDebuggerEnvironment->GetString() : AString::GetEmpty() );

	ASSERT( pn );

	return ProcessAlias( funcStartIter, pn );
}

//------------------------------------------------------------------------------
