// FunctionUnity
//------------------------------------------------------------------------------

#include "FunctionUnity.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFIterator.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFVariable.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h"

#include "Core/Strings/AStackString.h"

// UnityNode
//------------------------------------------------------------------------------
class UnityNode;

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionUnity::FunctionUnity()
: Function( "Unity" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::AcceptsHeader() const
{
	return true;
}

// NeedsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::NeedsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::Commit( const BFFIterator & funcStartIter ) const
{
	Array< AString > paths;
	Array< AString > pathsToExclude;
	const BFFVariable * inputPatternV;
	bool inputPathRecurse;
	const BFFVariable * outputPathV;
	const BFFVariable * outputPatternV;
	bool isolateWritableFiles;
	Array< AString > files;
	if ( !GetStrings( funcStartIter, pathsToExclude, ".UnityInputExcludePath", false ) ||
		 !GetString( funcStartIter, inputPatternV,	".UnityInputPattern" ) ||
		 !GetBool( funcStartIter, inputPathRecurse,".UnityInputPathRecurse", true ) ||
		 !GetStrings( funcStartIter, files, ".UnityInputFiles", false ) ||
		 !GetString( funcStartIter, outputPathV,	".UnityOutputPath", true ) ||
		 !GetString( funcStartIter, outputPatternV,	".UnityOutputPattern" ) ||
		 !GetStrings( funcStartIter, paths, ".UnityInputPath", false ) ||
		 !GetBool( funcStartIter, isolateWritableFiles,".UnityInputIsolateWritableFiles", false ) )
	{
		return false;
	}

	// num files is optional, but must be an integer
	const BFFVariable * numFilesV = BFFStackFrame::GetVar( ".UnityNumFiles" );
	if ( numFilesV )
	{
		if ( numFilesV->IsInt() == false )
		{
			Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "UnityNumFiles", numFilesV->GetType(), BFFVariable::VAR_INT );
			return false;
		}
		int numFiles = numFilesV->GetInt();
		if ( ( numFiles < 1 ) || ( numFiles > 999 ) )
		{
			Error::Error_1054_IntegerOutOfRange( funcStartIter, this, "UnityNumFiles", 1, 999 );
			return false;
		}
	}

	// clean exclude paths
	CleanFolderPaths( paths );
	CleanFolderPaths( pathsToExclude );	// exclude paths
	CleanFilePaths( files );			// explicit file list

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	const AString & inputPattern = inputPatternV ? inputPatternV->GetString() : (const AString &)AStackString<>( "*.cpp" );
	const AString & outputPattern = outputPatternV ? outputPatternV->GetString() : (const AString &)AStackString<>( "Unity*.cpp" );

	// create all of the DirectoryListNodes we need
	Array< DirectoryListNode * > dirNodes( paths.GetSize() );
	for ( auto it = paths.Begin(); it != paths.End(); ++it )
	{
		const AString & path = *it;

		// get node for the dir we depend on
		AStackString<> dirNodeName;
		DirectoryListNode::FormatName( path, 
									   inputPattern, 
									   inputPathRecurse, 
									   AString::GetEmpty(),
									   dirNodeName );
		Node * dirNode = ng.FindNode( dirNodeName );
		if ( dirNode == nullptr )
		{
			dirNode = ng.CreateDirectoryListNode( dirNodeName,
												  path,
												  inputPattern,
												  inputPathRecurse,
												  AString::GetEmpty() );
		}
		else if ( dirNode->GetType() != Node::DIRECTORY_LIST_NODE )
		{
			Error::Error_1102_UnexpectedType( funcStartIter, this, "UnityInputPath", dirNode->GetName(), dirNode->GetType(), Node::DIRECTORY_LIST_NODE );
			return false;
		}
		dirNodes.Append( dirNode->CastTo< DirectoryListNode >() );
	}

	// uses precompiled header?
	AStackString< 512 > precompiledHeader;
	const BFFVariable * pchV = nullptr;
	if ( !GetString( funcStartIter, pchV, ".UnityPCH" ) )
	{
		return false;
	}
	if ( pchV )
	{
		// Note: we deliberately don't clean the path to the PCH
		// as we don't depend on it, it doesn't need changing
		// and we don't want to mess with the users intent when possible
		precompiledHeader = pchV->GetString();
	}

	// determine files to exclude
	Array< AString > filesToExclude( 0, true );
	if ( !GetStrings( funcStartIter, filesToExclude, ".UnityInputExcludedFiles", false ) ) // not required
	{
		return false; // GetStrings will have emitted an error
	}
	// cleanup slashes (keep path relative)
	auto end = filesToExclude.End();
	for ( auto it = filesToExclude.Begin(); it != end; ++it )
	{
		( *it ).Replace( '/', '\\' );
	}

	// automatically exclude the associated CPP file for a PCH (if there is one)
	if ( precompiledHeader.EndsWithI( ".h" ) )
	{
		AStackString<> pchCPP( precompiledHeader.Get(), 
							   precompiledHeader.Get() + precompiledHeader.GetLength() - 2 );
		pchCPP += ".cpp";
		filesToExclude.Append( pchCPP );
	}

	// parsing logic should guarantee we have a string for our name
	ASSERT( m_AliasForFunction.IsEmpty() == false );

	AStackString< 512 > fullOutputPath;
	NodeGraph::CleanPath( outputPathV->GetString(), fullOutputPath );
	if ( fullOutputPath.EndsWith( '\\' ) == false )
	{
		fullOutputPath += '\\';
	}

	// Check for existing node
	if ( ng.FindNode( m_AliasForFunction ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, m_AliasForFunction );
		return false;
	}

	UnityNode * un = ng.CreateUnityNode( m_AliasForFunction, // name
										 dirNodes,
										 files,
										 fullOutputPath,
										 outputPattern,
										 numFilesV ? numFilesV->GetInt() : 1,
										 precompiledHeader,
										 pathsToExclude,
										 filesToExclude,
										 isolateWritableFiles );
	ASSERT( un ); (void)un;

	return true;
}

//------------------------------------------------------------------------------
