// FunctionObjectList
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FunctionObjectList.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFIterator.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFParser.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFVariable.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"
#include "Tools/FBuild/FBuildCore/Graph/AliasNode.h"
#include "Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectListNode.h"
#include "Tools/FBuild/FBuildCore/Graph/ObjectNode.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionObjectList::FunctionObjectList()
: Function( "ObjectList" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionObjectList::AcceptsHeader() const
{
	return true;
}

// NeedsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionObjectList::NeedsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionObjectList::Commit( const BFFIterator & funcStartIter ) const
{
	// make sure all required variables are defined
	const BFFVariable * compiler;
	const BFFVariable * compilerOptions;
	const BFFVariable * compilerOutputPath;
	const BFFVariable * compilerOutputExtension;
	if ( !GetString( funcStartIter, compiler, ".Compiler", true ) ||
		 !GetString( funcStartIter, compilerOptions, ".CompilerOptions", true ) ||
		 !GetString( funcStartIter, compilerOutputPath, ".CompilerOutputPath", true ) ||
		 !GetString( funcStartIter, compilerOutputExtension, ".CompilerOutputExtension", false ) )
	{
		return false;
	}

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// find or create the compiler node
	Node * cn = ng.FindNode( compiler->GetString() );
	FileNode * compilerNode = nullptr;
	if ( cn != nullptr )
	{
		if ( cn->GetType() == Node::ALIAS_NODE )
		{
			AliasNode * an = cn->CastTo< AliasNode >();
			cn = an->GetAliasedNodes()[ 0 ];
		}
		if ( cn->IsAFile() == false )
		{
			Error::Error_1103_NotAFile( funcStartIter, this, "Compiler", cn->GetName(), cn->GetType() );
			return false;
		}
		compilerNode = cn->CastTo< FileNode >();
	}
	else
	{
		compilerNode = ng.CreateFileNode( compiler->GetString() );
	}

	// Sanity check compile flags
	uint32_t objFlags = ObjectNode::DetermineFlags( compilerNode->GetName(), compilerOptions->GetString() );
	if ( ( objFlags & ObjectNode::FLAG_MSVC ) && ( objFlags & ObjectNode::FLAG_CREATING_PCH ) )
	{
		// must not specify use of precompiled header (must use the PCH specific options)
		Error::Error_1303_PCHCreateOptionOnlyAllowedOnPCH( funcStartIter, this, "/Yc", "CompilerOptions" );
		return false;
	}

	// Compiler Force Using
	Array< Node * > compilerForceUsing;
	if ( !GetNodeList( funcStartIter, ".CompilerForceUsing", compilerForceUsing, false ) )
	{
		return false; // GetNodeList will have emitted an error
	}

	// Pre-build dependencies
	Array< Node * > preBuildDependencies;
	if ( !GetNodeList( funcStartIter, ".PreBuildDependencies", preBuildDependencies, false ) )
	{
		return false; // GetNodeList will have emitted an error
	}

	// Precompiled Header support
	ObjectNode * precompiledHeaderNode = nullptr;
	const BFFVariable * pchInputFile = nullptr;
	const BFFVariable * pchOutputFile = nullptr;
	const BFFVariable * pchOptions = nullptr;
	if ( !GetString( funcStartIter, pchInputFile, ".PCHInputFile" ) ||
		 !GetString( funcStartIter, pchOutputFile, ".PCHOutputFile" ) ||
		 !GetString( funcStartIter, pchOptions, ".PCHOptions" ) )
	{
		return false;
	}
	if ( pchInputFile ) 
	{
		if ( !pchOutputFile || !pchOptions )
		{
			Error::Error_1300_MissingPCHArgs( funcStartIter, this );
			return false;
		}
		Node * pchInputNode = ng.FindNode( pchInputFile->GetString() );
		if ( pchInputNode )
		{
			// is it a file?
			if ( pchInputNode->IsAFile() == false )
			{
				Error::Error_1103_NotAFile( funcStartIter, this, "PCHInputFile", pchInputNode->GetName(), pchInputNode->GetType() );
				return false;
			}
		}
		else
		{
			// Create input node
			pchInputNode = ng.CreateFileNode( pchInputFile->GetString() );
		}

		if ( ng.FindNode( pchOutputFile->GetString() ) )
		{
			Error::Error_1301_AlreadyDefinedPCH( funcStartIter, this, pchOutputFile->GetString().Get() );
			return false;
		}

		uint32_t pchFlags = ObjectNode::DetermineFlags( compilerNode->GetName(), pchOptions->GetString() );
		if ( pchFlags & ObjectNode::FLAG_MSVC )
		{
			// sanity check arguments

			// PCH must have "Create PCH" (e.g. /Yc"PrecompiledHeader.h")
			if ( !( pchFlags & ObjectNode::FLAG_CREATING_PCH ) )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Yc", "PCHOptions" );
				return false;
			}
			// PCH must have "Precompiled Header to Use" (e.g. /Fp"PrecompiledHeader.pch")
			if ( pchOptions->GetString().Find( "/Fp" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fp", "PCHOptions" );
				return false;
			}
			// PCH must have object output option (e.g. /Fo"PrecompiledHeader.obj")
			if ( pchOptions->GetString().Find( "/Fo" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fo", "PCHOptions" );
				return false;
			}

			// Object using the PCH must have "Use PCH" option (e.g. /Yu"PrecompiledHeader.h")
			if ( !( objFlags & ObjectNode::FLAG_USING_PCH ) )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Yu", "CompilerOptions" );
				return false;
			}
			// Object using the PCH must have "Precompiled header to use" (e.g. /Fp"PrecompiledHeader.pch")
			if ( compilerOptions->GetString().Find( "/Fp" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fp", "CompilerOptions" );
				return false;
			}
		}

		precompiledHeaderNode = ng.CreateObjectNode( pchOutputFile->GetString(),
													 pchInputNode,
													 compilerNode,
													 pchOptions->GetString(),
													 nullptr,
													 pchFlags,
													 compilerForceUsing );
	}

	Array< Node * > staticDeps( 32, true );

	// do we want to build files via a unity blob?
	const BFFVariable * inputUnity = nullptr;
	if ( !GetString( funcStartIter, inputUnity, ".CompilerInputUnity" ) )
	{
		return false;
	}
	if ( inputUnity )
	{
		Node * n = ng.FindNode( inputUnity->GetString() );
		if ( n == nullptr )
		{
			Error::Error_1104_TargetNotDefined( funcStartIter, this, "CompilerInputUnity", inputUnity->GetString() );
			return false;
		}
		if ( n->GetType() != Node::UNITY_NODE )
		{
			Error::Error_1102_UnexpectedType( funcStartIter, this, "CompilerInputUnity", inputUnity->GetString(), n->GetType(), Node::UNITY_NODE );
			return false;
		}
		staticDeps.Append( n );
	}

	// do we want to build a files in a directory?
	const BFFVariable * inputPath = BFFStackFrame::GetVar( ".CompilerInputPath" );
	if ( inputPath )
	{
		// get the optional pattern and recurse options related to InputPath
		const BFFVariable * patternVar = nullptr;
		if ( !GetString( funcStartIter, patternVar, ".CompilerInputPattern", false ) )
		{
			return false; // GetString will have emitted an error
		}
		AStackString<> defaultWildCard( "*.cpp" );
		const AString & pattern = patternVar ? patternVar->GetString() : defaultWildCard;

		// recursive?  default to true
		bool recurse = true;
		if ( !GetBool( funcStartIter, recurse, ".CompilerInputPathRecurse", true, false ) )
		{
			return false; // GetBool will have emitted an error
		}

		// Support an exclusion path
		const BFFVariable * excludePathVar = nullptr;
		if ( !GetString( funcStartIter, excludePathVar, ".CompilerInputExcludePath", false ) )
		{
			return false; // GetString will have emitted an error
		}
		const AString & excludePath = excludePathVar ? excludePathVar->GetString() : AString::GetEmpty();
		AStackString< 512 > fullExcludePath;
		if ( excludePath.IsEmpty() == false )
		{
			NodeGraph::CleanPath( excludePath, fullExcludePath );
			if ( fullExcludePath.EndsWith( '\\' ) == false )
			{
				fullExcludePath += '\\';
			}
		}

		Array< AString > paths;
		if ( inputPath->IsString() )
		{
			paths.Append( inputPath->GetString() );
		}
		else if ( inputPath->IsArrayOfStrings() )
		{
			paths = inputPath->GetArrayOfStrings();
		}
		else
		{
			Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "CompilerInputPath", inputPath->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY_OF_STRINGS );
			return false;
		}

		CleanFolderPaths( paths );

		auto end = paths.End();
		for ( auto it = paths.Begin(); it != end; ++it )
		{
			const AString & path = *it;

			// get node for the dir we depend on
			AStackString<> name;
			DirectoryListNode::FormatName( path, pattern, recurse, fullExcludePath, name );
			Node * staticDep = ng.FindNode( name );
			if ( staticDep == nullptr )
			{
				staticDep = ng.CreateDirectoryListNode( name,
														path,
														pattern,
														recurse,
														fullExcludePath );
			}
			else if ( staticDep->GetType() != Node::DIRECTORY_LIST_NODE )
			{
				Error::Error_1102_UnexpectedType( funcStartIter, this, "CompilerInputPath", staticDep->GetName(), staticDep->GetType(), Node::DIRECTORY_LIST_NODE );
				return false;
			}

			staticDeps.Append( staticDep );
		}
	}

	// do we want to build a specific list of files?
	if ( !GetNodeList( funcStartIter, ".CompilerInputFiles", staticDeps, false ) )
	{
		// helper will emit error
		return false;
	}

	if ( staticDeps.IsEmpty() )
	{
		Error::Error_1006_NothingToBuild( funcStartIter, this );
		return false;
	}

	// parsing logic should guarantee we have a string for our name
	ASSERT( m_AliasForFunction.IsEmpty() == false );

	// Check for existing node
	if ( ng.FindNode( m_AliasForFunction ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, m_AliasForFunction );
		return false;
	}

	// Create library node which depends on the single file or list
	ObjectListNode * o = ng.CreateObjectListNode( m_AliasForFunction,
												  staticDeps,
												  compilerNode,
												  compilerOptions->GetString(),
												  compilerOutputPath->GetString(),
												  precompiledHeaderNode,
												  compilerForceUsing,
												  preBuildDependencies );
	if ( compilerOutputExtension )
	{
		o->m_ObjExtensionOverride = compilerOutputExtension->GetString();
	}

	return true;
}

//------------------------------------------------------------------------------
