// FunctionAlias
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FunctionCopy.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Flog.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFVariable.h"
#include "Tools/FBuild/FBuildCore/Graph/AliasNode.h"
#include "Tools/FBuild/FBuildCore/Graph/CopyNode.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionCopy::FunctionCopy()
: Function( "Copy" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionCopy::AcceptsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionCopy::Commit( const BFFIterator & funcStartIter ) const
{
	// make sure all required variables are defined
	Array< AString > sources( 16, true );
	const BFFVariable * dstFileV;
	if ( !GetStrings( funcStartIter, sources, ".Source", true ) ||
		 !GetString( funcStartIter, dstFileV, ".Dest", true ) )
	{
		return false; // GetString will have emitted errors
	}

	// check sources are not paths
	{
		const auto end = sources.End();
		for ( auto it = sources.Begin(); it != end; ++it )
		{
			const AString & srcFile( *it );

			// source must be a file, not a  path
			if ( srcFile.EndsWith( '\\' ) || srcFile.EndsWith( '/' ) )
			{
				Error::Error_1105_PathNotAllowed( funcStartIter, this, ".Source", srcFile );
				return false;
			}
		}
	}

	// get source node
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Array< Node * > srcNodes;
	{
		const auto end = sources.End();
		for ( auto it = sources.Begin(); it != end; ++it )
		{

			Node * srcNode = ng.FindNode( *it );
			if ( srcNode )
			{
				if ( GetSourceNodes( funcStartIter, srcNode, srcNodes ) == false )
				{
					return false;
				}
			}
			else
			{
				// source file not defined by use - assume an external file
				srcNodes.Append( ng.CreateFileNode( *it ) );
			}
		}
	}

	if ( srcNodes.IsEmpty() )
	{
		Error::Error_1006_NothingToBuild( funcStartIter, this );
		return false;
	}

	AStackString<> dstFile;
	NodeGraph::CleanPath( dstFileV->GetString(), dstFile );

	// make all the nodes for copies
	Array< Node * > copyNodes( srcNodes.GetSize(), false );
	const auto end = srcNodes.End();
	for ( auto it = srcNodes.Begin(); it != end; ++it )
	{
		AStackString<> dst( dstFile );

		// dest can be a file OR a path.  If it's a path, use the source filename part
		if ( dstFile.EndsWith( '\\' ) )
		{
			// find filename part of source
			const AString & srcName = ( *it )->GetName();
			const char * lastSlash = srcName.FindLast( '\\' );
			dst += lastSlash ? ( lastSlash + 1 )	// append filename part if found
								 : srcName.Get();	// otherwise append whole thing
		}

		// check node doesn't already exist
		if ( ng.FindNode( dst ) )
		{
			// TODO:C could have a specific error for multiple sources with only 1 output
			// to differentiate from two rules creating the same dst target
			Error::Error_1100_AlreadyDefined( funcStartIter, this, dst );
			return false;
		}

		// create our node
		Node * copyNode = ng.CreateCopyNode( dst, *it );
		copyNodes.Append( copyNode );
	}

	// handle alias creation
	return ProcessAlias( funcStartIter, copyNodes );
}

// GetSourceNodes
//------------------------------------------------------------------------------
bool FunctionCopy::GetSourceNodes( const BFFIterator & iter, Node * node, Array< Node * > & nodes ) const
{
	if ( node->GetType() == Node::ALIAS_NODE )
	{
		// resolve aliases to real nodes
		AliasNode * aliasNode = node->CastTo< AliasNode >();
		const Array< Node * > & aliasedNodes = aliasNode->GetAliasedNodes();
		auto end = aliasedNodes.End();
		for ( auto it = aliasedNodes.Begin(); it != end; ++it )
		{
			if ( !GetSourceNodes( iter, *it, nodes ) )
			{
				return false;
			}
		}
		return true;
	}
	else if ( node->IsAFile() )
	{
		// anything that results in a file is ok
		nodes.Append( node );
		return true;
	}

	// something we don't know how to handle
	Error::Error_1005_UnsupportedNodeType( iter, this, ".Source", node->GetName(), node->GetType() );
	return false;
}

//------------------------------------------------------------------------------
