// Function
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Function.h"
#include "FunctionAlias.h"
#include "FunctionCompiler.h"
#include "FunctionCopy.h"
#include "FunctionCSAssembly.h"
#include "FunctionDLL.h"
#include "FunctionExec.h"
#include "FunctionExecutable.h"
#include "FunctionForEach.h"
#include "FunctionLibrary.h"
#include "FunctionObjectList.h"
#include "FunctionPrint.h"
#include "FunctionSettings.h"
#include "FunctionTest.h"
#include "FunctionUnity.h"
#include "FunctionUsing.h"
#include "FunctionVCXProject.h"

#include "Tools/FBuild/FBuildCore/BFF/BFFIterator.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFParser.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h"
#include "Tools/FBuild/FBuildCore/BFF/BFFVariable.h"
#include "Tools/FBuild/FBuildCore/FBuild.h"
#include "Tools/FBuild/FBuildCore/Graph/AliasNode.h"
#include "Tools/FBuild/FBuildCore/Graph/FileNode.h"
#include "Tools/FBuild/FBuildCore/Graph/NodeGraph.h"

#include "Core/Strings/AStackString.h"

#include <stdarg.h>
#include <stdio.h>

// Static
//------------------------------------------------------------------------------
/*static*/ Function * Function::s_FirstFunction = nullptr;

// CONSTRUCTOR
//------------------------------------------------------------------------------
Function::Function( const char * name )
: m_NextFunction( nullptr )
, m_Name( name )
, m_Seen( false )
, m_AliasForFunction( 256 )
{
	if ( s_FirstFunction == nullptr )
	{
		s_FirstFunction = this;
		return;
	}
	Function * func = s_FirstFunction;
	while ( func )
	{
		if ( func->m_NextFunction == nullptr )
		{
			func->m_NextFunction = this;
			return;
		}
		func = func->m_NextFunction;
	}
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Function::~Function()
{
}

// Find
//------------------------------------------------------------------------------
/*static*/ const Function * Function::Find( const AString & name )
{
	Function * func = s_FirstFunction;
	while ( func )
	{
		if ( func->GetName() == name )
		{
			return func;
		}
		func = func->m_NextFunction;
	}
	return nullptr;
}

// Create
//------------------------------------------------------------------------------
/*static*/ void Function::Create()
{
	new FunctionAlias;
	new FunctionCompiler;
	new FunctionCopy;
	new FunctionCSAssembly;
	new FunctionDLL;
	new FunctionExec;
	new FunctionExecutable;
	new FunctionForEach;
	new FunctionLibrary;
	new FunctionPrint;
	new FunctionSettings;
	new FunctionTest;
	new FunctionUnity;
	new FunctionUsing;
	new FunctionVCXProject;
	new FunctionObjectList;
}

// Destroy
//------------------------------------------------------------------------------
/*static*/ void Function::Destroy()
{
	Function * func = s_FirstFunction;
	while ( func )
	{
		Function * nextFunc = func->m_NextFunction;
		delete func;
		func = nextFunc;
	}
	s_FirstFunction = nullptr;
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool Function::AcceptsHeader() const
{
	return false;
}

// NeedsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool Function::NeedsHeader() const
{
	return false;
}

// NeedsBody
//------------------------------------------------------------------------------
/*virtual*/ bool Function::NeedsBody() const
{
	return true;
}

// IsUnique
//------------------------------------------------------------------------------
/*virtual*/ bool Function::IsUnique() const
{
	return false;
}

// ParseFunction
//------------------------------------------------------------------------------
/*virtual*/ bool Function::ParseFunction( const BFFIterator & functionNameStart,
										  const BFFIterator * functionBodyStartToken, 
										  const BFFIterator * functionBodyStopToken,
										  const BFFIterator * functionHeaderStartToken,
										  const BFFIterator * functionHeaderStopToken ) const
{
	m_AliasForFunction.Clear();
	if ( AcceptsHeader() &&
		 functionHeaderStartToken && functionHeaderStopToken &&
		 ( functionHeaderStartToken->GetDistTo( *functionHeaderStopToken ) > 1 ) )
	{
		// find opening quote
		BFFIterator start( *functionHeaderStartToken );
		ASSERT( *start == BFFParser::BFF_FUNCTION_ARGS_OPEN );
		start++;
		start.SkipWhiteSpace();
		const char c = *start;
		if ( ( c != '"' ) && ( c != '\'' ) )
		{
			Error::Error_1001_MissingStringStartToken( start, this ); 
			return false;
		}
		BFFIterator stop( start );
		stop.SkipString( c );
		ASSERT( stop.GetCurrent() <= functionHeaderStopToken->GetCurrent() ); // should not be in this function if strings are not validly terminated
		if ( start.GetDistTo( stop ) <= 1 )
		{
			Error::Error_1003_EmptyStringNotAllowedInHeader( start, this );
			return false;
		}

		// store alias name for use in Commit
		start++; // skip past opening quote
		if ( BFFParser::PerformVariableSubstitutions( start, stop, m_AliasForFunction ) == false )
		{
			return false; // substitution will have emitted an error
		}
	}

	// parse the function body
	BFFParser subParser;
	BFFIterator subIter( *functionBodyStartToken );
	subIter++; // skip past opening body token
	subIter.SetMax( functionBodyStopToken->GetCurrent() ); // cap parsing to body end
	if ( subParser.Parse( subIter ) == false )
	{
		return false;
	}

	// complete the function
	return Commit( functionNameStart );
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool Function::Commit( const BFFIterator & funcStartIter ) const
{
	(void)funcStartIter;
	return true;
}

// GetString
//------------------------------------------------------------------------------
bool Function::GetString( const BFFIterator & iter, const BFFVariable * & var, const char * name, bool required ) const
{
	ASSERT( name );
	var = nullptr;

	const BFFVariable * v = BFFStackFrame::GetVar( name );

	if ( v == nullptr )
	{
		if ( required )
		{
			Error::Error_1101_MissingProperty( iter, this, AStackString<>( name ) );
			return false;
		}
		return true;
	}

	if ( v->IsString() == false )
	{
		Error::Error_1050_PropertyMustBeOfType( iter, this, name, v->GetType(), BFFVariable::VAR_STRING );
		return false;
	}
	if ( v->GetString().IsEmpty() )
	{
		Error::Error_1004_EmptyStringPropertyNotAllowed( iter, this, name );
		return false;
	}

	var = v;
	return true;
}

// GetStringOrArrayOfStrings
//------------------------------------------------------------------------------
bool Function::GetStringOrArrayOfStrings( const BFFIterator & iter, const BFFVariable * & var, const char * name, bool required ) const
{
	ASSERT( name );
	var = nullptr;

	const BFFVariable * v = BFFStackFrame::GetVar( name );

	if ( v == nullptr )
	{
		if ( required )
		{
			Error::Error_1101_MissingProperty( iter, this, AStackString<>( name ) );
			return false;
		}
		return true;
	}


	// UnityInputPath can be string or array of strings
	Array< AString > paths;
	if ( v->IsString() || v->IsArrayOfStrings() )
	{
		var = v;
		return true;
	}

	Error::Error_1050_PropertyMustBeOfType( iter, this, name, v->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY_OF_STRINGS );
	return false;
}

// GetBool
//------------------------------------------------------------------------------
bool Function::GetBool( const BFFIterator & iter, bool & var, const char * name, bool defaultValue, bool required ) const
{
	ASSERT( name );

	const BFFVariable * v = BFFStackFrame::GetVar( name );
	if ( v == nullptr )
	{
		if ( required )
		{
			Error::Error_1101_MissingProperty( iter, this, AStackString<>( name ) );
			return false;
		}
		var = defaultValue;
		return true;
	}

	if ( v->IsBool() == false )
	{
		Error::Error_1050_PropertyMustBeOfType( iter, this, name, v->GetType(), BFFVariable::VAR_BOOL );
		return false;
	}

	var = v->GetBool();
	return true;
}

// GetInt
//------------------------------------------------------------------------------
bool Function::GetInt( const BFFIterator & iter, int32_t & var, const char * name, int32_t defaultValue, bool required ) const
{
	ASSERT( name );

	const BFFVariable * v = BFFStackFrame::GetVar( name );
	if ( v == nullptr )
	{
		if ( required )
		{
			Error::Error_1101_MissingProperty( iter, this, AStackString<>( name ) );
			return false;
		}
		var = defaultValue;
		return true;
	}

	if ( v->IsInt() == false )
	{
		Error::Error_1050_PropertyMustBeOfType( iter, this, name, v->GetType(), BFFVariable::VAR_INT );
		return false;
	}

	var = v->GetInt();
	return true;
}

// GetNodeList
//------------------------------------------------------------------------------
bool Function::GetNodeList( const BFFIterator & iter, const char * name, Array< Node * > & nodes, bool required ) const
{
	ASSERT( name );

	const BFFVariable * var = BFFStackFrame::GetVar( name );
	if ( !var )
	{
		// missing
		if ( required )
		{
			Error::Error_1101_MissingProperty( iter, this, AStackString<>( name ) );
			return false; // required!
		}
		return true; // missing but not required
	}

	if ( var->IsArrayOfStrings() )
	{
		// an array of references
		const Array< AString > & nodeNames = var->GetArrayOfStrings();
		nodes.SetCapacity( nodes.GetSize() + nodeNames.GetSize() );
		for ( Array< AString >::Iter it = nodeNames.Begin();
				it != nodeNames.End();
				it++ )
		{
			if ( !GetNodeListRecurse( iter, name, nodes, *it ) )
			{
				// child func will have emitted error
				return false;
			}
		}
	}
	else if ( var->IsString() )
	{
		if ( !GetNodeListRecurse( iter, name, nodes, var->GetString() ) )
		{
			// child func will have emitted error
			return false;
		}
	}
	else
	{
		// unsupported type
		Error::Error_1050_PropertyMustBeOfType( iter, this, name, var->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY_OF_STRINGS );
		return false;
	}

	return true;
}

// GetNodeListRecurse
//------------------------------------------------------------------------------
bool Function::GetNodeListRecurse( const BFFIterator & iter, const char * name, Array< Node * > & nodes, const AString & nodeName ) const
{
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// get node
	Node * n = ng.FindNode( nodeName );
	if ( n == nullptr )
	{
		// not found - create a new file node
		n = ng.CreateFileNode( nodeName );
		nodes.Append( n );
		return true;
	}

	// found - is it a file?
	if ( n->IsAFile() )
	{
		// found file - just use as is
		nodes.Append( n );
		return true;
	}

	// found - is it an ObjectList?
	if ( n->GetType() == Node::OBJECT_LIST_NODE )
	{
		// use as-is
		nodes.Append( n );
		return true;
	}

	// found - is it a group?
	if ( n->GetType() == Node::ALIAS_NODE )
	{
		AliasNode * an = n->CastTo< AliasNode >();
		const Array< Node * > & aNodes = an->GetAliasedNodes();
		for ( auto it = aNodes.Begin(); it != aNodes.End(); ++it )
		{
			// TODO:C by passing as string we'll be looking up again for no reason
			const AString & subName = ( *it )->GetName();

			if ( !GetNodeListRecurse( iter, name, nodes, subName ) )
			{
				return false;
			}
		}
		return true;
	}

	// don't know how to handle this type of node
	Error::Error_1005_UnsupportedNodeType( iter, this, name, n->GetName(), n->GetType() );
	return false;
}

// GetStrings
//------------------------------------------------------------------------------
bool Function::GetStrings( const BFFIterator & iter, Array< AString > & strings, const char * name, bool required ) const
{
	const BFFVariable * var;
	if ( !GetStringOrArrayOfStrings( iter, var, name, required ) )
	{
		return false; // problem (GetStringOrArrayOfStrings will have emitted error)
	}
	if ( !var )
	{
		ASSERT( !required );
		return true; // not required and not provided: nothing to do
	}

	if ( var->GetType() == BFFVariable::VAR_STRING )
	{
		strings.Append( var->GetString() );
	}
	else if ( var->GetType() == BFFVariable::VAR_ARRAY_OF_STRINGS )
	{
		const Array< AString > & vStrings = var->GetArrayOfStrings();
		strings.Append( vStrings );
	}
	else
	{
		ASSERT( false );
	}
	return true;
}

// CleanFolderPaths
//------------------------------------------------------------------------------
void Function::CleanFolderPaths( Array< AString > & folders ) const
{
	AStackString< 512 > tmp;

	auto const end = folders.End();
	for ( auto it = folders.Begin(); it != end; ++it )
	{
		// make full path, clean slashes etc
		NodeGraph::CleanPath( *it, tmp );

		// ensure path is slash-terminated
		if ( tmp.EndsWith( '\\' ) == false )
		{
			tmp += '\\';
		}

		// replace original
		*it = tmp;
	}
}

//------------------------------------------------------------------------------
void Function::CleanFilePaths( Array< AString > & files ) const
{
	AStackString< 512 > tmp;

	auto const end = files.End();
	for ( auto it = files.Begin(); it != end; ++it )
	{
		// make full path, clean slashes etc
		NodeGraph::CleanPath( *it, tmp );

		// replace original
		*it = tmp;
	}
}

// ProcessAlias
//------------------------------------------------------------------------------
bool Function::ProcessAlias( const BFFIterator & iter, Node * nodeToAlias ) const
{
	Array< Node * > nodesToAlias( 1, false );
	nodesToAlias.Append( nodeToAlias );
	return ProcessAlias( iter, nodesToAlias );
}

// ProcessAlias
//------------------------------------------------------------------------------
bool Function::ProcessAlias( const BFFIterator & iter, Array< Node * > & nodesToAlias ) const
{
	if ( m_AliasForFunction.IsEmpty() )
	{
		return true; // no alias required
	}

	// check for duplicates
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	if ( ng.FindNode( m_AliasForFunction ) )
	{
		Error::Error_1100_AlreadyDefined( iter, this, m_AliasForFunction );
		return false;
	}

	// create an alias against the node
	VERIFY( ng.CreateAliasNode( m_AliasForFunction, nodesToAlias ) );

	// clear the string so it can't be used again
	m_AliasForFunction.Clear();

	return true;
}

//------------------------------------------------------------------------------
