// Process.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Process.h"

#include "Core/Env/Assert.h"
#include "Core/Math/Conversions.h"
#include "Core/Process/Thread.h"
#include "Core/Strings/AStackString.h"
#include "Core/Strings/AString.h"

#if defined( __WINDOWS__ )
    #include <windows.h>
#endif

// CONSTRUCTOR
//------------------------------------------------------------------------------
Process::Process()
: m_Started( false )
{
    #if defined( __WINDOWS__ )
        CTASSERT( sizeof( m_ProcessInfo ) == sizeof( PROCESS_INFORMATION ) );
    #endif
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Process::~Process()
{
	if ( m_Started )
	{
		WaitForExit();
	}
}

// Spawn
//------------------------------------------------------------------------------
bool Process::Spawn( const char * executable,
					 const char * args,
					 const char * workingDir,
					 const char * environment )
{
	ASSERT( !m_Started );
	ASSERT( executable );

    #if defined( __WINDOWS__ )
        // Set up the start up info struct.
        STARTUPINFO si;
        ZeroMemory( &si, sizeof(STARTUPINFO) );
		si.cb = sizeof( STARTUPINFO );
        si.dwFlags |= STARTF_USESHOWWINDOW;
        si.wShowWindow = SW_HIDE;
        
        SECURITY_ATTRIBUTES sa;
        ZeroMemory( &sa, sizeof( SECURITY_ATTRIBUTES ) );
        sa.nLength = sizeof(SECURITY_ATTRIBUTES);
        sa.bInheritHandle = TRUE;
        sa.lpSecurityDescriptor = nullptr;

        // create the pipes
        if ( ! CreatePipe( &m_StdOutRead, &m_StdOutWrite, &sa, MEGABYTE ) )
        {
            return false;
        }
        SetHandleInformation( m_StdOutRead, HANDLE_FLAG_INHERIT, 0 );

        if ( ! CreatePipe( &m_StdErrRead, &m_StdErrWrite, &sa, MEGABYTE ) )
        {
            VERIFY( CloseHandle( m_StdOutRead ) );
            VERIFY( CloseHandle( m_StdOutWrite ) );
            return false;
        }
        SetHandleInformation( m_StdErrRead, HANDLE_FLAG_INHERIT, 0 );

        si.hStdOutput = m_StdOutWrite;
        si.hStdError = m_StdErrWrite;
        si.hStdInput = GetStdHandle(STD_INPUT_HANDLE); // m_StdInRead;
        si.dwFlags |= STARTF_USESTDHANDLES;
        
        // Make sure the first arg is the executable
        // We also need to make a copy, as CreateProcess can write back to this string
        AStackString< 1024 > fullArgs;
        fullArgs += '\"';
        fullArgs += executable;
        fullArgs += '\"';
		if ( args )
		{
	        fullArgs += ' ';
	        fullArgs += args;
		}
        //fullArgs.Format( "\"%s\" %s", executable, args );

        // create the child
        if ( !CreateProcess( nullptr, //executable,
                              fullArgs.Get(),
                              nullptr,
                              nullptr,
                              TRUE, // inherit handles
                              0,
                              (void *)environment,
                              workingDir,
                              &si,
                              (LPPROCESS_INFORMATION)&m_ProcessInfo ) )
        {
            return false;
        }

        m_Started = true;
        return true;
    #elif defined( __APPLE__ )
        return false; // TODO:MAC Implement Process
    #else
        #error Unknown platform
    #endif
}

// IsRunning
//----------------------------------------------------------
bool Process::IsRunning() const
{
	ASSERT( m_Started );

    #if defined( __WINDOWS__ )
        switch ( WaitForSingleObject( GetProcessInfo().hProcess, 0 ) )
        {
            case WAIT_OBJECT_0:
                return false;

            case WAIT_TIMEOUT:
                return true;
        }
        ASSERT( false ); // we should never get here
        return false;
    #elif defined( __APPLE__ )
        return false; // TODO:MAC Implement Process
    #else
        #error Unknown platform
    #endif
}

// WaitForExit
//------------------------------------------------------------------------------
int Process::WaitForExit()
{
	ASSERT( m_Started );
	m_Started = false;

    #if defined( __WINDOWS__ )
        // wait for it to finish
        VERIFY( WaitForSingleObject( GetProcessInfo().hProcess, INFINITE ) == WAIT_OBJECT_0 );

        // get the result code
        DWORD exitCode = 0;
        VERIFY( GetExitCodeProcess( GetProcessInfo().hProcess, (LPDWORD)&exitCode ) );

        // cleanup
        VERIFY( CloseHandle( GetProcessInfo().hProcess ) );
        VERIFY( CloseHandle( GetProcessInfo().hThread ) );

    //	VERIFY( CloseHandle( m_StdInRead ) );
    //	VERIFY( CloseHandle( m_StdInWrite ) );
        VERIFY( CloseHandle( m_StdOutRead ) );
        VERIFY( CloseHandle( m_StdOutWrite ) );
        VERIFY( CloseHandle( m_StdErrRead ) );
        VERIFY( CloseHandle( m_StdErrWrite ) );

        return exitCode;
    #elif defined( __APPLE__ )
        return 0; // TODO:MAC Implement Process
    #else
        #error Unknown platform
    #endif
}

// ReadAllData
//------------------------------------------------------------------------------
void Process::ReadAllData( AutoPtr< char > & outMem, uint32_t * outMemSize,
						   AutoPtr< char > & errMem, uint32_t * errMemSize )
{
    #if defined( __WINDOWS__ )
        // we'll capture into these growing buffers
    //	char * bufferOut = nullptr;
    //	char * bufferErr = nullptr;
        uint32_t outSize = 0;
        uint32_t errSize = 0;
        uint32_t outBufferSize = 0;
        uint32_t errBufferSize = 0;

        bool processExited = false;
        for ( ;; )
        {
            uint32_t prevOutSize = outSize;
            uint32_t prevErrSize = errSize;
            /*bufferOut =*/ Read( m_StdOutRead, outMem, outSize, outBufferSize );
            /*bufferErr =*/ Read( m_StdErrRead, errMem, errSize, errBufferSize );

            // did we get some data?
            if ( ( prevOutSize != outSize ) || ( prevErrSize != errSize ) )
            {
                continue; // try reading again right away incase there is more
            }

            // nothing to read right now
            if ( IsRunning() )
            {
                // no data available, but process is still going, so wait
                Thread::Sleep( 32 );
                continue;
            }

            // process exited - is this the first time to this point?
            if ( processExited == false )
            {
                processExited = true;
                continue; // get remaining output
            }

            break; // all done
        }

        // if owner asks for pointers, they now own the mem
    //	if ( memOut ) { *memOut = bufferOut; } else { delete[] memOut; }
    //	if ( memErr ) { *memErr = bufferErr; } else { delete[] memErr; }
        if ( outMemSize ) { *outMemSize = outSize; }
        if ( errMemSize ) { *errMemSize = errSize; }
    #elif defined( __APPLE__ )
        // TODO:MAC Implement Process
    #else
        #error Unknown platform
    #endif
}

// Read
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    void Process::Read( HANDLE handle, AutoPtr< char > & buffer, uint32_t & sizeSoFar, uint32_t & bufferSize )
    {
        // anything available?
        DWORD bytesAvail;
        VERIFY( PeekNamedPipe( handle, nullptr, 0, nullptr, (LPDWORD)&bytesAvail, nullptr ) );
        if ( bytesAvail == 0 )
        {
            return;// buffer;
        }

        // will it fit in the buffer we have?
        if ( ( sizeSoFar + bytesAvail ) > bufferSize )
        {
            // no - allocate a bigger buffer (also handles the first time with no buffer)

            // TODO:B look at a new container type (like a linked list of 1mb buffers) to avoid the wasteage here
            // The caller has to take a copy to avoid the overhead if they want to hang onto the data
            // grow buffer in at least 8MB chunks, to prevent future reallocations
            uint32_t newBufferSize = Math::Max< uint32_t >( sizeSoFar + bytesAvail, bufferSize + ( 16 * MEGABYTE ) );
            char * newBuffer = (char *)::Alloc( newBufferSize + 1 ); // +1 so we can always add a null char
            if ( buffer.Get() )
            {
                // transfer and free old buffer
                memcpy( newBuffer, buffer.Get(), sizeSoFar );
            }
            buffer = newBuffer; // will take care of deletion of old buffer
            bufferSize = newBufferSize;
        }

        ASSERT( sizeSoFar + bytesAvail <= bufferSize ); // sanity check

        // read the new data
        DWORD bytesReadNow = 0;
        VERIFY( ReadFile( handle, buffer.Get() + sizeSoFar, bytesAvail, (LPDWORD)&bytesReadNow, 0 ) );
        ASSERT( bytesReadNow == bytesAvail );
        sizeSoFar += bytesReadNow;

        // keep data null char terminated for caller convenience
        buffer.Get()[ sizeSoFar ] = '\000';
    }
#endif

// ReadStdOut
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    char * Process::ReadStdOut( uint32_t * bytesRead )
    {
        return Read( m_StdOutRead, bytesRead );
    }
#endif

// ReadStdOut
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    char * Process::ReadStdErr( uint32_t * bytesRead )
    {
        return Read( m_StdErrRead, bytesRead );
    }
#endif

// ReadStdOut
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    uint32_t Process::ReadStdOut( char * outputBuffer, uint32_t outputBufferSize )
    {
        return Read( m_StdOutRead, outputBuffer, outputBufferSize );
    }
#endif

// ReadStdErr
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    uint32_t Process::ReadStdErr( char * outputBuffer, uint32_t outputBufferSize )
    {
        return Read( m_StdErrRead, outputBuffer, outputBufferSize );
    }
#endif

// Read
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    char * Process::Read( HANDLE handle, uint32_t * bytesRead )
    {
        // see if there's anything in the pipe
        DWORD bytesAvail;
        VERIFY( PeekNamedPipe( handle, nullptr, 0, nullptr, (LPDWORD)&bytesAvail, nullptr ) );
        if ( bytesAvail == 0 )
        {
            if ( bytesRead )
            {
                *bytesRead = 0;
            }
            return nullptr;
        }

        // allocate output buffer
        char * mem = (char *)::Alloc( bytesAvail + 1 ); // null terminate for convenience
        mem[ bytesAvail ] = 0;

        // read the data
        DWORD bytesReadNow = 0;
        VERIFY( ReadFile( handle, mem, bytesAvail, (LPDWORD)&bytesReadNow, 0 ) );
        ASSERT( bytesReadNow == bytesAvail );
        if ( bytesRead )
        {
            *bytesRead = bytesReadNow;
        }
        return mem;
    }
#endif

// Read
//------------------------------------------------------------------------------
#if defined( __WINDOWS__ )
    uint32_t Process::Read( HANDLE handle, char * outputBuffer, uint32_t outputBufferSize )
    {
        // see if there's anything in the pipe
        DWORD bytesAvail;
        VERIFY( PeekNamedPipe( handle, nullptr, 0, 0, (LPDWORD)&bytesAvail, 0 ) );
        if ( bytesAvail == 0 )
        {
            return 0;
        }

        // if there is more available than we have space for, just read as much as we can
        uint32_t bytesToRead = Math::Min<uint32_t>( outputBufferSize, bytesAvail );

        // read the data
        DWORD bytesReadNow = 0;
        VERIFY( ReadFile( handle, outputBuffer, bytesToRead, (LPDWORD)&bytesReadNow, 0 ) );
        ASSERT( bytesReadNow == bytesToRead );
        return bytesToRead;
    }
#endif

//------------------------------------------------------------------------------
