// TCPConnectionPool
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------
#include "TCPConnectionPool.h"

// Core
#include "Core/Mem/Mem.h"
#include "Core/Strings/AString.h"
#include "Core/Strings/AStackString.h"

// System
#if defined( __WINDOWS__ )
    #include <winsock2.h>
    #include <windows.h>
#elif defined( __APPLE__ )
    #include <string.h>
    #include <errno.h>
    #include <netdb.h>
    #include <arpa/inet.h>
    #include <sys/ioctl.h>
    #include <sys/socket.h>
    #include <netinet/in.h>
    #include <fcntl.h>
    #include <unistd.h>
    // TODO:MAC Implement TCPConnectionPool
    #define INVALID_SOCKET (~0) // TODO:MAC Check that this is valid
    #define SOCKET_ERROR -1
#else
    #error Unknown platform
#endif

//------------------------------------------------------------------------------
// For Debugging
//------------------------------------------------------------------------------
//#define TCPCONNECTION_DEBUG
#ifdef TCPCONNECTION_DEBUG
    #include "Core/Tracing/Tracing.h"
	#define TCPDEBUG( ... ) DEBUGSPAM( __VA_ARGS__ )
#else
	#define TCPDEBUG( ... )
#endif

// CONSTRUCTOR - ConnectionInfo
//------------------------------------------------------------------------------
ConnectionInfo::ConnectionInfo( TCPConnectionPool * ownerPool )
: m_Socket( INVALID_SOCKET )
, m_RemoteAddress( 0 )
, m_RemotePort( 0 )
, m_ThreadQuitNotification( false )
, m_TCPConnectionPool( ownerPool )
, m_UserData( nullptr )
#ifdef DEBUG
, m_InUse( false )
#endif
{
	ASSERT( ownerPool );
}

// CONSTRUCTOR
//------------------------------------------------------------------------------
TCPConnectionPool::TCPConnectionPool()
	: m_ListenConnection( nullptr )
	, m_Connections( 8, true )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
TCPConnectionPool::~TCPConnectionPool()
{
	ShutdownAllConnections();
}

// ShutdownAllConnections
//------------------------------------------------------------------------------
void TCPConnectionPool::ShutdownAllConnections()
{
	m_ConnectionsMutex.Lock();

	// signal all remaining connections to close

	// listening connection
	if ( m_ListenConnection )
	{
		Disconnect( m_ListenConnection );
	}

	// incoming connections
	for ( size_t i=0; i<m_Connections.GetSize(); ++i )
	{
		ConnectionInfo * ci = m_Connections[ i ];
		Disconnect( ci );
	}

	// wait for connections to be closed
	while ( m_ListenConnection ||
			!m_Connections.IsEmpty() )
	{
		m_ConnectionsMutex.Unlock();
        Thread::Sleep( 1 );
		m_ConnectionsMutex.Lock();
	}
	m_ConnectionsMutex.Unlock();
}

// GetAddressAsString
//------------------------------------------------------------------------------
/*static*/ void TCPConnectionPool::GetAddressAsString( uint32_t addr, AString & address )
{
    address.Format( "%u.%u.%u.%u", (unsigned int)( addr & 0x000000FF ) ,
                                   (unsigned int)( addr & 0x0000FF00 ) >> 8,
                                   (unsigned int)( addr & 0x00FF0000 ) >> 16,
                                    (unsigned int)( addr & 0xFF000000 ) >> 24 );
}

// Listen
//------------------------------------------------------------------------------
bool TCPConnectionPool::Listen( uint16_t port )
{
	// must not be listening already
	ASSERT( m_ListenConnection == nullptr );

    // create the socket
    TCPSocket sockfd = socket( AF_INET, SOCK_STREAM, 0 );
    if ( sockfd == INVALID_SOCKET )
    {
        TCPDEBUG( "Create socket failed (Listen): %i\n", GetLastError() );
		return false;
    }

    // allow socket re-use
    static const char yes = 1;
    int ret = setsockopt( sockfd, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof( yes ) );
    if ( ret != 0 )
    {
        TCPDEBUG( "setsockopt failed: %i\n", GetLastError() );
        CloseSocket( sockfd );
        return false;
    }

    // set up the listen params
    struct sockaddr_in addrInfo;
    memset( &addrInfo, 0, sizeof( addrInfo ) );
    addrInfo.sin_family = AF_INET;
    addrInfo.sin_port = htons( port );
    addrInfo.sin_addr.s_addr = INADDR_ANY;

    // bind
    if ( bind( sockfd, (struct sockaddr *)&addrInfo, sizeof( addrInfo ) ) != 0 )
    {
        TCPDEBUG( "Bind failed: %i\n", GetLastError() );
        CloseSocket( sockfd );
        return false;
    }

    // listen
    TCPDEBUG( "Listen on port %i (%x)\n", port, sockfd );
    if ( listen( sockfd, 0 ) == SOCKET_ERROR ) // no backlog
	{
	    TCPDEBUG( "Listen FAILED %i (%x)\n", port, sockfd );
        CloseSocket( sockfd );
        return false;
	}

    // spawn the handler thread
	uint32_t loopback = 127 & ( 1 << 24 ); // 127.0.0.1
    CreateListenThread( sockfd, loopback, port );

    // everything is ok - we are now listening, managing connections on the other thread
    return true;
}

// Connect
//------------------------------------------------------------------------------
const ConnectionInfo * TCPConnectionPool::Connect( const AString & host, uint16_t port )
{
	ASSERT( !host.IsEmpty() );

	TCPSocket sockfd = 0;

    // get IP
    uint32_t hostIP = GetIP( host );
	if( hostIP == 0 )
	{
		TCPDEBUG( "Failed to get address for '%s'" , host.Get() );
		return nullptr;
	}

    // create a socket
    sockfd = socket( AF_INET, SOCK_STREAM, 0 );

    // outright failure?
    if ( sockfd == INVALID_SOCKET )
    {
        TCPDEBUG( "Create socket failed (Connect): %i\n", GetLastError() );
        return nullptr;
    }

	// set send/recv timeout
	uint32_t bufferSize = ( 10 * 1024 * 1024 );
    int ret = setsockopt( sockfd, SOL_SOCKET, SO_RCVBUF, (const char *)&bufferSize, sizeof( bufferSize ) );
    if ( ret != 0 )
    {
        TCPDEBUG( "setsockopt SO_RCVBUF failed: %i\n", GetLastError() );
	    CloseSocket( sockfd );
		return nullptr;
    }
    ret = setsockopt( sockfd, SOL_SOCKET, SO_SNDBUF, (const char *)&bufferSize, sizeof( bufferSize ) );
    if ( ret != 0 )
    {
        TCPDEBUG( "setsockopt SO_SNDBUF failed: %i\n", GetLastError() );
	    CloseSocket( sockfd );
		return nullptr;
    }

    // we have a socket now
    //m_Socket = sockfd;

    // setup destination address
    struct sockaddr_in destAddr;
    memset( &destAddr, 0, sizeof( destAddr ) );
    destAddr.sin_family = AF_INET;
    destAddr.sin_port = htons( port );
    destAddr.sin_addr.s_addr = hostIP;

    // initiate connection
    if ( connect(sockfd, (struct sockaddr *)&destAddr, sizeof( destAddr ) ) != 0 )
    {
        // connection failed - probably noone listening
        TCPDEBUG( "connect() failed: %i (host:%s port:%u)\n", GetLastError(), host.Get(), port );
	    CloseSocket( sockfd );
		return nullptr;
    }

	// set non-blocking
	u_long nonBlocking = 1;
    #if defined( __WINDOWS__ )
        ioctlsocket( sockfd, FIONBIO, &nonBlocking );
    #elif defined( __APPLE__ )
        ioctl( sockfd, FIONBIO, &nonBlocking );
    #else
        #error Unknown platform
    #endif
	return CreateConnectionThread( sockfd, hostIP, port );
}

// Disconnect
//------------------------------------------------------------------------------
void TCPConnectionPool::Disconnect( const ConnectionInfo * ci )
{
	ASSERT( ci );

	//
	// The ConnectionInfo is only valid while we still have a
	// pointer to it in our list of connections (or as the special
	// listener connection)
	//

	// ensure the connection thread isn't busy destroying itself
	MutexHolder mh( m_ConnectionsMutex );

	if ( ci == m_ListenConnection )
	{
		ci->m_ThreadQuitNotification = true;
		return;
	}

	auto iter = m_Connections.Find( ci );
	if ( iter != nullptr )
	{
		ci->m_ThreadQuitNotification = true;
		return;
	}

	// connection is no longer valid.... we handle this gracefully
	// as the connection might be lost while trying to disconnect
	// on another thread
}

// GetNumConnections
//------------------------------------------------------------------------------
size_t TCPConnectionPool::GetNumConnections() const
{
	MutexHolder mh( m_ConnectionsMutex );
	return m_Connections.GetSize();
}

// Send
//------------------------------------------------------------------------------
bool TCPConnectionPool::Send( const ConnectionInfo * connection, const void * data, size_t size )
{
	ASSERT( connection );

	// closing connection, possibly from a previous failure
	if ( connection->m_ThreadQuitNotification )
	{
		return false;
	}

#ifdef DEBUG
	ASSERT( connection->m_InUse == false );
	connection->m_InUse = true;
#endif

	ASSERT( connection->m_Socket != INVALID_SOCKET );

	TCPDEBUG( "Send: %i (%x)\n", size, connection->m_Socket );

    // send size of subsequent data
	uint32_t sizeData = (uint32_t)size;
	uint32_t bytesToSend = 4;
	while ( bytesToSend > 0 )
	{
		int sent = (int)send( connection->m_Socket, ( (const char *)&sizeData ) + 4 - bytesToSend, bytesToSend, 0 );
        if ( sent <= 0 )
        {
            if ( WouldBlock() )
            {
                Thread::Sleep( 1 );
                continue;
            }
			// error
			TCPDEBUG( "send error A.  Send: %i (Error: %i) (%x)\n", sent, GetLastError(), connection->m_Socket );
			Disconnect( connection );
			#ifdef DEBUG
				connection->m_InUse = false;
			#endif
			return false;
		}
		bytesToSend -= sent;
	}

    // loop until we send all data
    size_t bytesRemaining = size;
    const char * dataAsChar = (const char *)data;
    while ( bytesRemaining > 0 )
    {
        int sent = (int)send( connection->m_Socket, dataAsChar, (uint32_t)bytesRemaining, 0 );
		if ( sent <= 0 )
		{
			if ( WouldBlock() )
			{
                Thread::Sleep( 1 );
				continue;
			}
			// error
			TCPDEBUG( "send error B.  Send: %i (Error: %i) (%x)\n", sent, GetLastError(), connection->m_Socket );
			Disconnect( connection );
			#ifdef DEBUG
				connection->m_InUse = false;
			#endif
            return false;
        }
        bytesRemaining -= sent;
        dataAsChar += sent;
    }
	#ifdef DEBUG
		connection->m_InUse = false;
	#endif
    return true;
}

// Broadcast
//------------------------------------------------------------------------------
bool TCPConnectionPool::Broadcast( const void * data, size_t size )
{
	MutexHolder mh( m_ConnectionsMutex );

	bool result = true;

	auto it = m_Connections.Begin();
	auto end = m_Connections.End();
	while ( it < end )
	{
		result |= Send( *it, data, size );
		it++;
	}
	return result;
}

// AllocBuffer
//------------------------------------------------------------------------------
/*virtual*/ void * TCPConnectionPool::AllocBuffer( uint32_t size )
{
	return ::Alloc( size );
}

// FreeBuffer
//------------------------------------------------------------------------------
/*virtual*/ void TCPConnectionPool::FreeBuffer( void * data )
{
	::Free( data );
}

// GetIP
//------------------------------------------------------------------------------
uint32_t TCPConnectionPool::GetIP( const AString & a_Host ) const
{
	uint32_t ip = 0;
	if( inet_addr( a_Host.Get() ) == INADDR_NONE )
	{
		hostent * host = gethostbyname( a_Host.Get() );
		if ( host )
		{
			ip = *( ( unsigned int * )host->h_addr );
		}
	}
	else
	{
		ip = inet_addr( a_Host.Get() );
	}
    return ip;
}

// HandleRead
//------------------------------------------------------------------------------
bool TCPConnectionPool::HandleRead( ConnectionInfo * ci )
{
    // work out how many bytes there are
    uint32_t size( 0 );
	uint32_t bytesToRead = 4;
	while ( bytesToRead > 0 )
	{
	    int numBytes = (int)recv( ci->m_Socket, ( (char *)&size ) + 4 - bytesToRead, bytesToRead, 0 );
		if ( numBytes <= 0 )
		{
			if ( WouldBlock() )
			{
                Thread::Sleep( 1 );
				continue;
			}
			TCPDEBUG( "recv error A.  Read: %i (Error: %i) (%x)\n", numBytes, GetLastError(), ci->m_Socket );
			return false;
		}
		bytesToRead -= numBytes;
	}

    TCPDEBUG( "Handle read: %i (%x)\n", size, ci->m_Socket );

    // get output location
    void * buffer = AllocBuffer( size );

    // discard packet?
    ASSERT( buffer );
    
    // read data into the user supplied buffer
    uint32_t bytesRemaining = size;
    char * dest = (char *)buffer;
    while ( bytesRemaining > 0 )
    {
        int numBytes = (int)recv( ci->m_Socket, dest, bytesRemaining, 0 );
		if ( numBytes <= 0 )
		{
			if ( WouldBlock() )
			{
                Thread::Sleep( 1 );
				continue;
			}
			TCPDEBUG( "recv error B.  Read: %i (Error: %i) (%x)\n", numBytes, GetLastError(), ci->m_Socket );
			return false;
		}
        bytesRemaining -= numBytes;
        dest += numBytes;
    }

    // tell user the data is in their buffer
    OnReceive( ci, buffer, size );

	FreeBuffer( buffer );
 
    return true;
}

// GetLastError
//------------------------------------------------------------------------------
int TCPConnectionPool::GetLastError() const
{
    #if defined( __WINDOWS__ )
        return WSAGetLastError();
    #elif defined( __APPLE__ )
        return errno;
    #else
        #error Unknown platform
    #endif
}

// WouldBlock
//------------------------------------------------------------------------------
bool TCPConnectionPool::WouldBlock() const
{
    #if defined( __WINDOWS__ )
        return ( WSAGetLastError() == WSAEWOULDBLOCK );
    #elif defined( __APPLE__ )
        return ( ( errno == EAGAIN ) || ( errno == EWOULDBLOCK ) );
    #else
        #error Unknown platform
    #endif
}

// CloseSocket
//------------------------------------------------------------------------------
int TCPConnectionPool::CloseSocket( TCPSocket a_Socket ) const
{
    #if defined( __WINDOWS__ )
        return closesocket( a_Socket );
    #elif defined( __APPLE__ )
        return close( a_Socket );
    #else
        #error Unknown platform
    #endif
}

// Select
//------------------------------------------------------------------------------
int TCPConnectionPool::Select( TCPSocket UNUSED( a_MaxSocketPlusOne ),
                    		   struct fd_set * a_ReadSocketSet,
							   struct fd_set * a_WriteSocketSet,
                    		   struct fd_set * a_ExceptionSocketSet,
							   struct timeval * a_TimeOut ) const
{
    return select( 0, // ignored by Windows
                    a_ReadSocketSet,
                    a_WriteSocketSet,
                    a_ExceptionSocketSet,
                    a_TimeOut );
}

// Accept
//------------------------------------------------------------------------------
TCPSocket TCPConnectionPool::Accept( TCPSocket a_Socket,
									 struct sockaddr * a_Address,
									 int * a_AddressSize ) const
{
    #if defined( __WINDOWS__ )
        return accept( a_Socket, a_Address, a_AddressSize );
    #elif defined( __APPLE__ )
        return accept( a_Socket, a_Address, (unsigned int *)a_AddressSize );
    #endif
}

// CreateThread
//------------------------------------------------------------------------------
void TCPConnectionPool::CreateListenThread( TCPSocket socket, uint32_t host, uint16_t port )
{
	MutexHolder mh( m_ConnectionsMutex );

	m_ListenConnection = new ConnectionInfo( this );
	m_ListenConnection->m_Socket = socket;
	m_ListenConnection->m_RemoteAddress = host;
	m_ListenConnection->m_RemotePort = port;
    m_ListenConnection->m_ThreadQuitNotification = false;


    // Spawn thread to handle socket
	Thread::ThreadHandle h = Thread::CreateThread( &ListenThreadWrapperFunction,
										 "TCPConnectionPoolListenThread",
										 ( 32 * KILOBYTE ),
										 m_ListenConnection ); // user data argument
    #if defined( __WINDOWS__ )
        ASSERT( h != INVALID_THREAD_HANDLE )
        CloseHandle( h ); // we don't need this anymore
    #elif defined( __APPLE__ )
        (void)h; // TODO:MAC release thread handle
    #else
        #error Unknown platform
    #endif
}

// ThreadWrapperFunction
//------------------------------------------------------------------------------
/*static*/ uint32_t TCPConnectionPool::ListenThreadWrapperFunction( void * data )
{
	ConnectionInfo * ci = (ConnectionInfo *)data;
	ci->m_TCPConnectionPool->ListenThreadFunction( ci );
    return 0;
}

// ListenThreadFunction
//------------------------------------------------------------------------------
void TCPConnectionPool::ListenThreadFunction( ConnectionInfo * ci )
{
	ASSERT( ci->m_Socket != INVALID_SOCKET );

    struct sockaddr_in remoteAddrInfo;
    int remoteAddrInfoSize = sizeof( remoteAddrInfo );

	while ( ci->m_ThreadQuitNotification == false )
    {
        // timout for select() operations
        // (modified by select, so we must recreate it)
        struct timeval timeout;
        timeout.tv_sec = 0;
        timeout.tv_usec = 10000; // 10 ms

        // create a socket set (with just our listen socket in it)
        // (modified by the select() function, so we must recreate it)
        fd_set set;
        FD_ZERO( &set );
		PRAGMA_DISABLE_PUSH_MSVC( 6319 ) // warning C6319: Use of the comma-operator in a tested expression...
        FD_SET( (uint32_t)ci->m_Socket, &set );
		PRAGMA_DISABLE_POP_MSVC // 6319

        // peek
        int num = Select( ci->m_Socket+1, &set, NULL, NULL, &timeout );
        if ( num == 0 )
        {
            // timeout expired - loop again (checking quit notification)
            continue;
        }

		// new connection

		// get a socket for the new connection
		TCPSocket newSocket = Accept( ci->m_Socket, (struct sockaddr *)&remoteAddrInfo, &remoteAddrInfoSize );

		// handle errors or socket shutdown
		if ( newSocket == INVALID_SOCKET )
		{
			TCPDEBUG( "accept failed: %i\n", GetLastError() );
			break;
		}

		#ifdef TCPCONNECTION_DEBUG
			AStackString<32> addr;
			GetAddressAsString( remoteAddrInfo.sin_addr.s_addr, addr );
			TCPDEBUG( "Connection accepted from %s : %i (%x)\n",  addr.Get(), ntohs( remoteAddrInfo.sin_port ), newSocket );
		#endif

		// set non-blocking
		u_long nonBlocking = 1;
        #if defined( __WINDOWS__ )
            ioctlsocket( newSocket, FIONBIO, &nonBlocking );
        #elif defined( __APPLE__ )
            ioctl( newSocket, FIONBIO, &nonBlocking );
        #else
            #error Unknown platform
        #endif
        
		// set send/recv timeout
		uint32_t bufferSize = ( 10 * 1024 * 1024 );
		int ret = setsockopt( newSocket, SOL_SOCKET, SO_RCVBUF, (const char *)&bufferSize, sizeof( bufferSize ) );
		if ( ret != 0 )
		{
			TCPDEBUG( "setsockopt SO_RCVBUF failed: %i\n", GetLastError() );
			break;
		}
		ret = setsockopt( newSocket, SOL_SOCKET, SO_SNDBUF, (const char *)&bufferSize, sizeof( bufferSize ) );
		if ( ret != 0 )
		{
			TCPDEBUG( "setsockopt SO_SNDBUF failed: %i\n", GetLastError() );
			break;
		}

		// keep the new connected socket
		CreateConnectionThread( newSocket, 
								remoteAddrInfo.sin_addr.s_addr,
								ntohs( remoteAddrInfo.sin_port ) );

		continue; // keep listening for more connections
	}

    // close the socket
	CloseSocket( ci->m_Socket );
	ci->m_Socket = INVALID_SOCKET;

	{
		// clear connection (might already be null
		// if simultaneously closed on another thread
		// but we'll hapily set it null redundantly
		MutexHolder mh( m_ConnectionsMutex );
		ASSERT( m_ListenConnection == ci );
		m_ListenConnection = nullptr;
	}

	delete ci;

    // thread exit
    TCPDEBUG( "Listen thread exited\n" );
}

// CreateConnectionThread
//------------------------------------------------------------------------------
ConnectionInfo * TCPConnectionPool::CreateConnectionThread( TCPSocket socket, uint32_t host, uint16_t port )
{
	MutexHolder mh( m_ConnectionsMutex );

	ConnectionInfo * ci = new ConnectionInfo( this );
	ci->m_Socket = socket;
	ci->m_RemoteAddress = host;
	ci->m_RemotePort = port;
	ci->m_ThreadQuitNotification = false;

    #ifdef TCPCONNECTION_DEBUG
        AStackString<32> addr;
        GetAddressAsString( ci->m_RemoteAddress, addr );
        TCPDEBUG( "Connected to %s : %i (%x)\n", addr.Get(), port, socket );
    #endif

	// Spawn thread to handle socket
	Thread::ThreadHandle h = Thread::CreateThread( &ConnectionThreadWrapperFunction,
											"TCPConnectionPoolConnectionThread",
											( 32 * KILOBYTE ),
											ci ); // user data argument
    #if defined( __WINDOWS__ )
        ASSERT( h != INVALID_THREAD_HANDLE )
        CloseHandle( h ); // we don't need this anymore
    #elif defined( __APPLE__ )
    (void)h; // TODO:MAC release thread handle
    #else
        #error Unknown platform
    #endif

	m_Connections.Append( ci );

	return ci;
}

// ConnectionThreadWrapperFunction
//------------------------------------------------------------------------------
/*static*/ uint32_t TCPConnectionPool::ConnectionThreadWrapperFunction( void * data )
{
	ConnectionInfo * ci = (ConnectionInfo *)data;
	ci->m_TCPConnectionPool->ConnectionThreadFunction( ci );
    return 0;
}

// ConnectionThreadFunction
//------------------------------------------------------------------------------
void TCPConnectionPool::ConnectionThreadFunction( ConnectionInfo * ci )
{
	ASSERT( ci );
	ASSERT( ci->m_Socket != INVALID_SOCKET );

    OnConnected( ci ); // Do callback

    // process socket events
	while ( ci->m_ThreadQuitNotification == false )
    {
        // timout for select() operations
        // (modified by select, so we must recreate it)
        struct timeval timeout;
        timeout.tv_sec = 0;
        timeout.tv_usec = 10000; // 10 ms

        // create a socket set (with just our socket in it)
        // (modified by the select() function, so we must recreate it)
        fd_set readSet;
        FD_ZERO( &readSet );
		PRAGMA_DISABLE_PUSH_MSVC( 6319 ) // warning C6319: Use of the comma-operator in a tested expression...
        FD_SET( (uint32_t)ci->m_Socket, &readSet );
		PRAGMA_DISABLE_POP_MSVC // C6319

        int num = Select( ci->m_Socket+1, &readSet, NULL, NULL, &timeout );
        if ( num == 0 )
        {
            // timeout expired - loop again (checking quit notification)
            continue;
        }

		if ( ci->m_ThreadQuitNotification == true )
		{
			break; // don't bother reading any pending data if shutting down
		}

        // Something happened, work out what it is
        if ( FD_ISSET( ci->m_Socket, &readSet ) )
        {
            if ( HandleRead( ci ) == false )
            {
                break;
            }
        }
        else
        {
            ASSERT( false && "Unexpected" );
        }
    }

    OnDisconnected( ci ); // Do callback

    // close the socket
    CloseSocket( ci->m_Socket );
	ci->m_Socket = INVALID_SOCKET;
    //ci->m_Thread = INVALID_THREAD_HANDLE;

	{
		// try to remove from connection list
		// could validly be removed by another
		// thread already due to simultaneously
		// closing a connection while it is dropped
		MutexHolder mh( m_ConnectionsMutex );
		auto iter = m_Connections.Find( ci );
		ASSERT( iter );
		m_Connections.Erase( iter );
	}

	delete ci;

    // thread exit
    TCPDEBUG( "connection thread exited\n" );
}

//------------------------------------------------------------------------------
