// Mem.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Mem.h"
#include "Core/Env/Assert.h"
#include "Core/Env/Types.h"

#include <stdlib.h>

// Defines
//------------------------------------------------------------------------------
#ifdef DEBUG
	#define MEM_FILL_NEW_ALLOCATIONS
	#define MEM_FILL_NEW_ALLOCATION_PATTERN ( 0x7FBAAAAD )
#endif

// FillMem
//------------------------------------------------------------------------------
void FillMem( void * ptr, size_t size, uint32_t pattern )
{
	// this function assumes at least 32bit alignment
	ASSERT( uintptr_t( ptr ) % sizeof( uint32_t ) == 0 );

	// fill whole words
	const size_t numWords = size / sizeof( uint32_t );
	uint32_t * it = reinterpret_cast< uint32_t * >( ptr );
	const uint32_t * end = it + numWords;
	while ( it != end )
	{
		*it = pattern;
		++it;
	}

	// fill remaining bytes
	const size_t remainder =  size - ( numWords * sizeof( uint32_t ) );
	if ( remainder )
	{
		// assuming little-endian format
		char bytes[ 3 ] = { (char)( ( pattern & 0x000000FF ) ),
							(char)( ( pattern & 0x0000FF00 ) >> 8 ),
							(char)( ( pattern & 0x00FF0000 ) >> 16 ) };
		const char * b = bytes;
		char * cit = reinterpret_cast< char * >( it );
		switch( remainder )
		{
			case 3: *cit = *b; ++cit; ++b;
			case 2: *cit = *b; ++cit; ++b;
			case 1: *cit = *b;
		}
	}
}

// Alloc
//------------------------------------------------------------------------------
void * Alloc( size_t size )
{
	return Alloc( size, 4 );
}

// Alloc
//------------------------------------------------------------------------------
void * Alloc( size_t size, size_t alignment )
{
	void * mem;

	#if defined( __APPLE__ )
		(void)alignment; //TODO:MAC Implement aligned allocation
		mem = malloc( size );
	#else
		mem = _aligned_malloc( size, alignment );
	#endif

	#ifdef MEM_FILL_NEW_ALLOCATIONS
		FillMem( mem, size, MEM_FILL_NEW_ALLOCATION_PATTERN );
	#endif

	return mem;
}

// Free
//------------------------------------------------------------------------------
void Free( void * ptr )
{
	#if defined( __APPLE__ )
		free( ptr ); //TODO:MAC Implement aligned deallocation
	#else
		_aligned_free( ptr );
	#endif
}

//------------------------------------------------------------------------------
