// TestCopy.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include <TestFramework/UnitTest.h>
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Graph/CopyNode.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

#include <Core/FileIO/FileIO.h>
#include <Core/Strings/AStackString.h>

// TestCopy
//------------------------------------------------------------------------------
class TestCopy : public UnitTest
{
private:
	DECLARE_TESTS

	// make sure the CopyNode works
	void SingleCopyNode() const;
	void MultipleCopyNodes() const;

	// make sure the FunctionCopy works
	void TestCopyFunction_FileToFile() const;
	void TestCopyFunction_FileToDir() const;
};

// Register Tests
//------------------------------------------------------------------------------
REGISTER_TESTS_BEGIN( TestCopy )
	REGISTER_TEST( SingleCopyNode )
	REGISTER_TEST( MultipleCopyNodes )
	REGISTER_TEST( TestCopyFunction_FileToFile )
	REGISTER_TEST( TestCopyFunction_FileToDir )
REGISTER_TESTS_END

// SingleCopyNode
//------------------------------------------------------------------------------
void TestCopy::SingleCopyNode() const
{
	FBuild fb;
	NodeGraph & ng = fb.GetDependencyGraph();

	const AStackString<> testFileName( "Data/TestGraph/library.cpp" );
	const AStackString<> testFileNameCopy( "../../../../ftmp/Test/Graph/library.copynode.cpp" );

	// check files are in expected states
	TEST_ASSERT( FileIO::FileExists( testFileName.Get() ) == true );
	FileIO::FileDelete( testFileNameCopy.Get() );
	TEST_ASSERT( FileIO::FileExists( testFileNameCopy.Get() ) == false );

	// make a fileNode for the source
	FileNode * srcNode = ng.CreateFileNode( testFileName );

	// and an ObjectNode for the output
	Node * dstNode = ng.CreateCopyNode( testFileNameCopy, srcNode );

	TEST_ASSERT( fb.Build( dstNode ) );

	TEST_ASSERT( FileIO::FileExists( testFileNameCopy.Get() ) );
}

// MultipleCopyNodes
//------------------------------------------------------------------------------
void TestCopy::MultipleCopyNodes() const
{
	FBuild fb;
	NodeGraph & ng = fb.GetDependencyGraph();

	const AStackString<> srcFile( "Data/TestGraph/library.cpp" );
	const AStackString<> dstFileA( "../../../../ftmp/Test/Graph/library.multiplecopynodes1.cpp" );
	const AStackString<> dstFileB( "../../../../ftmp/Test/Graph/library.multiplecopynodes2.cpp" );
	const AStackString<> dstFileC( "../../../../ftmp/Test/Graph/library.multiplecopynodes3.cpp" );

	// check files are as expected before starting test
	FileIO::FileDelete( dstFileA.Get() );
	FileIO::FileDelete( dstFileB.Get() );
	FileIO::FileDelete( dstFileC.Get() );
	TEST_ASSERT( FileIO::FileExists( srcFile.Get() ) == true );
	TEST_ASSERT( FileIO::FileExists( dstFileA.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( dstFileB.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( dstFileC.Get() ) == false );

	// make a fileNode for the source
	FileNode * srcNode = ng.CreateFileNode( srcFile );

	Node * copyA = ng.CreateCopyNode( dstFileA, srcNode );
	Node * copyB = ng.CreateCopyNode( dstFileB, (FileNode *)copyA );
	Node * copyC = ng.CreateCopyNode( dstFileC, (FileNode *)copyB );

	TEST_ASSERT( fb.Build( copyC ) );

	TEST_ASSERT( FileIO::FileExists( dstFileA.Get() ) );
	TEST_ASSERT( FileIO::FileExists( dstFileB.Get() ) );
	TEST_ASSERT( FileIO::FileExists( dstFileC.Get() ) );
}

// TestCopyFunction_FileToFile
//------------------------------------------------------------------------------
void TestCopy::TestCopyFunction_FileToFile() const
{
	FBuildOptions options;
	options.m_ConfigFile = "Data/TestCopy/copy.bff";
	FBuild fBuild( options );
	TEST_ASSERT( fBuild.Initialize() );

	const AStackString<> dst( "..\\..\\..\\..\\ftmp\\Test\\Copy\\copy.bff.copy" );

	// clean up anything left over from previous runs
	FileIO::FileDelete( dst.Get() );
	TEST_ASSERT( FileIO::FileExists( dst.Get() ) == false );

	// build (via alias)
	TEST_ASSERT( fBuild.Build( AStackString<>( "TestCopyFileToFile" ) ) );

	// make sure all output is where it is expected
	TEST_ASSERT( FileIO::FileExists( dst.Get() ) );
}


// TestCopyFunction_FileToDir
//------------------------------------------------------------------------------
void TestCopy::TestCopyFunction_FileToDir() const
{
	FBuildOptions options;
	options.m_ConfigFile = "Data/TestCopy/copy.bff";
	FBuild fBuild( options );
	TEST_ASSERT( fBuild.Initialize() );

	const AStackString<> dst( "..\\..\\..\\..\\ftmp\\Test\\Copy\\copy.bff" );

	// clean up anything left over from previous runs
	FileIO::FileDelete( dst.Get() );
	TEST_ASSERT( FileIO::FileExists( dst.Get() ) == false );

	// build (via alias)
	TEST_ASSERT( fBuild.Build( AStackString<>( "TestCopyFileToDir" ) ) );

	// make sure all output is where it is expected
	TEST_ASSERT( FileIO::FileExists( dst.Get() ) );
}


//------------------------------------------------------------------------------
