// TestCompressor.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include <TestFramework/UnitTest.h>

#include <Tools/FBuild/FBuildCore/Helpers/Compressor.h>

#include <Core/Containers/AutoPtr.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Strings/AString.h>
#include <Core/Time/Timer.h>
#include <Core/Tracing/Tracing.h>

#include <memory.h>

// TestCompressor
//------------------------------------------------------------------------------
class TestCompressor : public UnitTest
{
private:
	DECLARE_TESTS //_RUN_ONLY_THIS_TEST

	void CompressSimple() const;
	void CompressPreprocessedFile() const;
	void CompressObjFile() const;

	void CompressSimpleHelper( const char * data, 
							   size_t size, 
							   size_t expectedCompressedSize,
							   bool shouldCompress ) const;
	void CompressHelper( const char * fileName ) const;
};

// Register Tests
//------------------------------------------------------------------------------
REGISTER_TESTS_BEGIN( TestCompressor )
	REGISTER_TEST( CompressSimple );
	REGISTER_TEST( CompressPreprocessedFile )
	REGISTER_TEST( CompressObjFile )
REGISTER_TESTS_END

// CompressSimple
//------------------------------------------------------------------------------
void TestCompressor::CompressSimple() const
{
	// 8 characters in a row, should be 3 misses (27bits) followed by 5 hits (5bits)
	// = 32bits (4 bytes), plus 4 bytes compression header = 8 bytes,
	CompressSimpleHelper( "AAAAAAAA",
						  8,
						  8,
						  false );

	// 16 characters in a row, should be 3 misses (27bits) followed by 13 hits (13bits)
	// = 40bits (5 bytes, 8 rounded up), plus 4 bytes compression header = 12 bytes,
	CompressSimpleHelper( "AAAAAAAAAAAAAAAA",
						  16,
						  12,
						  true );

	// a more representative piece of data
	const char * testData = "#include \"a.cpp\"\r\n#include \"b.cpp\"\r\n"; 
	CompressSimpleHelper( testData, AString::StrLen( testData ), 0, true );

	// check for internal worst case checks
	CompressSimpleHelper( "A", 1, 0, false );
}

// CompressSimpleHelper
//------------------------------------------------------------------------------
void TestCompressor::CompressSimpleHelper( const char * data, 
										   size_t size, 
										   size_t expectedCompressedSize,
										   bool shouldCompress ) const
{
	// compress
	Compressor c;
	const bool compressed = c.Compress( data, size );
	TEST_ASSERT( compressed == shouldCompress );
	const size_t compressedSize = c.GetResultSize();
	if ( expectedCompressedSize > 0 )
	{
		TEST_ASSERT( compressedSize == expectedCompressedSize );
	}
	void const * compressedMem = c.GetResult();

	// decompress
	Compressor d;
	d.Decompress( compressedMem );
	const size_t decompressedSize = d.GetResultSize();
	TEST_ASSERT( decompressedSize == size );
	TEST_ASSERT( memcmp( data, d.GetResult(), size ) == 0 );
}

// CompressPreprocessedFile
//------------------------------------------------------------------------------
void TestCompressor::CompressPreprocessedFile() const
{
	CompressHelper( "Data/TestCompressor/TestPreprocessedFile.cpp" );
}

//------------------------------------------------------------------------------
void TestCompressor::CompressObjFile() const
{
	CompressHelper( "Data/TestCompressor/TestObjFile.o" );
}

// CompressHelper
//------------------------------------------------------------------------------
void TestCompressor::CompressHelper( const char * fileName ) const
{
	// read some test data into a file
	AutoPtr< void > data;
	size_t dataSize;
	{
		FileStream fs;
		TEST_ASSERT( fs.Open( fileName ) );
		dataSize = (size_t)fs.GetFileSize();
		data = new char[ dataSize ] ;
		TEST_ASSERT( (uint32_t)fs.Read( data.Get(), dataSize ) == dataSize );
	}

	OUTPUT( "File           : %s\n", fileName );
	OUTPUT( "Size           : %u\n", (uint32_t)dataSize );

	// compress the data to obtain size
	Compressor comp;
	comp.Compress( data.Get(), dataSize );
	size_t compressedSize = comp.GetResultSize();
	AutoPtr< char > compressedData( new char[ compressedSize ] );
	memcpy( compressedData.Get(), comp.GetResult(), compressedSize );
	float compressedPerc = ( (float)compressedSize / (float)dataSize ) * 100.0f;
	OUTPUT( "Compressed Size: %u (%2.1f%% of original)\n", (uint32_t)compressedSize, compressedPerc );

	// decompress to check we get original data back
	Compressor decomp;
	decomp.Decompress( compressedData.Get() );
	size_t uncompressedSize = decomp.GetResultSize();
	TEST_ASSERT( uncompressedSize == dataSize );
	for ( size_t i=0; i<uncompressedSize; ++i )
	{
		TEST_ASSERT( ( (char *)data.Get() )[ i ] == ( (char *)decomp.GetResult() )[ i ] );
	}

	// speed checks
	//--------------
	const size_t NUM_REPEATS( 100 );

	// compress the data several times to get more stable throughput value
	Timer t;
	for ( size_t i=0; i<NUM_REPEATS; ++i )
	{
		Compressor c;
		c.Compress( data.Get(), dataSize );
		compressedSize = c.GetResultSize();
	}
	float compressTimeTaken = t.GetElapsed();
	double compressThroughputMBs = ( ( (double)dataSize / 1024.0 * (double)NUM_REPEATS ) / compressTimeTaken ) / 1024.0;
	OUTPUT( "     Comp Speed: %2.1f MB/s - %2.3fs (%u repeats)\n", (float)compressThroughputMBs, compressTimeTaken, NUM_REPEATS );
	
	// decompress the data
	Timer t2;
	for ( size_t i=0; i<NUM_REPEATS; ++i )
	{
		Compressor d;
		d.Decompress( compressedData.Get() );
		TEST_ASSERT( d.GetResultSize() == dataSize );
	}
	float decompressTimeTaken = t2.GetElapsed();
	double decompressThroughputMBs = ( ( (double)dataSize / 1024.0 * (double)NUM_REPEATS ) / decompressTimeTaken ) / 1024.0;
	OUTPUT( "   Decomp Speed: %2.1f MB/s - %2.3fs (%u repeats)\n", (float)decompressThroughputMBs, decompressTimeTaken, NUM_REPEATS );

	// time memcpy to compare with
	Timer t0;
	for ( size_t i=0; i<NUM_REPEATS; ++i )
	{
		char * mem = new char[ dataSize ];
		memcpy( mem, data.Get(), dataSize );
		delete [] mem;
	}
	float memcpyTimeTaken = t0.GetElapsed();
	double memcpyThroughputMBs = ( ( (double)dataSize / 1024.0 * (double)NUM_REPEATS ) / memcpyTimeTaken ) / 1024.0;
	OUTPUT( "   MemCpy Speed: %2.1f MB/s - %2.3fs (%u repeats)\n", (float)memcpyThroughputMBs, memcpyTimeTaken, NUM_REPEATS );
}

//------------------------------------------------------------------------------
