// TestBuildFBuild.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include <TestFramework/UnitTest.h>

#include <Tools/FBuild/FBuildCore/FBuild.h>

#include <Core/FileIO/FileIO.h>
#include <Core/Strings/AStackString.h>

// Globals
//------------------------------------------------------------------------------
const char * dbFile = "../ftmp/Test/TestFBuild.db";

// TestBuildFBuild
//------------------------------------------------------------------------------
class TestBuildFBuild : public UnitTest
{
private:
	DECLARE_TESTS

	void BuildClean( FBuildOptions options = FBuildOptions() ) const;
	void Build( FBuildOptions options = FBuildOptions() ) const;
	void BuildCleanWithCache() const;
	void BuildWithCache() const;

	void GetCodeDir( AString & codeDir ) const;
};

// Register Tests
//------------------------------------------------------------------------------
REGISTER_TESTS_BEGIN( TestBuildFBuild )
	REGISTER_TEST( BuildClean )
	REGISTER_TEST( Build )
	REGISTER_TEST( BuildCleanWithCache )
	REGISTER_TEST( BuildWithCache )
REGISTER_TESTS_END

// BuildClean
//------------------------------------------------------------------------------
void TestBuildFBuild::BuildClean( FBuildOptions options ) const
{
	AStackString<> codeDir;
	GetCodeDir( codeDir );

	options.m_ConfigFile = "Tools/FBuild/FBuildTest/Data/TestBuildFBuild/fbuild.bff";
	//options.m_ShowProgress = true;
	//options.m_ShowInfo = true;
	options.m_ShowSummary = true;
	options.SetWorkingDir( codeDir );

	FBuild fBuild( options );
	TEST_ASSERT( fBuild.Initialize() );

	// delete files from previous runs
	Array< AString > files( 1024, true );
	FileIO::GetFiles( AStackString<>( "../ftmp/" ), AStackString<>( "*" ), true, &files );
	for ( Array< AString >::Iter it = files.Begin();
		  it != files.End();
		  it++ )
	{
		FileIO::FileDelete( (*it).Get() );
	}

	const AStackString<> lib( "../ftmp/Win32/Debug/Core/core.lib" );
	const AStackString<> lib2( "../ftmp/Win32/Debug/Tools/FBuild/FBuildCore/fbuildcore.lib" );
	const AStackString<> lib3( "../ftmp/Win32/Release/Tools/FBuild/FBuildApp/fbuildapp.lib" );
	const AStackString<> exe( "../ftmp/Win32/Release/Tools/FBuild/FBuildApp/fbuild.exe" );
	const AStackString<> target( "fbuild" );

	// clean up anything left over from previous runs
	FileIO::FileDelete( lib.Get() );
	FileIO::FileDelete( lib2.Get() );
	FileIO::FileDelete( lib3.Get() );
	FileIO::FileDelete( exe.Get() );
	FileIO::FileDelete( dbFile );
	TEST_ASSERT( FileIO::FileExists( lib.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( lib2.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( lib3.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( exe.Get() ) == false );
	TEST_ASSERT( FileIO::FileExists( dbFile ) == false );

	TEST_ASSERT( fBuild.Build( target ) );

	// make sure a lot of nodes were actually built
	// (if we're using the cache, then we need to check built+hits)
	if ( options.m_UseCacheWrite )
	{
		TEST_ASSERT( ( fBuild.GetStats().GetNodesBuilt() +
					   fBuild.GetStats().GetCacheHits() ) > 100 );
	}
	else
	{
		TEST_ASSERT( fBuild.GetStats().GetNodesBuilt() > 100 );
	}

	// make sure all output files are as expected
	TEST_ASSERT( FileIO::FileExists( lib.Get() ) );
	TEST_ASSERT( FileIO::FileExists( lib2.Get() ) );
	TEST_ASSERT( FileIO::FileExists( lib3.Get() ) );
	TEST_ASSERT( FileIO::FileExists( exe.Get() ) );

	// save the db file - make sure it exists
	TEST_ASSERT( fBuild.SaveDependencyGraph( dbFile ) );
	TEST_ASSERT( FileIO::FileExists( dbFile ) );
}

// Build
//------------------------------------------------------------------------------
void TestBuildFBuild::Build( FBuildOptions options ) const
{
	AStackString<> codeDir;
	GetCodeDir( codeDir );

	options.m_ConfigFile = "Tools/FBuild/FBuildTest/Data/TestBuildFBuild/fbuild.bff";
	//options.m_ShowProgress = true;
	//options.m_ShowInfo = true;
	options.m_ShowSummary = true;
	options.SetWorkingDir( codeDir );

	FBuild fBuild( options );

	// load dep graph from previous build
	TEST_ASSERT( fBuild.Initialize( dbFile ) );

	const AStackString<> target( "fbuild" );

	// nothing should rebuild
	TEST_ASSERT( fBuild.Build( target ) );

	// check number of things rebuilt is very small
	// (some things always rebuild (like directory nodes) so we can't check 0)
	TEST_ASSERT( fBuild.GetStats().GetNodesBuilt() <= 20 );
	TEST_ASSERT( fBuild.GetStats().GetNodesBuilt() < fBuild.GetStats().GetNodesProcessed() );
	TEST_ASSERT( fBuild.GetStats().GetCacheHits() == 0 );
	TEST_ASSERT( fBuild.GetStats().GetCacheMisses() == 0 );
	TEST_ASSERT( fBuild.GetStats().GetCacheStores() == 0 );
}

// BuildCleanWithCache
//------------------------------------------------------------------------------
void TestBuildFBuild::BuildCleanWithCache() const
{
	FBuildOptions options;
	options.m_UseCacheRead = true;
	options.m_UseCacheWrite = true;
	BuildClean( options );
}

// BuildWithCache
//------------------------------------------------------------------------------
void TestBuildFBuild::BuildWithCache() const
{
	FBuildOptions options;
	options.m_UseCacheRead = true;
	options.m_UseCacheWrite = true;
	Build( options );
}

// GetCodeDir
//------------------------------------------------------------------------------
void TestBuildFBuild::GetCodeDir( AString & codeDir ) const
{
	// we want the working dir to be the 'Code' directory
	TEST_ASSERT( FileIO::GetCurrentDir( codeDir ) );
	const char * codePos = codeDir.Find( "\\Code\\" );
	TEST_ASSERT( codePos );
	codeDir.SetLength( (uint16_t)( codePos - codeDir.Get() + 6 ) );
}

//------------------------------------------------------------------------------
