// TestBFFParsing.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include <TestFramework/UnitTest.h>

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFParser.h>

#include <Core/Containers/AutoPtr.h>
#include <Core/FileIO/FileStream.h>

// TestBFFParsing
//------------------------------------------------------------------------------
class TestBFFParsing : public UnitTest
{
private:
	DECLARE_TESTS

	void Empty() const;
	void AlmostEmpty() const;
	void Comments() const;
	void Strings() const;
	void Arrays() const;
	void Integers() const;

	void Parse( const char * fileName ) const;
};

// Register Tests
//------------------------------------------------------------------------------
REGISTER_TESTS_BEGIN( TestBFFParsing )
	REGISTER_TEST( Empty )
	REGISTER_TEST( AlmostEmpty )
	REGISTER_TEST( Comments )
	REGISTER_TEST( Strings )
	REGISTER_TEST( Arrays )
	REGISTER_TEST( Integers )
REGISTER_TESTS_END

// Empty
//------------------------------------------------------------------------------
void TestBFFParsing::Empty() const
{
	// an empty file should pass without problem
	char buffer[ 1 ] = { '\000' }; // post data sentinel
	BFFParser p;
	TEST_ASSERT( p.Parse( buffer, 0, "empty.bff" ) );
}

// AlmostEmpty
//------------------------------------------------------------------------------
void TestBFFParsing::AlmostEmpty() const
{
	// an empty file should pass without problem
	char * buffer = "\r\n\000"; // empty line + post data sentinel
	BFFParser p;
	TEST_ASSERT( p.Parse( buffer, 2, "empty.bff" ) );
}

// Comments
//------------------------------------------------------------------------------
void TestBFFParsing::Comments() const
{
	Parse( "Data/TestBFFParsing/comments.bff" );
}

// Strings
//------------------------------------------------------------------------------
void TestBFFParsing::Strings() const
{
	Parse( "Data/TestBFFParsing/strings.bff" );
}

// Arrays
//------------------------------------------------------------------------------
void TestBFFParsing::Arrays() const
{
	Parse( "Data/TestBFFParsing/arrays.bff" );
}

// Integers
//------------------------------------------------------------------------------
void TestBFFParsing::Integers() const
{
	Parse( "Data/TestBFFParsing/integers.bff" );
}

// Parse
//------------------------------------------------------------------------------
void TestBFFParsing::Parse( const char * fileName ) const
{
	FileStream f;
	TEST_ASSERT( f.Open( fileName, FileStream::READ_ONLY ) );
	uint32_t fileSize = (uint32_t)f.GetFileSize();
	AutoPtr< char > mem( new char[ fileSize + 1 ] );
	mem.Get()[ fileSize ] = '\000'; // parser requires sentinel
	TEST_ASSERT( f.Read( mem.Get(), fileSize ) == fileSize );

	FBuild fBuild;
	BFFParser p;
	TEST_ASSERT( p.Parse( mem.Get(), fileSize, fileName ) );
}

//------------------------------------------------------------------------------
