// WorkerThread
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "WorkerThread.h"
#include "Job.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/Node.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/JobQueueRemote.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Process/Thread.h>
#include <Core/Time/Timer.h>

#include <Windows.h>

// Static
//------------------------------------------------------------------------------
THREAD_LOCAL static uint32_t s_WorkerThreadThreadIndex = 0;
AString WorkerThread::s_TmpRoot;

//------------------------------------------------------------------------------
#pragma warning( push )
#pragma warning( disable : 28021 ) // The parameter '_Param_(4)' being annotated with 'aliases memory' must be a pointer.
WorkerThread::WorkerThread( uint32_t threadIndex )
: m_ShouldExit( false )
, m_Exited( false )
, m_ThreadIndex( threadIndex )
{
	// first worker thread obtains temp path
	if ( s_TmpRoot.IsEmpty() )
	{
		VERIFY( FileIO::GetTempDir( s_TmpRoot ) );
		s_TmpRoot += ".fbuild.tmp\\";
		VERIFY( FileIO::EnsurePathExists( s_TmpRoot ) );
	}

	// Start thread
	Thread::ThreadHandle h = Thread::CreateThread( ThreadWrapperFunc,
												   "WorkerThread",
												   64 * KILOBYTE,
												   this );
	ASSERT( h != nullptr );
	::CloseHandle( h ); // we don't want to keep this, so free it now
}
#pragma warning( pop )

//------------------------------------------------------------------------------
WorkerThread::~WorkerThread()
{
}

// InitTmpDir
//------------------------------------------------------------------------------
/*static*/ void WorkerThread::InitTmpDir()
{
	VERIFY( FileIO::GetTempDir( s_TmpRoot ) );
	s_TmpRoot += ".fbuild.tmp\\";
	VERIFY( FileIO::EnsurePathExists( s_TmpRoot ) );
}

// WaitForStop
//------------------------------------------------------------------------------
void WorkerThread::WaitForStop()
{
	while ( m_Exited == false )
	{
		Sleep( 1 );
	}
}

// GetThreadIndex
//------------------------------------------------------------------------------
/*static*/ uint32_t WorkerThread::GetThreadIndex()
{
	return s_WorkerThreadThreadIndex;
}

// MainWrapper
//------------------------------------------------------------------------------
/*static*/ uint32_t WorkerThread::ThreadWrapperFunc( void * param )
{
	WorkerThread * wt = reinterpret_cast< WorkerThread * >( param );
	s_WorkerThreadThreadIndex = wt->m_ThreadIndex;
	wt->Main();
	return 0;
}

// Main
//------------------------------------------------------------------------------
/*virtual*/ void WorkerThread::Main()
{
	while ( m_ShouldExit == false )
	{
		// try to find some work to do
		Job * job = JobQueue::IsValid() ? JobQueue::Get().GetJobToProcess() : nullptr;
		if ( job != nullptr )
		{
			// make sure state is as expected
			ASSERT( job->GetNode()->GetState() == Node::BUILDING );

			// process the work
			Node::BuildResult result = JobQueue::DoBuild( job );
			if ( result == Node::NODE_RESULT_NEED_SECOND_BUILD_PASS )
			{
				JobQueue::Get().QueueJob2( job );
			}
			else
			{
				JobQueue::Get().FinishedProcessingJob( job, ( result != Node::NODE_RESULT_FAILED ), false );
			}

			// loop again to get another job
			continue;
		}

		// no local job, see if we can do one from the remote queue
		job = JobQueue::IsValid() ? JobQueue::Get().GetDistributableJobToProcess() : nullptr;
		if ( job != nullptr )
		{
			// process the work
			Node::BuildResult result = JobQueueRemote::DoBuild( job );
			JobQueue::Get().FinishedProcessingJob( job, ( result != Node::NODE_RESULT_FAILED ), true ); // returning a remote job

			// loop again to get another job
			continue;
		}

		// no work to do right now
		Sleep( 1 ); // wait and try again later
	}

	m_Exited = true;
}

// CreateTempFile
//------------------------------------------------------------------------------
/*static*/ void WorkerThread::CreateTempFilePath( const char * extension,
												  AString & tmpFileName )
{
	ASSERT( extension );
	ASSERT( !s_TmpRoot.IsEmpty() );

	// get the index for the worker thread
	// (for the main thread, this will be 0 which is OK)
	const uint32_t threadIndex = WorkerThread::GetThreadIndex();

	tmpFileName.Format( "%score_%u.%s", s_TmpRoot.Get(), threadIndex, extension );
}

// CreateTempFile
//------------------------------------------------------------------------------
/*static*/ bool WorkerThread::CreateTempFile( const AString & tmpFileName,
										FileStream & file )
{
	ASSERT( tmpFileName.IsEmpty() == false );
	ASSERT( tmpFileName[ 1 ] == ':' );
	return file.Open( tmpFileName.Get(), FileStream::WRITE_ONLY );
}

//------------------------------------------------------------------------------
