// JobQueue
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "JobQueue.h"
#include "Job.h"
#include "WorkerThread.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/Node.h>

#include <Core/Time/Timer.h>
#include <Core/FileIO/FileIO.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
JobQueue::JobQueue( uint32_t numWorkerThreads ) :
	m_PendingJobs( 1024, true ),
	m_PendingJobs2( 1024, true ),
	m_PendingJobs2MemoryUsage( 0 ),
	m_NumJobsActive( 0 ),
	m_NumJobsDistActive( 0 ),
	m_CompletedJobs( 1024, true ),
	m_CompletedJobsFailed( 1024, true ),
	m_Workers( numWorkerThreads, false )
{
	WorkerThread::InitTmpDir();

	for ( uint32_t i=0; i<numWorkerThreads; ++i )
	{
		// identify each worker with an id starting from 1
		// (the "main" thread is considered 0)
		uint32_t threadIndex = ( i + 1 );
		m_Workers.Append( new WorkerThread( threadIndex ) );
	}
}

// DESTRUCTOR
//------------------------------------------------------------------------------
JobQueue::~JobQueue()
{
	// signal all workers to stop - ok if this has already been done
	SignalStopWorkers();

	// wait for workers to finish - ok if they stopped before this
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		m_Workers[ i ]->WaitForStop();
		delete m_Workers[ i ];
	}
}

// SignalStopWorkers (Main Thread)
//------------------------------------------------------------------------------
void JobQueue::SignalStopWorkers()
{
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		m_Workers[ i ]->Stop();
	}
}

// HaveWorkersStopped
//------------------------------------------------------------------------------
bool JobQueue::HaveWorkersStopped() const
{
	const size_t numWorkerThreads = m_Workers.GetSize();
	for ( size_t i=0; i<numWorkerThreads; ++i )
	{
		if ( m_Workers[ i ]->HasExited() == false )
		{
			return false;
		}
	}
	return true;
}

// GetNumDistributableJobsAvailable
//------------------------------------------------------------------------------
size_t JobQueue::GetNumDistributableJobsAvailable() const
{
	MutexHolder m( m_PendingJobsMutex );
	return m_PendingJobs2.GetSize();
}

// GetJobStats
//------------------------------------------------------------------------------
void JobQueue::GetJobStats( uint32_t & numJobs, 
						    uint32_t & numJobsActive, 
							uint32_t & numJobsDist, 
							uint32_t & numJobsDistActive ) const
{
	MutexHolder m( m_PendingJobsMutex );
	numJobs = (uint32_t)m_PendingJobs.GetSize();
	numJobsActive = m_NumJobsActive;
	numJobsDist = (uint32_t)m_PendingJobs2.GetSize();
	numJobsDistActive = m_NumJobsDistActive;
}

// QueueJob (Main Thread)
//------------------------------------------------------------------------------
void JobQueue::QueueJob( Node * node )
{
	ASSERT( node->GetState() == Node::DYNAMIC_DEPS_DONE );

	// trivial build tasks are processed immediately and returned
	// likewise if we have no workers
	if ( ( node->GetControlFlags() & Node::FLAG_TRIVIAL_BUILD ) ||
		 ( m_Workers.GetSize() == 0 ) )
	{
		Job job( node );

		Node::BuildResult result = DoBuild( &job );

		if ( ( result == Node::NODE_RESULT_OK ) || ( result == Node::NODE_RESULT_OK_CACHE ) )
		{
			node->SetState( Node::UP_TO_DATE );
		}
		else if ( result == Node::NODE_RESULT_FAILED )
		{
			node->SetState( Node::FAILED );			
		}
		else
		{
			ASSERT( false ); // unexpected state
		}
		return;
	}

	// mark as building
	node->SetState( Node::BUILDING );

	Job * job = new Job( node );

	// stick in queue
	MutexHolder m( m_PendingJobsMutex );
	m_PendingJobs.Append( job );
}

// QueueJob2
//------------------------------------------------------------------------------
void JobQueue::QueueJob2( Job * job )
{
	ASSERT( job->GetNode()->GetState() == Node::BUILDING );

	MutexHolder m( m_PendingJobsMutex );
	m_PendingJobs2.Append( job );

	// track size of distributable jobs
	m_PendingJobs2MemoryUsage += job->GetDataSize();

	--m_NumJobsActive; // job converts from active to pending remote
}

// GetDistributableJobToProcess
//------------------------------------------------------------------------------
Job * JobQueue::GetDistributableJobToProcess()
{
	MutexHolder m( m_PendingJobsMutex );
	if ( m_PendingJobs2.IsEmpty() )
	{
		return nullptr;
	}

	// building jobs in the order they are queued
	Job * job = m_PendingJobs2[ 0 ];
	m_PendingJobs2.PopFront();

	// track size of distributable jobs
	m_PendingJobs2MemoryUsage -= job->GetDataSize();
	++m_NumJobsDistActive;

	return job;
}


// ReturnUnfinishedDistributableJob
//------------------------------------------------------------------------------
void JobQueue::ReturnUnfinishedDistributableJob( Job * job )
{
	MutexHolder m( m_PendingJobsMutex );
	m_PendingJobs2.Append( job );
	--m_NumJobsDistActive;

	// track size of distributable jobs
	m_PendingJobs2MemoryUsage += job->GetDataSize();
}

// FinalizeCompletedJobs (Main Thread)
//------------------------------------------------------------------------------
void JobQueue::FinalizeCompletedJobs()
{
	MutexHolder m( m_CompletedJobsMutex );

	// completed jobs
	for ( auto i = m_CompletedJobs.Begin();
			i != m_CompletedJobs.End();
			i++ )
	{
		Job * job = ( *i );
		job->GetNode()->SetState( Node::UP_TO_DATE );
		delete job;
	}
	m_CompletedJobs.Clear();

	// failed jobs
	for ( auto i = m_CompletedJobsFailed.Begin();
			i != m_CompletedJobsFailed.End();
			i++ )
	{
		Job * job = ( *i );
		job->GetNode()->SetState( Node::FAILED );
		delete job;
	}
	m_CompletedJobsFailed.Clear();
}

// GetJobToProcess (Worker Thread)
//------------------------------------------------------------------------------
Job * JobQueue::GetJobToProcess()
{
	MutexHolder m( m_PendingJobsMutex );
	if ( m_PendingJobs.IsEmpty() )
	{
		return nullptr;
	}

	// building jobs in the order they are queued
	Job * job = m_PendingJobs[ 0 ];
	m_PendingJobs.PopFront();
	++m_NumJobsActive;
	return job;
}

// FinishedProcessingJob (Worker Thread)
//------------------------------------------------------------------------------
void JobQueue::FinishedProcessingJob( Job * job, bool success, bool wasRemote )
{
	ASSERT( job->GetNode()->GetState() == Node::BUILDING );

	MutexHolder m( m_CompletedJobsMutex );
	if ( success )
	{
		m_CompletedJobs.Append( job );
	}
	else
	{
		m_CompletedJobsFailed.Append( job );
	}
	if ( wasRemote )
	{
		--m_NumJobsDistActive;
	}
	else
	{
		--m_NumJobsActive;
	}
}

// DoBuild
//------------------------------------------------------------------------------
/*static*/ Node::BuildResult JobQueue::DoBuild( Job * job )
{
	Timer timer; // track how long the item takes

	Node * node = job->GetNode();

	// make sure the output path exists for files
	if ( node->IsAFile() )
	{
		if ( Node::EnsurePathExistsForFile( node->GetName() ) == false )
		{
			// error already output by EnsurePathExistsForFile
			return Node::NODE_RESULT_FAILED;
		}
	}

	Node::BuildResult result = node->DoBuild( job );

	uint32_t timeTakenMS = uint32_t( timer.GetElapsed() * 1000.0f );

	if ( result == Node::NODE_RESULT_OK )
	{
		// record new build time only if built (i.e. if cached or failed, the time
		// does not represent how long it takes to create this resource)
		node->SetLastBuildTime( timeTakenMS );
		node->SetStatFlag( Node::STATS_BUILT );
		FLOG_INFO( "-Build: %u ms\t%s", timeTakenMS, node->GetName().Get() );
	}

	if ( result == Node::NODE_RESULT_NEED_SECOND_BUILD_PASS )
	{
		// nothing to check
	}
	else if ( node->IsAFile() )
	{
		if ( result == Node::NODE_RESULT_FAILED )
		{
			if ( node->GetControlFlags() & Node::FLAG_NO_DELETE_ON_FAIL )
			{
				// node failed, but builder wants result left on disc
			}
			else
			{
				// build of file failed - if there is a file....
				if ( FileIO::FileExists( node->GetName().Get() ) )
				{
					// ... it is invalid, so try to delete it
					if ( FileIO::FileDelete( node->GetName().Get() ) == false )
					{
						// failed to delete it - this might cause future build problems!
						FLOG_ERROR( "Post failure deletion failed for '%s'", node->GetName().Get() );
					}
				}
			}
		}
		else
		{
			// build completed ok, or retrieved from cache...
			ASSERT( ( result == Node::NODE_RESULT_OK ) || ( result == Node::NODE_RESULT_OK_CACHE ) );

			// ... ensure file exists (to detect builder logic problems)
			if ( !FileIO::FileExists( node->GetName().Get() ) )
			{
				FLOG_ERROR( "File missing despite success for '%s'", node->GetName().Get() );
				result = Node::NODE_RESULT_FAILED;
			}
		}
	}

	// log processing time
	node->AddProcessingTime( timeTakenMS );

	return result;
}

//------------------------------------------------------------------------------
