// Server.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Server.h"
#include "Protocol.h"

#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/Job.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/JobQueueRemote.h>

#include <Core/FileIO/ConstMemoryStream.h>
#include <Core/FileIO/MemoryStream.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
Server::Server()
	: m_ShouldExit( false )
	, m_Exited( false )
	, m_ClientList( 32, true )
{
	m_Thread = Thread::CreateThread( ThreadFuncStatic,
									 "Server",
									 ( 64 * KILOBYTE ),
									 this );
	ASSERT( m_Thread );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Server::~Server()
{
	m_ShouldExit = true;
	while ( m_Exited == false )
	{
		Sleep( 1 );
	}

	ShutdownAllConnections();

	CloseHandle( m_Thread );
}

// OnConnected
//------------------------------------------------------------------------------
/*virtual*/ void Server::OnConnected( const ConnectionInfo * connection )
{
//#pragma warning( push )
//#pragma warning( disable : 28197 ) // Possibly leaking memory 'ws'.
	ClientState * cs = new ClientState( connection );
//#pragma warning( pop )
	connection->SetUserData( cs );
	
	MutexHolder mh( m_ClientListMutex );
	m_ClientList.Append( cs );
}

//------------------------------------------------------------------------------
/*virtual*/ void Server::OnDisconnected( const ConnectionInfo * connection )
{
	ASSERT( connection );
	ClientState * cs = (ClientState *)connection->GetUserData();
	ASSERT( cs );

	// Unhook any jobs which are queued or in progress for this client
	// - deletes the queued jobs
	// - unhooks the UserData for in-progress jobs so the result is discarded on completion
	JobQueueRemote & jqr = JobQueueRemote::Get();
	jqr.CancelJobsWithUserData( cs );

	// free the serverstate structure
	MutexHolder mh( m_ClientListMutex );
	auto iter = m_ClientList.Find( cs );
	ASSERT( iter );
	m_ClientList.Erase( iter );
	delete cs;
}

// OnReceive
//------------------------------------------------------------------------------
/*virtual*/ void Server::OnReceive( const ConnectionInfo * connection, void * data, uint32_t size )
{
	ClientState * cs = (ClientState *)connection->GetUserData();
	ASSERT( cs );

	// are we expecting a msg, or the payload for a msg?
	void * payload = nullptr;
	size_t payloadSize = 0;
	if ( cs->m_CurrentMessage == nullptr )
	{
		// message
		cs->m_CurrentMessage = reinterpret_cast< const Protocol::IMessage * >( data );
		if ( cs->m_CurrentMessage->HasPayload() )
		{
			return;
		}
	}
	else
	{
		// payload
		ASSERT( cs->m_CurrentMessage->HasPayload() );
		payload = data;
		payloadSize = size;
	}


	// determine message type
	const Protocol::IMessage * imsg = cs->m_CurrentMessage;
	Protocol::MessageType messageType = imsg->GetType();

	switch ( messageType )
	{
		case Protocol::MSG_CONNECTION:
		{
			const Protocol::MsgConnection * msg = static_cast< const Protocol::MsgConnection * >( imsg );
			Process( connection, msg ); 
			break;
		}
		case Protocol::MSG_STATUS:
		{
			const Protocol::MsgStatus * msg = static_cast< const Protocol::MsgStatus * >( imsg );
			Process( connection, msg );
			break;
		}
		case Protocol::MSG_NO_JOB_AVAILABLE:
		{
			const Protocol::MsgNoJobAvailable * msg = static_cast< const Protocol::MsgNoJobAvailable * >( imsg );
			Process( connection, msg );
			break;
		}
		case Protocol::MSG_JOB:
		{
			const Protocol::MsgJob * msg = static_cast< const Protocol::MsgJob * >( imsg );
			Process( connection, msg, payload, payloadSize );
			break;
		}
		default:
		{
			// unknown message type
			ASSERT( false ); // this indicates a protocol bug
			Disconnect( connection );
			break;
		}
	}

	// free everything
	delete [] (void *)( cs->m_CurrentMessage );
	delete [] payload;
	cs->m_CurrentMessage = nullptr;
}

// FreeBuffer
//------------------------------------------------------------------------------
/*virtual*/ void Server::FreeBuffer( void * data )
{
	(void)data; // nothing to do - memory is managed in OnReceive
}

// Process( MsgConnection )
//------------------------------------------------------------------------------
void Server::Process( const ConnectionInfo * connection, const Protocol::MsgConnection * msg )
{
	// check for valid/supported protocol version
	if ( msg->GetProtocolVersion() != Protocol::PROTOCOL_VERSION )
	{
		AStackString<> remoteAddr;
		TCPConnectionPool::GetAddressAsString( connection->GetRemoteAddress(), remoteAddr );
		FLOG_WARN( "Disconnecting '%s' due to bad protocol version\n", remoteAddr.Get() );
		Disconnect( connection );
		return;
	}
}

// Process( MsgStatus )
//------------------------------------------------------------------------------
void Server::Process( const ConnectionInfo * connection, const Protocol::MsgStatus * msg )
{
	// take note of latest status of client
	ClientState * cs = (ClientState *)connection->GetUserData();
	cs->m_NumJobsAvailable = msg->GetNumJobsAvailable();
}

// Process( MsgNoJobAvailable )
//------------------------------------------------------------------------------
void Server::Process( const ConnectionInfo * connection, const Protocol::MsgNoJobAvailable * )
{
	// We requested a job, but the client didn't have any left
	ClientState * cs = (ClientState *)connection->GetUserData();
	ASSERT( cs->m_NumJobsRequested > 0 );
	cs->m_NumJobsRequested--;
}

// Process( MsgJob )
//------------------------------------------------------------------------------
void Server::Process( const ConnectionInfo * connection, const Protocol::MsgJob * msg, const void * payload, size_t payloadSize )
{
	ClientState * cs = (ClientState *)connection->GetUserData();
	ASSERT( cs->m_NumJobsRequested > 0 );
	cs->m_NumJobsRequested--;
	cs->m_NumJobsActive++;
	(void)msg;

	// deserialize job
	ConstMemoryStream ms( payload, payloadSize );

	Job * job = new Job( ms );
	job->SetUserData( cs );

	JobQueueRemote::Get().QueueJob( job );
}

// ThreadFuncStatic
//------------------------------------------------------------------------------
/*static*/ uint32_t Server::ThreadFuncStatic( void * param )
{
	Server * s = (Server *)param;
	s->ThreadFunc();
	return 0;
}

// ThreadFunc
//------------------------------------------------------------------------------
void Server::ThreadFunc()
{
	while ( m_ShouldExit == false )
	{
		FindNeedyClients();

		FinalizeCompletedJobs();

		Sleep( 1 );
	}

	m_Exited = true;
}

// FindNeedyClients
//------------------------------------------------------------------------------
void Server::FindNeedyClients()
{
	MutexHolder mh( m_ClientListMutex );

	// determine job availability
	size_t availableJobs = JobQueueRemote::Get().GetNumWorkers();

	auto iter = m_ClientList.Begin();
	auto end = m_ClientList.End();
	for ( ; iter != end; ++iter )
	{
		ClientState * cs = *iter;

		// any jobs requested or in progress reduce the available count
		size_t reservedJobs = cs->m_NumJobsRequested +
							  cs->m_NumJobsActive;
		ASSERT( reservedJobs <= availableJobs );
		availableJobs -= reservedJobs;
		if ( availableJobs == 0 )
		{
			return;
		}
	}

	// we have some jobs available

	// sort clients to find neediest first
	m_ClientList.SortDeref();

	iter = m_ClientList.Begin();
	for ( ; iter != end; ++iter )
	{
		ClientState * cs = *iter;

		size_t reservedJobs = cs->m_NumJobsRequested +
							  cs->m_NumJobsActive;
		if ( reservedJobs >= cs->m_NumJobsAvailable )
		{
			continue; // we've maxed out the requests to this worker
		}

		// request job from this client
		Protocol::MsgRequestJob msg;
		msg.Send( cs->m_Connection );
		cs->m_NumJobsRequested++;
		availableJobs--;
		if ( availableJobs == 0 )
		{
			return; // used up all jobs
		}
	}
}

// FinalizeCompletedJobs
//------------------------------------------------------------------------------
void Server::FinalizeCompletedJobs()
{
	JobQueueRemote & jcr = JobQueueRemote::Get();
	while ( Job * job = jcr.GetCompletedJob() )
	{
		// get associated connection
		ClientState * cs = (ClientState *)job->GetUserData();

		MutexHolder mh( m_ClientListMutex );

		bool connectionStillActive = ( m_ClientList.Find( cs ) != nullptr );
		if ( connectionStillActive )
		{
			Node::State result = job->GetNode()->GetState();
			ASSERT( ( result == Node::UP_TO_DATE ) || ( result == Node::FAILED ) );

			MemoryStream ms;
			ms.Write( job->GetJobId() );
			ms.Write( job->GetNode()->GetName() );
			ms.Write( result == Node::UP_TO_DATE );
			ms.Write( job->GetNode()->GetLastBuildTime() );

			// write the data - build result for success, or output+errors for failure
			ms.Write( (uint32_t)job->GetDataSize() );
			ms.WriteBuffer( job->GetData(), job->GetDataSize() );

			ASSERT( cs->m_NumJobsActive );
			cs->m_NumJobsActive--;

			Protocol::MsgJobResult msg;
			msg.Send( cs->m_Connection, ms );
		}
		else
		{
			// we might get here without finding the connection
			// (if the connection was lost before we completed)
		}

		delete job->GetNode();
		delete job;
	}
}

//------------------------------------------------------------------------------
