// Protocol.cpp - Defines network communication protocol
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Protocol.h"

#include <Core/FileIO/MemoryStream.h>
#include <Core/Network/TCPConnectionPool.h>

// Forward Declarations
//------------------------------------------------------------------------------

// IMessage
//------------------------------------------------------------------------------
Protocol::IMessage::IMessage( Protocol::MessageType msgType, uint32_t msgSize, bool hasPayload )
	: m_MsgType( msgType )
	, m_MsgSize( msgSize )
	, m_HasPayload( hasPayload )
{
}

// IMessage::Send
//------------------------------------------------------------------------------
void Protocol::IMessage::Send( const ConnectionInfo * connection ) const
{
	ASSERT( connection );
	ASSERT( m_HasPayload == false ); // must implement custom function

	// the packet is the contents of the derived class
	TCPConnectionPool & pool = connection->GetTCPConnectionPool();
	pool.Send( connection, this, m_MsgSize );
}

// IMessage::Broadcast
//------------------------------------------------------------------------------
void Protocol::IMessage::Broadcast( TCPConnectionPool * pool ) const
{
	ASSERT( pool );
	ASSERT( m_HasPayload == false ); // must implement custom function

	// the packet is the contents of the derived class
	pool->Broadcast( this, m_MsgSize );
}

// MsgConnection
//------------------------------------------------------------------------------
Protocol::MsgConnection::MsgConnection()
	: Protocol::IMessage( Protocol::MSG_CONNECTION, sizeof( MsgConnection ), false )
	, m_ProtocolVersion( PROTOCOL_VERSION )
{
}

// MsgStatus
//------------------------------------------------------------------------------
Protocol::MsgStatus::MsgStatus( uint32_t numJobsAvailable )
	: Protocol::IMessage( Protocol::MSG_STATUS, sizeof( MsgStatus ), false )
	, m_NumJobsAvailable( numJobsAvailable )
{
}

// MsgRequestJob
//------------------------------------------------------------------------------
Protocol::MsgRequestJob::MsgRequestJob()
	: Protocol::IMessage( Protocol::MSG_REQUEST_JOB, sizeof( MsgRequestJob ), false )
{
}

// MsgNoJobAvailable
//------------------------------------------------------------------------------
Protocol::MsgNoJobAvailable::MsgNoJobAvailable()
	: Protocol::IMessage( Protocol::MSG_NO_JOB_AVAILABLE, sizeof( MsgNoJobAvailable ), false )
{
}

// MsgJob
//------------------------------------------------------------------------------
Protocol::MsgJob::MsgJob()
	: Protocol::IMessage( Protocol::MSG_JOB, sizeof( MsgJob ), true )
{
}

// MsgJob::Send
//------------------------------------------------------------------------------
void Protocol::MsgJob::Send( const ConnectionInfo * connection, const MemoryStream & ms ) const
{
	TCPConnectionPool & pool = connection->GetTCPConnectionPool();
	pool.Send( connection, this, m_MsgSize );
	pool.Send( connection, ms.GetData(), ms.GetSize() );
}


// MsgJobResult
//------------------------------------------------------------------------------
Protocol::MsgJobResult::MsgJobResult()
	: Protocol::IMessage( Protocol::MSG_JOB_RESULT, sizeof( MsgJobResult ), true )
{
}

// MsgJobResult::Send
//------------------------------------------------------------------------------
void Protocol::MsgJobResult::Send( const ConnectionInfo * connection, const MemoryStream & ms ) const
{
	TCPConnectionPool & pool = connection->GetTCPConnectionPool();
	pool.Send( connection, this, m_MsgSize );
	pool.Send( connection, ms.GetData(), ms.GetSize() );
}

//------------------------------------------------------------------------------
