// FBuildStats
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FBuildStats.h"

#include <Core/Strings/AStackString.h>
#include <Core/Tracing/Tracing.h>

// NodeCostSorter
//------------------------------------------------------------------------------
class NodeCostSorter
{
public:
	inline bool operator () ( const Node * a, const Node * b ) const
	{
		return ( a->GetLastBuildTime() > b->GetLastBuildTime() );
	}
};

// CONSTRUCTOR - FBuildStats
//------------------------------------------------------------------------------
FBuildStats::FBuildStats()
	: m_NodeTimeTotalms( 0 )
	, m_NodeTimeProgressms( 0 )
	, m_TotalBuildTime( 0.0f )
	, m_TotalLocalCPUTimeMS( 0 )
	, m_MostExpensiveNodes( 10, false )
	, m_LowestTimeOfExpensiveNodes( 0 )
{}

// CONSTRUCTOR - FBuildStats::Stats
//------------------------------------------------------------------------------
FBuildStats::Stats::Stats()
	: m_NumProcessed( 0 )
	, m_NumBuilt( 0 )
	, m_NumCacheHits( 0 )
	, m_NumCacheMisses( 0 )
	, m_NumCacheStores( 0 )
{}

// GatherPostBuildStatistics
//------------------------------------------------------------------------------
void FBuildStats::GatherPostBuildStatistics( Node * node )
{
	// recurse and gather the per-node-type statistics
	GatherPostBuildStatisticsRecurse( node );

	NodeCostSorter ncs;
	m_MostExpensiveNodes.Sort( ncs );

	// Total the stats
	for ( uint32_t i=0; i< Node::NUM_NODE_TYPES; ++i )
	{
		m_Totals.m_NumProcessed		+= m_PerTypeStats[ i ].m_NumProcessed;
		m_Totals.m_NumBuilt			+= m_PerTypeStats[ i ].m_NumBuilt;
		m_Totals.m_NumCacheHits		+= m_PerTypeStats[ i ].m_NumCacheHits;
		m_Totals.m_NumCacheMisses	+= m_PerTypeStats[ i ].m_NumCacheMisses;
		m_Totals.m_NumCacheStores	+= m_PerTypeStats[ i ].m_NumCacheStores;
	}
}

// OutputSummary
//------------------------------------------------------------------------------
void FBuildStats::OutputSummary() const
{

	// Top 10 cost items
	if ( m_MostExpensiveNodes.IsEmpty() == false )
	{
		OUTPUT( "----------------- Most Expensive -----------------\n" );
		OUTPUT( "Time\tName:\n" );
		for ( uint32_t i=0; i<m_MostExpensiveNodes.GetSize(); ++i )
		{
			const Node * n = m_MostExpensiveNodes[ i ];
			OUTPUT( "%2.2fs\t%s\n", ( (float)n->GetLastBuildTime() / 1000.0f ), n->GetName().Get() );
		}
		OUTPUT( "\n" );
	}

	OUTPUT( "----------------- Summary -----------------\n" );

	// Per-Node type stats
	// NOTE: Only showing the interesting nodes
	OUTPUT( "                                 /----- Cache -----\\\n" );
	OUTPUT( "Build:          Seen    Built   Hit     Miss    Store\n" );
	for ( uint32_t i=0; i< Node::NUM_NODE_TYPES; ++i )
	{
		// only show nodes of interest
		const char * typeName = nullptr;
		switch ( i )
		{
			case Node::OBJECT_NODE:		typeName = "Object    ";	break;
			case Node::LIBRARY_NODE:	typeName = "Library   ";	break;
			case Node::LINKER_NODE:		typeName = "Link      ";	break;
			default: break;
		}
		if ( typeName == nullptr )
		{
			continue;
		}

		const Stats & stats = m_PerTypeStats[ i ];
		OUTPUT( " - %s : %u\t%u\t%u\t%u\t%u\n", // TODO:C Robustify column formatting
						typeName,
						stats.m_NumProcessed,
						stats.m_NumBuilt,
						stats.m_NumCacheHits,
						stats.m_NumCacheMisses,
						stats.m_NumCacheStores );
	}
	OUTPUT( "Cache:\n" );
	{
		const uint32_t hits = m_Totals.m_NumCacheHits;
		const uint32_t misses = m_Totals.m_NumCacheMisses;
		const uint32_t stores = m_Totals.m_NumCacheStores;
		float hitPerc = 0.0f;
		if ( hits > 0 || misses > 0 )
		{
			hitPerc = ( (float)hits / float( hits + misses ) * 100.0f );
		}
		OUTPUT( " - Hits       : %u (%2.1f %%)\n", hits, hitPerc );
		OUTPUT( " - Misses     : %u\n", misses );
		OUTPUT( " - Stores     : %u\n", stores );
	}

	AStackString<> buffer;
	FormatTime( m_TotalBuildTime, buffer );
	OUTPUT( "Time:\n" );
	OUTPUT( " - Real       : %s\n", buffer.Get() );
	float totalLocalCPUInSeconds = (float)( (double)m_TotalLocalCPUTimeMS / (double)1000 );
	FormatTime( totalLocalCPUInSeconds, buffer );
	float localRatio = ( totalLocalCPUInSeconds / m_TotalBuildTime );
	OUTPUT( " - Local CPU  : %s (%2.1f:1)\n", buffer.Get(), localRatio );
//	OUTPUT( " - Remote CPU : --:--:-- (-.-x)\n" );	// TODO:C Find a way to implement this (when distributed compilation is implemented)
	OUTPUT( "-------------------------------------------\n" );
}

// GatherPostBuildStatisticsRecurse
//------------------------------------------------------------------------------
void FBuildStats::GatherPostBuildStatisticsRecurse( Node * node )
{
	// have we seen this node when gathering stats?
	if ( node->GetStatFlag( Node::STATS_STATS_PROCESSED ) )
	{
		return;
	}

	Node::Type nodeType = node->GetType();
	Stats & stats = m_PerTypeStats[ nodeType ];

	stats.m_NumProcessed++;

	m_TotalLocalCPUTimeMS += node->GetProcessingTime();

	if ( node->GetStatFlag( Node::STATS_BUILT ) )
	{
		stats.m_NumBuilt++;

		// Is this node more costly than the cheapest one in our Top-N list?
		// (if the list is not full yet, then the cheapest value will still be 0
		//  so it will get added no matter what)
		if ( node->GetLastBuildTime() > m_LowestTimeOfExpensiveNodes )
		{
			// if array full, erase the cheapest
			if ( m_MostExpensiveNodes.IsAtCapacity() == true )
			{
				const Array< Node * >::Iter end = m_MostExpensiveNodes.End();
				for ( Array< Node * >::Iter it = m_MostExpensiveNodes.Begin(); it != end; it++ )
				{
					if ( ( *it )->GetLastBuildTime() == m_LowestTimeOfExpensiveNodes )
					{
						m_MostExpensiveNodes.Erase( it );
						break;
					}
				}
			}

			// add our node
			m_MostExpensiveNodes.Append( node );

			// is it full now?
			if ( m_MostExpensiveNodes.IsAtCapacity() == true )
			{
				// calculate actual least expensive
				m_LowestTimeOfExpensiveNodes = m_MostExpensiveNodes[ 0 ]->GetLastBuildTime();
				for ( uint32_t i=1; i<m_MostExpensiveNodes.GetCapacity(); ++i )
				{
					m_LowestTimeOfExpensiveNodes = Math::Min( m_LowestTimeOfExpensiveNodes, m_MostExpensiveNodes[ i ]->GetLastBuildTime() );
				}
			}
		}
	}
	if ( node->GetStatFlag( Node::STATS_CACHE_HIT ) )
	{
		stats.m_NumCacheHits++;
	}
	if ( node->GetStatFlag( Node::STATS_CACHE_MISS ) )
	{
		stats.m_NumCacheMisses++;
	}
	if ( node->GetStatFlag( Node::STATS_CACHE_STORE ) )
	{
		stats.m_NumCacheStores++;
	}

	// mark this node as processed to prevent multiple recursion
	node->SetStatFlag( Node::STATS_STATS_PROCESSED );

	// handle static deps
	{
		const Array< Node * > & staticDeps = node->GetStaticDependencies();
		const Array< Node * >::Iter end = staticDeps.End();
		for ( Array< Node * >::Iter it = staticDeps.Begin();
			  it != end;
			  it++ )
		{
			GatherPostBuildStatisticsRecurse( *it );
		}
	}

	// handle dynamic deps
	{
		const Array< Node * > & dynamicDeps = node->GetDynamicDependencies();
		const Array< Node * >::Iter end = dynamicDeps.End();
		for ( Array< Node * >::Iter it = dynamicDeps.Begin();
			  it != end;
			  it++ )
		{
			GatherPostBuildStatisticsRecurse( *it );
		}
	}
}

// FormatTime
//------------------------------------------------------------------------------
void FBuildStats::FormatTime( float timeInSeconds , AString & buffer ) const
{
	buffer.Clear();

	uint32_t days = (uint32_t)( timeInSeconds / ( 24.0f * 60.0f * 60.0f ) );
	timeInSeconds -= ( (float)days * ( 24.0f * 60.0f * 60.0f ) );
	uint32_t hours = (uint32_t)( timeInSeconds / ( 60.0f * 60.0f ) );
	timeInSeconds -= ( (float)hours * ( 60.0f * 60.0f ) );
	uint32_t mins = (uint32_t)( timeInSeconds / 60.0f );
	timeInSeconds -= ( (float)mins * 60.0f );

	AStackString<> temp;

	if ( days > 0 )
	{
		temp.Format( "%u days, ", days );
		buffer += temp;
	}
	if ( hours > 0 )
	{
		temp.Format( "%uh:", hours );
		buffer += temp;
	}
	if ( mins > 0 )
	{
		temp.Format( "%2um:", mins );
		buffer += temp;
	}

	temp.Format( "%2.2fs", timeInSeconds );
	buffer += temp;
}

//------------------------------------------------------------------------------
