// CIncludeParser
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "CIncludeParser.h"

#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

#include <Core/Math/CRC32.h>
#include <Core/Strings/AStackString.h>

#include <string.h>

#include <Core/Tracing/Tracing.h>

//------------------------------------------------------------------------------
CIncludeParser::CIncludeParser()
: m_Includes( 0, true )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
CIncludeParser::~CIncludeParser()
{
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseMSCL_Output( const char * compilerOutput, 
									   size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "Note: including file:", false );
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseMSCL_Preprocessed( const char * compilerOutput, 
											 size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "#line", true );
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::ParseGCC_Preprocessed( const char * compilerOutput, 
											size_t compilerOutputSize )
{
	return Parse( compilerOutput, compilerOutputSize, "#line", true );
}

// Parse
//------------------------------------------------------------------------------
bool CIncludeParser::Parse( const char * compilerOutput, 
							size_t compilerOutputSize,
							const char * startOfLineString,
							bool quoted )
{
	m_Includes.SetCapacity( 4096 );

	// store crc of includes found so far to prevent adding duplicates
	Array< uint32_t > crcs( 4096, true );

	const char * pos = compilerOutput;
	const char * end = compilerOutput + compilerOutputSize;

	const size_t startOfLineStringLen = strlen( startOfLineString );

	const char firstChar = startOfLineString[ 0 ];

	uint32_t lastCRC = 0;

	while ( pos < end )
	{
		if ( *pos != firstChar )
		{
			pos++;
			continue;
		}

		// # must be at start of line (either the first char of the file, or after a carriage return)
		if ( ( pos != compilerOutput ) && ( pos[-1] != '\n' ) )
		{
			pos++;
			continue;
		}

		const char * src = startOfLineString;
		const char * localEnd = pos + startOfLineStringLen;
		while ( ( *pos++ == *src++ ) && ( pos < localEnd ) ) {}

		if ( pos != localEnd )
		{
			continue;
		}

		// find end of line
		const char * lineEnd = pos + 1;
		while ( ( lineEnd < end ) && 
				( ( *lineEnd != '\r' ) && ( *lineEnd != '\n' ) ) )
		{
			lineEnd++;
		}

		if ( quoted )
		{
			// find surrounding quotes
			while ( ( pos < lineEnd ) && ( *pos != '"' ) )
			{
				pos++;
			}
			pos++;
			while ( ( lineEnd > pos ) && ( *lineEnd != '"' ) )
			{
				lineEnd--;
			}
			if ( ( lineEnd - pos ) < 1 )
			{
				return false; // caller is responsible for emitting error msg
			}
		}
		else
		{
			// advance past whitespace before path		
			while ( ( *pos == ' ' ) || ( *pos == '\t' ) )
			{
				pos++;
			}
		}

		// record include
		AStackString< 256 > include( pos, lineEnd );
		AStackString< 256 > cleanInclude;
		NodeGraph::CleanPath( include, cleanInclude );
		uint32_t crc = CRC32::CalcLower( cleanInclude );
		if ( crc == lastCRC )
		{
			continue;
		}
		lastCRC = crc;
		if ( crcs.Find( crc ) == nullptr )
		{
			crcs.Append( crc );
			m_Includes.Append( cleanInclude );
		}

		// continue looking for next include
	}

	return true;
}

//------------------------------------------------------------------------------
