// UnityNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "UnityNode.h"
#include "DirectoryListNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/ObjectNode.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Process/Process.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
UnityNode::UnityNode( const AString & unityName,
						const Array< DirectoryListNode * > & dirNodes,
						const AString & outputPath,
						const AString & outputPattern,
						uint32_t numUnityFilesToCreate,
						const AString & precompiledHeader,
						const Array< AString > & filesToExclude )
: Node( unityName, Node::UNITY_NODE, Node::FLAG_TRIVIAL_BUILD )
, m_StaticDependencies( dirNodes.GetSize(), false )
, m_OutputPath( outputPath )
, m_OutputPattern( outputPattern )
, m_NumUnityFilesToCreate( numUnityFilesToCreate )
, m_UnityFileNames( numUnityFilesToCreate, false )
, m_PrecompiledHeader( precompiledHeader )
, m_FilesToExclude( filesToExclude )
{
	m_LastBuildTimeMs = 100; // higher default than a file node

	// depend on the input nodes
	auto end = dirNodes.End();
	for ( auto it = dirNodes.Begin(); it !=end; ++it )
	{
		m_StaticDependencies.Append( *it );
	}

	ASSERT( m_NumUnityFilesToCreate > 0 );

	// ensure path is properly formatted
	ASSERT( m_OutputPath.EndsWith( '\\' ) );

	// generate the destination unity file names
	// TODO:C move this processing to the FunctionUnity
	AStackString<> tmp;
	for ( size_t i=0; i< m_NumUnityFilesToCreate; ++i )
	{
		tmp.Format( "%u", i + 1 ); // number from 1

		AStackString<> unityFileName( m_OutputPath );
		unityFileName += m_OutputPattern;
		unityFileName.Replace( "*", tmp.Get() );

		m_UnityFileNames.Append( unityFileName );
	}
}

// DESTRUCTOR
//------------------------------------------------------------------------------
UnityNode::~UnityNode()
{
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool UnityNode::DetermineNeedToBuild( bool UNUSED( forceClean ) )
{
	return true; // Unity nodes always "build"
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult UnityNode::DoBuild( Job * UNUSED( job ) )
{
	bool hasOutputMessage = false; // print msg first time we actually save a file

	// Ensure dest path exists
	// NOTE: Normally a node doesn't need to worry about this, but because
	// UnityNode outputs files that do not match the node-name (and doesn't
	// inherit from FileNoe), we have to handle it ourselves
	// TODO:C Would be good to refactor things to avoid this special case
	if ( EnsurePathExistsForFile( m_OutputPath ) == false )
	{
		return NODE_RESULT_FAILED;
	}

	// get the files
	Array< AString > unfilteredFiles( 4096, true );
	auto sEnd = m_StaticDependencies.End();
	for ( auto sIt = m_StaticDependencies.Begin(); sIt != sEnd; ++sIt )
	{
		DirectoryListNode * dirNode = ( *sIt )->CastTo< DirectoryListNode >();
		auto filesEnd = dirNode->GetFiles().End();
		for ( auto filesIt = dirNode->GetFiles().Begin(); filesIt != filesEnd; ++filesIt )
		{
			unfilteredFiles.Append( *filesIt );
		}
	}

	// filter out files we want to discard
	Array< const AString * > files( unfilteredFiles.GetSize() );
	auto it = unfilteredFiles.Begin();
	auto end = unfilteredFiles.End();
	for ( ; it != end; ++it )
	{
		bool keep = true;
		auto fit = m_FilesToExclude.Begin();
		auto fend = m_FilesToExclude.End();
		for ( ; fit != fend; ++fit )
		{
			if ( it->EndsWithI( *fit ) )
			{
				keep = false;
				break;
			}
		}
		if ( keep )
		{
			files.Append( it );
		}
	}

	// how many files should go in each unity file?
	const size_t numFiles = files.GetSize();
	size_t numFilesPerUnity = numFiles / m_NumUnityFilesToCreate;

	// ensure that we round up so no files are lost
	if ( ( numFilesPerUnity * m_NumUnityFilesToCreate ) < numFiles )
	{
		numFilesPerUnity++;
		ASSERT( ( numFilesPerUnity * m_NumUnityFilesToCreate ) >= numFiles );
	}

	size_t index = 0;

	// create each unity file
	for ( size_t i=0; i<m_NumUnityFilesToCreate; ++i )
	{
		// header
		AStackString<4096> output( "// Auto-generated Unity file - do not modify\r\n\r\n" );
		
		// precompiled header
		if ( !m_PrecompiledHeader.IsEmpty() )
		{
			output += "#include \"";
			output += m_PrecompiledHeader;
			output += "\"\r\n";
		}

		// write all the includes
		for ( size_t j=0; j<numFilesPerUnity; ++j )
		{
			// handle cases where there's not enough files to put in each unity
			if ( index >= numFiles )
			{
				break;
			}

			// write include
			output += "#include <";
			output += *(files[ index++ ]);
			output += ">\r\n";
		}
		output += "\r\n";

		// generate the destination unity file name
		const AString & unityName = m_UnityFileNames[ i ];

		// need to write the unity file?
		bool needToWrite = false;
		FileStream f;
		if ( FBuild::Get().GetOptions().m_ForceCleanBuild )
		{
			needToWrite = true; // clean build forces regeneration
		}
		else
		{
			if ( f.Open( unityName.Get(), FileStream::READ_ONLY ) )
			{
				const size_t fileSize( (size_t)f.GetFileSize() );
				if ( output.GetLength() != fileSize )
				{
					// output not the same size as the file on disc
					needToWrite = true;
				}
				else
				{
					// files the same size - are the contents the same?
					char * mem = new char[ fileSize ];
					if ( f.Read( mem, fileSize ) != fileSize )
					{
						// problem reading file - try to write it again
						needToWrite = true;
					}
					else
					{
						if ( AString::StrNCmp( mem, output.Get(), fileSize ) != 0 )
						{
							// contents differ
							needToWrite = true;
						}
					}
				}
				f.Close();
			}
			else
			{
				// file missing - must create
				needToWrite = true;
			}
		}

		// needs updating?
		if ( needToWrite )
		{
			if ( hasOutputMessage == false )
			{
				FLOG_BUILD( "Uni: %s\n", GetName().Get() );
				hasOutputMessage = true;
			}

			if ( f.Open( unityName.Get(), FileStream::WRITE_ONLY ) == false )
			{
				FLOG_ERROR( "Failed to create Unity file '%s'", unityName.Get() );
				return NODE_RESULT_FAILED;
			}

			if ( f.Write( output.Get(), output.GetLength() ) != output.GetLength() )
			{
				FLOG_ERROR( "Error writing Unity file '%s'", unityName.Get() );
				return NODE_RESULT_FAILED;
			}

			f.Close();
		}
	}


	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & UnityNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * UnityNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( AStackString<>,	outputPath );
	NODE_LOAD( AStackString<>,	outputPattern );
	NODE_LOAD( uint32_t,		numFiles );
	NODE_LOAD_DEPS( 1,			staticDeps );
	NODE_LOAD( AStackString<>,	precompiledHeader );
	NODE_LOAD( Array< AString >, filesToExclude );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	UnityNode * n = ng.CreateUnityNode( name, 
								 reinterpret_cast< Array< DirectoryListNode * > & >( staticDeps ), // all static deps are DirectoryListNode
								 outputPath, 
								 outputPattern, 
								 numFiles,
								 precompiledHeader,
								 filesToExclude );
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool UnityNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_OutputPath );
	NODE_SAVE( m_OutputPattern );
	NODE_SAVE( m_NumUnityFilesToCreate );
	NODE_SAVE_DEPS( m_StaticDependencies );
	NODE_SAVE( m_PrecompiledHeader );
	NODE_SAVE( m_FilesToExclude );
	return true;
}

//------------------------------------------------------------------------------
