// TestNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "TestNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Math/Conversions.h>
#include <Core/Strings/AStackString.h>
#include <Core/Process/Process.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
TestNode::TestNode( const AString & testOutput,
					FileNode * testExecutable,
					const AString & arguments,
					const AString & workingDir )
	: FileNode( testOutput, 
				ControlFlag::FLAG_NO_DELETE_ON_FAIL ) // keep output on test fail
	, m_StaticDependencies( 1, false )
	, m_Executable( testExecutable )
	, m_Arguments( arguments )
	, m_WorkingDir( workingDir )
{
	ASSERT( testExecutable );
	m_StaticDependencies.Append( testExecutable );
	m_Type = TEST_NODE;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
TestNode::~TestNode()
{
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool TestNode::DetermineNeedToBuild( bool forceClean )
{
	// note the stamp of our output file for later comparison to the test exe
	// (will be 0 if file is missing)
	uint64_t timeStamp = FileIO::GetFileLastWriteTime( m_Name );

	// the test output is missing, so the test needs running
	if ( timeStamp == 0 )
	{
		return true;
	}

	// the test output doesn't match what we saw last time, so the test needs running
	if ( timeStamp != m_TimeStamp )
	{
		return true;
	}

	// test exe is newer than output, so it needs re-running
	if ( m_Executable->GetTimeStamp() > timeStamp )
	{
		return true;
	}

	if ( forceClean )
	{
		return true;
	}

	return false;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult TestNode::DoBuild( Job * UNUSED( job ) )
{
	FLOG_BUILD( "Running Test: %s\n", GetName().Get() );

	// If the workingDir is empty, use the current dir for the process
	const char * workingDir = m_WorkingDir.IsEmpty() ? nullptr : m_WorkingDir.Get();

	// spawn the process
	Process p;
	bool spawnOK = p.Spawn( m_Executable->GetName().Get(),
							m_Arguments.Get(),
							workingDir,
							FBuild::Get().GetEnvironmentString() );

	if ( !spawnOK )
	{
		FLOG_ERROR( "Failed to spawn process for '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	ASSERT( !p.IsRunning() );
	// Get result
	int result = p.WaitForExit();
	if ( result != 0 )
	{
		// something went wrong, print details
		Node::DumpOutput( memOut.Get(), memOutSize );
		Node::DumpOutput( memErr.Get(), memErrSize );
	}

	// write the test output (saved for pass or fail)
	FileStream fs;
	if ( fs.Open( GetName().Get(), FileStream::WRITE_ONLY ) == false )
	{
		FLOG_ERROR( "Failed to open test output file '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}
	if ( ( memOut.Get() && ( fs.Write( memOut.Get(), memOutSize ) == false ) ) ||
		 ( memErr.Get() && ( fs.Write( memErr.Get(), memErrSize ) == false ) ) )
	{
		FLOG_ERROR( "Failed to write test output file '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}
	fs.Close();

	// did the test fail?
	if ( result != 0 )
	{
		FLOG_ERROR( "Test failed (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// test passed 
	// we only keep the "last modified" time of the test output for passed tests
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );
	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & TestNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool TestNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_TimeStamp );
	NODE_SAVE( m_Executable->GetName() );
	NODE_SAVE( m_Arguments );
	NODE_SAVE( m_WorkingDir );
	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * TestNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	fileName );
	NODE_LOAD( uint64_t,		timeStamp );
	NODE_LOAD( AStackString<>,	executable );
	NODE_LOAD( AStackString<>,	arguments );
	NODE_LOAD( AStackString<>,	workingDir );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	Node * execNode = ng.FindNode( executable );
	ASSERT( execNode ); // load/save logic should ensure the src was saved first
	ASSERT( execNode->IsAFile() );

	TestNode * n = ng.CreateTestNode( fileName, 
									  (FileNode *)execNode,
									  arguments,
									  workingDir );
	ASSERT( n );
	n->m_TimeStamp = timeStamp;
	return n;
}

//------------------------------------------------------------------------------
