// ObjectNode.h - a node that builds a single object from a source file
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_GRAPH_OBJECTNODE_H
#define FBUILD_GRAPH_OBJECTNODE_H

// Includes
//------------------------------------------------------------------------------
#include "FileNode.h"
#include <Core/Containers/Array.h>
#include <Core/Containers/AutoPtr.h>

// Forward Declarations
//------------------------------------------------------------------------------

// ObjectNode
//------------------------------------------------------------------------------
class ObjectNode : public FileNode
{
public:
	explicit ObjectNode( const AString & objectName,
						 Node * inputNode,
						 Node * compilerNode,
						 const AString & compilerArgs,
						 Node * precompiledHeader,
						 uint32_t flags );
	virtual ~ObjectNode();

	static inline Node::Type GetType() { return Node::OBJECT_NODE; }

	virtual const Array< Node * > & GetStaticDependencies() const;
	virtual const Array< Node * > & GetDynamicDependencies() const;

	enum Flags
	{
		FLAG_CAN_BE_CACHED		=	0x01,
		FLAG_CAN_BE_DISTRIBUTED	=	0x02,
		FLAG_USING_PCH			=	0x04,
		FLAG_GCC				=	0x10,
		FLAG_MSVC				=	0x20,
		FLAG_CREATING_PCH		=   0x40,
	};

	virtual bool Save( IOStream & stream ) const;
	static Node * Load( IOStream & stream, bool remote );
private:
	virtual bool DoDynamicDependencies( bool forceClean );
	virtual bool DetermineNeedToBuild( bool forceClean );
	virtual BuildResult DoBuild( Job * job );
	virtual BuildResult DoBuild2( Job * job );

	BuildResult DoBuildMSCL_NoCache();
	BuildResult DoBuildWithPreProcessor( Job * job );
	BuildResult DoBuildWithPreProcessor2( Job * job );
	BuildResult DoBuildOther();

	bool ProcessIncludesMSCL( const char * output, uint32_t outputSize, bool preprocessed );
	bool ProcessIncludesGCC( const char * output, uint32_t outputSize );
	bool ProcessIncludesWithPreProcessor( Job * job );

	void GetCacheName( const void * memOut, size_t memOutSize,
					   AString & cacheName ) const;
	bool RetrieveFromCache( const AString & cacheFile );
	void WriteToCache( const AString & cacheFile );

	static void DumpOutput( const char * data, uint32_t dataSize );

	void EmitCompilationMessage( const AString & fullArgs ) const;

	void BuildFullArgs( AString & fullArgs, bool preProcess ) const;
	bool BuildPreprocessedOutput( const AString & fullArgs, Job * job ) const;
	bool WriteTmpFile( Job * job, AString & tmpFileName ) const;
	bool BuildFinalOutput( Job * job, const AString & fullArgs ) const;

	inline bool GetFlag( uint32_t flag ) const { return ( ( m_Flags & flag ) != 0 ); }

	Array< Node * > m_StaticDependencies;
	Array< Node * > m_DynamicDependencies;

	Array< AString > m_Includes;
	bool m_HasIncludes;
	uint32_t m_Flags;
	AString m_CompilerArgs;
};

//------------------------------------------------------------------------------
#endif // FBUILD_GRAPH_OBJECTNODE_H
