// ObjectNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "ObjectNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Helpers/CIncludeParser.h>
#include <Tools/FBuild/FBuildCore/Helpers/Compressor.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/Job.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/WorkerThread.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Math/CRC32.h>
#include <Core/Process/Process.h>
#include <Core/Tracing/Tracing.h>
#include <Core/Strings/AStackString.h>

#include <windows.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::ObjectNode( const AString & objectName,
					    Node * inputNode,
					    Node * compilerNode,
						const AString & compilerArgs,
						Node * precompiledHeader,
						uint32_t flags )
: FileNode( objectName, Node::FLAG_NONE )
, m_StaticDependencies( 3, false )
, m_DynamicDependencies( 0, true )
, m_Includes( 0, true )
, m_HasIncludes( false )
, m_Flags( flags )
, m_CompilerArgs( compilerArgs )
{
	ASSERT( compilerNode );
	m_StaticDependencies.Append( compilerNode );

	ASSERT( inputNode );
	m_StaticDependencies.Append( inputNode );

	if ( precompiledHeader )
	{
		m_StaticDependencies.Append( precompiledHeader );
	}

	m_Type = OBJECT_NODE;
	m_LastBuildTimeMs = 5000; // higher default than a file node
}

// DESTRUCTOR
//------------------------------------------------------------------------------
ObjectNode::~ObjectNode()
{
}

// DoDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ bool ObjectNode::DoDynamicDependencies( bool forceClean )
{
	if ( forceClean == true )
	{
		return true; // includes will be re-built as part of build step
	}

	ASSERT( m_DynamicDependencies.GetSize() == 0 );

	//Timer t;

	if ( m_HasIncludes )
	{
		NodeGraph & ng = FBuild::Get().GetDependencyGraph();

		// convert includes to nodes
		m_DynamicDependencies.SetCapacity( m_Includes.GetSize() );
		for ( Array< AString >::ConstIter it = m_Includes.Begin();
			  it != m_Includes.End();
			  it++ )
		{
			Node * fn = FBuild::Get().GetDependencyGraph().FindNode( *it );
			if ( fn == nullptr )
			{
				fn = ng.CreateFileNode( *it );
			}
			else if ( fn->IsAFile() == false )
			{
				FLOG_ERROR( "'%s' is not a FileNode (type: %s)", fn->GetName().Get(), fn->GetTypeName() );
				return false;
			}
			m_DynamicDependencies.Append( fn );
		}
	}

	//float time = t.GetElapsed();
	//FLOG_WARN( "DynamicDeps: %2.3f\t%s (%u includes)", time, GetName().Get(), m_Includes.GetSize() );

	return true;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuild( Job * job )
{
	// delete previous file
	if ( FileIO::FileExists( GetName().Get() ) )
	{
		if ( FileIO::FileDelete( GetName().Get() ) == false )
		{
			FLOG_ERROR( "Failed to delete file before build '%s'", GetName().Get() );
			return NODE_RESULT_FAILED;
		}
	}
	if ( GetFlag( FLAG_MSVC ) && GetFlag( FLAG_CREATING_PCH ) )
	{
		AStackString<> pchObj( GetName() );
		pchObj += ".obj";
		if ( FileIO::FileExists( pchObj.Get() ) )
		{
			if ( FileIO::FileDelete( pchObj.Get() ) == false )
			{
				FLOG_ERROR( "Failed to delete file before build '%s'", GetName().Get() );
				return NODE_RESULT_FAILED;
			}
		}
	}

	bool useCache = GetFlag( FLAG_CAN_BE_CACHED ) &&
					( FBuild::Get().GetOptions().m_UseCacheRead ||
					  FBuild::Get().GetOptions().m_UseCacheWrite );
	bool useDist = GetFlag( FLAG_CAN_BE_DISTRIBUTED );
	bool usePreProcessor = ( useCache || useDist || GetFlag( FLAG_GCC ) );

	if ( usePreProcessor )
	{
		return DoBuildWithPreProcessor( job );
	}

	if ( GetFlag( FLAG_MSVC ) )
	{
		return DoBuildMSCL_NoCache();
	}

	return DoBuildOther();
}

// DoBuild_Remote
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuild2( Job * job )
{
	return DoBuildWithPreProcessor2( job );
}

// DoBuildMSCL_NoCache
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuildMSCL_NoCache()
{
	// Format compiler args string
	AStackString< 8 * KILOBYTE > fullArgs;
	fullArgs = m_CompilerArgs;

	fullArgs += " /showIncludes"; // we'll extract dependency information from this

	FileNode * compiler = m_StaticDependencies[ 0 ]->CastTo< FileNode >();
	FileNode * sourceFile = m_StaticDependencies[ 1 ]->CastTo< FileNode >();

	// substitute "function variables" if needed
	fullArgs.Replace( "%1", sourceFile->GetName().Get() );
	fullArgs.Replace( "%2", m_Name.Get() );

	if ( GetFlag( FLAG_CREATING_PCH ) )
	{
		// convert 'PrecompiledHeader.pch' to 'PrecompiledHeader.pch.obj'
		AStackString<> objName( m_Name );
		objName += ".obj";
		fullArgs.Replace( "%3", objName.Get() );
	}

	EmitCompilationMessage( fullArgs );

//  TODO:C Use this
//	if ( BuildFinalOutput( fullArgs ) == false )
//	{
//		return NODE_RESULT_FAILED; // BuildFinalOutput will have emitted error
//	}

	// spawn the process
	Process p;
	if ( false == p.Spawn( compiler->GetName().Get(), 
						   fullArgs.Get(),
						   nullptr, // workingDir - use the exe launch dir
						   FBuild::Get().GetEnvironmentString() ) )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	// Get result
	ASSERT( !p.IsRunning() );
	int result = p.WaitForExit();

	if ( memErr.Get() )
	{ 
		DumpOutput( memErr.Get(), memErrSize );
	}

	if ( result != 0 )
	{
		DumpOutput( memOut.Get(), memOutSize );
		FLOG_ERROR( "Failed to build Object (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// compiled ok, try to extract includes
	if ( ProcessIncludesMSCL( memOut.Get(), memOutSize, false) == false )
	{
		return NODE_RESULT_FAILED; // ProcessIncludesMSCL will have emitted an error
	}

	// record new file time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	return NODE_RESULT_OK;
}

// DoBuildWithPreProcessor
//------------------------------------------------------------------------------
Node::BuildResult ObjectNode::DoBuildWithPreProcessor( Job * job )
{
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, true );

	if ( BuildPreprocessedOutput( fullArgs, job ) == false )
	{
		return NODE_RESULT_FAILED; // BuildPreprocessedOutput will have emitted an error
	}

	// preprocessed ok, try to extract includes
	if ( ProcessIncludesWithPreProcessor( job ) == false )
	{
		return NODE_RESULT_FAILED; // ProcessIncludesWithPreProcessor will have emitted an error
	}

	// calculate the cache entry lookup
	AStackString<> cacheFile;
	if ( FBuild::Get().GetOptions().m_UseCacheRead ||
		 FBuild::Get().GetOptions().m_UseCacheWrite )
	{
		GetCacheName( job->GetData(), job->GetDataSize(), cacheFile );
	}

	// try to get from cache
	if ( RetrieveFromCache( cacheFile ) )
	{
		return NODE_RESULT_OK_CACHE;
	}

	// can we do the rest of the work remotely?
	if ( GetFlag( FLAG_CAN_BE_DISTRIBUTED ) &&
		 FBuild::Get().GetOptions().m_AllowDistributed &&
		 JobQueue::Get().GetDistributableJobsMemUsage() < ( 512 * MEGABYTE ) )
	{
		// compress job data
		Compressor c;
		c.Compress( job->GetData(), job->GetDataSize() );
		size_t compressedSize = c.GetResultSize();
		job->OwnData( c.ReleaseResult(), compressedSize, true );

		// yes... re-queue for secondary build
		return NODE_RESULT_NEED_SECOND_BUILD_PASS;
	}

	// can't do the work remotely, so do it right now
	Node::BuildResult result = DoBuildWithPreProcessor2( job );
	if ( result != Node::NODE_RESULT_OK )
	{
		return result;
	}

	// record new file time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	if ( m_TimeStamp && FBuild::Get().GetOptions().m_UseCacheWrite )
	{
		WriteToCache( cacheFile );
	}

	return Node::NODE_RESULT_OK;
}

// DoBuildWithPreProcessor2
//------------------------------------------------------------------------------
Node::BuildResult ObjectNode::DoBuildWithPreProcessor2( Job * job )
{
	AStackString< 8 * KILOBYTE > fullArgs;
	BuildFullArgs( fullArgs, false );

	bool usePreProcessedOutput = true;

	if ( GetFlag( FLAG_MSVC ) && GetFlag( FLAG_USING_PCH ) && job->IsLocal() )
	{
		// for MSVC, if using precompiled headers, we don't want to compile from the preprocessed
		// output, as (assuming it's well tuned) it's quicker to build using the precompiled header
		// (but we can only do this if building locally, as remotely we don't have access to the
		// pch)
		usePreProcessedOutput = false;
	}
	
	if ( usePreProcessedOutput )
	{
		AStackString<> tmpFileName;
		if ( WriteTmpFile( job, tmpFileName ) == false )
		{
			return NODE_RESULT_FAILED; // WriteTmpFile will have emitted an error
		}

		// use preprocesed output path as input
		Node * sourceFile = m_StaticDependencies[ 1 ];
		fullArgs.Replace( sourceFile->GetName().Get(), tmpFileName.Get() );

		// remove pch related flags ('Yu' and 'Fp')
		if ( GetFlag( FLAG_USING_PCH ) )
		{
			Array< AString > args;
			fullArgs.Tokenize( args );
			fullArgs.Clear();
			auto iter = args.Begin();
			const auto end = args.End();
			for ( ; iter != end; ++iter )
			{
				if ( iter->BeginsWith( "/Yu" ) ||
					 iter->BeginsWith( "/Fp" ) )
				{
					continue;
				}
				if ( iter != args.Begin() )
				{
					fullArgs += ' ';
				}
				fullArgs += *iter;
			}
		}

	}

	if ( BuildFinalOutput( job, fullArgs ) == false )
	{
		return NODE_RESULT_FAILED; // BuildFinalOutput will have emitted error
	}
	return NODE_RESULT_OK;
}

// DoBuildOther
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ObjectNode::DoBuildOther()
{
	// Format compiler args string
	AStackString< 8 * KILOBYTE > fullArgs;
	fullArgs = m_CompilerArgs;

	// substitute "function variables" if needed
	FileNode * sourceFile = m_StaticDependencies[ 1 ]->CastTo< FileNode >();
	fullArgs.Replace( "%1", sourceFile->GetName().Get() );
	fullArgs.Replace( "%2", m_Name.Get() );

	EmitCompilationMessage( fullArgs );

	FileNode * compiler = m_StaticDependencies[ 0 ]->CastTo< FileNode >();

	// spawn the process
	Process p;
	if ( false == p.Spawn( compiler->GetName().Get(), 
						   fullArgs.Get(),
						   nullptr, // use exe launch dir as the working dir 
						   FBuild::Get().GetEnvironmentString() ) )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	// Get result
	ASSERT( !p.IsRunning() );
	int result = p.WaitForExit();
	bool ok = ( result == 0 );

	if ( memErr.Get() )
	{ 
		DumpOutput( memErr.Get(), memErrSize );
	}

	if ( !ok )
	{
		// something went wrong, print details
		DumpOutput( memOut.Get(), memOutSize );
		FLOG_ERROR( "Failed to build Object (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// record new file time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	// we can't track includes, but we don't want a rebuild every time
	// so pretend we have extracted includes for next time
	// TODO:A This is sketchy.  We need to either:
	//  - ensure the type of thing we are building has none
	//  - parse them out ourselves by recursively traversing the sources
	//  - build it every time
	m_HasIncludes = true;

	return NODE_RESULT_OK;
}

// GetStaticDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & ObjectNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// GetDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & ObjectNode::GetDynamicDependencies() const
{
	return m_DynamicDependencies;
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
bool ObjectNode::DetermineNeedToBuild( bool forceClean )
{
	// 
	uint64_t lastWriteTime = FileIO::GetFileLastWriteTime( m_Name );
	if ( lastWriteTime == 0 )
	{
		// output file is missing, so it must be built
		return true;
	}

	if ( GetTimeStamp() == 0 )
	{
		// file exists, be we don't know about it
		// (built by someone else, or we're doing a clean build)
		return true;
	}

	if ( lastWriteTime != GetTimeStamp() )
	{
		// file exists and we've built before, but they are out of sync
		// (probably modified or built by an external process)
		return true;
	}

	FileNode * staticDep = m_StaticDependencies[ 1 ]->CastTo< FileNode >();
	ASSERT( staticDep->GetTimeStamp() != 0 );  // Should never attempt build if a static dep is missing

	// is our output older than the source file?
	// also handles the case where 
	if ( m_TimeStamp < staticDep->GetTimeStamp() )
	{
		return true;
	}

	if ( forceClean )
	{
		return true;
	}

	// do we have a list of includes from a previous build?
	// NOTE: We can't check the size of the array, as it's valid to have no includes
	if ( m_HasIncludes == false )
	{
		// no - we need to compile and store the list of includes
		return true;
	}

	// we have a list of includes - are any out of date or missing?
	for ( Array< Node * >::ConstIter it = m_DynamicDependencies.Begin();
			it != m_DynamicDependencies.End();
			it++ )
	{
		FileNode * fn = ( *it )->CastTo< FileNode >();
		if ( fn->GetTimeStamp() == 0 )
		{
			// file missing
			return true;
		}

		if ( m_TimeStamp < fn->GetTimeStamp() )
		{
			// file changed
			return true;
		}
	}

	// nothing needs building
	return false;
}

// ProcessIncludesMSCL
//------------------------------------------------------------------------------
bool ObjectNode::ProcessIncludesMSCL( const char * output, uint32_t outputSize, bool preprocessed )
{
	Timer t;

	ASSERT( output && outputSize );

	CIncludeParser parser;
	bool result = preprocessed ? parser.ParseMSCL_Preprocessed( output, outputSize )
							   : parser.ParseMSCL_Output( output, outputSize );
	if ( result == false )
	{
		FLOG_ERROR( "Failed to process includes for '%s'", GetName().Get() );
		return false;
	}

	// record that we have a list of includes
	// (we need a flag because we can't use the array size
	// as a determinator, because the file might not include anything)
	m_Includes = parser.GetIncludes();
	m_HasIncludes = true;

	FLOG_INFO( "Process Includes: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), m_Name.Get() );

	return true;
}

// ProcessIncludesGCC
//------------------------------------------------------------------------------
bool ObjectNode::ProcessIncludesGCC( const char * output, uint32_t outputSize )
{
	Timer t;

	ASSERT( output && outputSize );

	CIncludeParser parser;
	bool result = parser.ParseGCC_Preprocessed( output, outputSize );
	if ( result == false )
	{
		FLOG_ERROR( "Failed to process includes for '%s'", GetName().Get() );
		return false;
	}

	// record that we have a list of includes
	// (we need a flag because we can't use the array size
	// as a determinator, because the file might not include anything)
	m_Includes = parser.GetIncludes();
	m_HasIncludes = true;

	FLOG_INFO( "Process Includes: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), m_Name.Get() );

	return true;
}

// ProcessIncludesWithPreProcessor
//------------------------------------------------------------------------------
bool ObjectNode::ProcessIncludesWithPreProcessor( Job * job )
{
	Timer t;

	const char  * output = (char *)job->GetData();
	const size_t outputSize = job->GetDataSize();

	ASSERT( output && outputSize );

	CIncludeParser parser;
	bool result = GetFlag( FLAG_MSVC ) ? parser.ParseMSCL_Preprocessed( output, outputSize )
									   : parser.ParseGCC_Preprocessed( output, outputSize );
	if ( result == false )
	{
		FLOG_ERROR( "Failed to process includes for '%s'", GetName().Get() );
		return false;
	}

	// record that we have a list of includes
	// (we need a flag because we can't use the array size
	// as a determinator, because the file might not include anything)
	m_Includes = parser.GetIncludes();
	m_HasIncludes = true;

	FLOG_INFO( "Process Includes: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), m_Name.Get() );

	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * ObjectNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( uint64_t,		timeStamp );
	NODE_LOAD_DEPS( 3,			staticDeps );
	NODE_LOAD( bool,			hasIncludes );
	NODE_LOAD( Array< AString >,includes );
	NODE_LOAD( uint32_t,		flags );
	NODE_LOAD( AStackString<>,	compilerArgs );

	ASSERT( staticDeps.GetSize() >= 2 );
	Node * compiler = staticDeps[ 0 ];
	Node * staticDepNode = staticDeps[ 1 ];

	Node * precompiledHeader = nullptr;
	if ( staticDeps.GetSize() == 3 )
	{
		precompiledHeader = staticDeps[ 2 ];
	}

	Node * on = nullptr;
	if ( remote )
	{
		AStackString<> dummyName( "$$dummyName$$" );
		on = new ObjectNode( dummyName, staticDepNode, compiler, compilerArgs, precompiledHeader, flags );
	}
	else
	{
		NodeGraph & ng = FBuild::Get().GetDependencyGraph();
		on = ng.CreateObjectNode( name, staticDepNode, compiler, compilerArgs, precompiledHeader, flags );
	}

	ObjectNode * objNode = on->CastTo< ObjectNode >();
	objNode->m_HasIncludes = hasIncludes;
	objNode->m_Includes = includes;
	objNode->m_TimeStamp = timeStamp;

	return objNode;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool ObjectNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_TimeStamp );
	NODE_SAVE_DEPS( m_StaticDependencies );
	NODE_SAVE( m_HasIncludes );
	NODE_SAVE( m_Includes );
	NODE_SAVE( m_Flags );
	NODE_SAVE( m_CompilerArgs );
	return true;
}

// DumpOutput
//------------------------------------------------------------------------------
/*static*/ void ObjectNode::DumpOutput( const char * data, uint32_t dataSize )
{
	if ( ( data != nullptr ) && ( dataSize > 0 ) )
	{
		Array< AString > exclusions( 2, false );
		exclusions.Append( AString( "Note: including file:" ) );
		exclusions.Append( AString( "#line" ) );
		Node::DumpOutput( data, dataSize, &exclusions );
	}
}

// GetCacheName
//------------------------------------------------------------------------------
void ObjectNode::GetCacheName( const void * memOut, size_t memOutSize,
										  AString & cacheName ) const
{
	Timer t;

	// hash the pre-processed intput data
	uint32_t a = CRC32::Calc( memOut, memOutSize );

	// hash the build "environment"
	uint32_t b = CRC32::Start();
	b = CRC32::Update( b, m_CompilerArgs.Get(), m_CompilerArgs.GetLength() );
	// TODO:B Exclude preprocessor control defines (the preprocessed input has considered those already)
	// TODO:B - Include compiler version/compiler hash
	b = CRC32::Stop( b );

	FBuild::Get().GetCacheFileName( a, b, cacheName );

	FLOG_INFO( "Cache hash: %u ms - %u kb '%s'\n", 
				uint32_t( t.GetElapsed() * 1000.0f ), 
				uint32_t( memOutSize / KILOBYTE ), 
				cacheName.Get() );
}

// RetrieveFromCache
//------------------------------------------------------------------------------
bool ObjectNode::RetrieveFromCache( const AString & cacheFileName )
{
	if ( FBuild::Get().GetOptions().m_UseCacheRead == false )
	{
		return false;
	}

	Timer t;
	FileStream cacheFile;
	if ( cacheFile.Open( cacheFileName.Get(), FileStream::READ_ONLY ) )
	{
		const size_t cacheFileSize = (size_t)cacheFile.GetFileSize();
		AutoPtr< char > mem( new char[ cacheFileSize ] );
		if ( cacheFile.Read( mem.Get(), cacheFileSize ) == cacheFileSize )
		{
			// we expect a cache file header indicating compression status
			const uint32_t headerVal = *( (uint32_t *)mem.Get() );
			ASSERT( ( headerVal == 'OBJ0' ) || ( headerVal == 'OBJ1' ) );

			FileStream objFile;
			if ( !objFile.Open( m_Name.Get(), FileStream::WRITE_ONLY ) )
			{
				FLOG_ERROR( "Failed to open local file during cache retrieval '%s'", m_Name.Get() );
				return false;
			}

			// header contains compressed state
			const bool compressed = ( headerVal == 'OBJ1' );
			const void * data = ( mem.Get() + sizeof( uint32_t ) );
			size_t dataSize = cacheFileSize - sizeof( uint32_t );
			Compressor c;
			if ( compressed )
			{
				c.Decompress( data );
				data = c.GetResult();
				dataSize = c.GetResultSize();
			}

			if ( objFile.Write( data, dataSize ) != dataSize )
			{
				FLOG_ERROR( "Failed to write to local file during cache retrieval '%s'", m_Name.Get() );
				return false;
			}

			// Get current "system time" and convert to "file time"
			SYSTEMTIME st;
			FILETIME ft;
			GetSystemTime( &st );
			VERIFY( SystemTimeToFileTime( &st, &ft ) );
			uint64_t fileTimeNow = ( (uint64_t)ft.dwLowDateTime | ( (uint64_t)ft.dwHighDateTime << 32 ) );
	
			// set the time on the local file
			if ( objFile.SetLastWriteTime( fileTimeNow ) == false )
			{
				FLOG_ERROR( "Failed to set timestamp on file after cache hit '%s' (%u)", m_Name.Get(), GetLastError() );
			}

			objFile.Close();

			FileIO::WorkAroundForWindowsFilePermissionProblem( m_Name );

			// the file time we set and local file system might have different 
			// granularity for timekeeping, so we need to update with the actual time written
			m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

			FLOG_INFO( "Cache hit: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
			SetStatFlag( Node::STATS_CACHE_HIT );
			return true;
		}
	}

	FLOG_INFO( "Cache miss: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
	SetStatFlag( Node::STATS_CACHE_MISS );
	return false;
}

// WriteToCache
//------------------------------------------------------------------------------
void ObjectNode::WriteToCache( const AString & cacheFileName )
{
	ASSERT( !cacheFileName.IsEmpty() );

	Timer t;

	// make sure the cache output path exists
	AStackString<> cachePath( cacheFileName.Get(), cacheFileName.FindLast( '\\' ) );
	if ( !FileIO::EnsurePathExists( cachePath ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// open the compiled object which we want to store to the cache
	FileStream objFile;
	if ( !objFile.Open( m_Name.Get(), FileStream::READ_ONLY ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// open output cache (tmp) file
	AStackString<> cacheFileTmpName( cacheFileName );
	cacheFileTmpName += ".tmp";
	FileStream cacheTmpFile;
	if( !cacheTmpFile.Open( cacheFileTmpName.Get(), FileStream::WRITE_ONLY ) )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// read obj all into memory
	const size_t objFileSize = (size_t)objFile.GetFileSize();
	AutoPtr< char > mem( new char[ objFileSize ] );
	if ( objFile.Read( mem.Get(), objFileSize ) != objFileSize )
	{
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// try to compress
	Compressor c;
	const bool compressed = c.Compress( mem.Get(), objFileSize );
	const void * data = compressed ? c.GetResult() : mem.Get();
	const size_t dataSize = compressed ? c.GetResultSize() : objFileSize;

	// cache file contains a header to indicate compression
	const uint32_t headerVal = compressed ? 'OBJ1' : 'OBJ0';

	// write header and data
	bool cacheTmpWriteOk = cacheTmpFile.Write( headerVal ) && ( cacheTmpFile.Write( data, dataSize ) == dataSize );

	cacheTmpFile.Close();

	if ( !cacheTmpWriteOk )
	{
		// failed to write to cache tmp file
		FileIO::FileDelete( cacheFileTmpName.Get() ); // try to cleanup failure
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// rename tmp file to real file
	if ( FileIO::FileMove( cacheFileTmpName, cacheFileName ) == false )
	{
		// problem renaming file
		FileIO::FileDelete( cacheFileTmpName.Get() ); // try to cleanup tmp file
		FileIO::FileDelete( cacheFileName.Get() ); // clean up dst file in case it is corrupt
		FLOG_INFO( "Cache store fail: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
		return;
	}

	// cache store complete
	FLOG_INFO( "Cache store: %u ms '%s'\n", uint32_t( t.GetElapsed() * 1000.0f ), cacheFileName.Get() );
	SetStatFlag( Node::STATS_CACHE_STORE );
}

// EmitCompilationMessage
//------------------------------------------------------------------------------
void ObjectNode::EmitCompilationMessage( const AString & fullArgs ) const
{
	// print basic or detailed output, depending on options
	// we combine everything into one string to ensure it is contiguous in
	// the output
	AStackString<> output;
	output += "Obj: ";
	//FileNode * sourceFile = m_StaticDependencies[ 1 ]->CastTo< FileNode >();
	//output += sourceFile->GetName().FindLast( '\\' ) + 1;
	//output += " (";
	output += GetName();
	//output += ")\n";
	output += '\n';
	if ( FLog::ShowInfo() )
	{
		output += m_StaticDependencies[ 0 ]->GetName();
		output += ' ';
		output += fullArgs;
		output += '\n';
	}
	FLOG_BUILD( "%s", output.Get() );
}

// BuildFullArgs
//------------------------------------------------------------------------------
void ObjectNode::BuildFullArgs( AString & fullArgs, bool preProcess ) const
{
	// Format compiler args string
	fullArgs = m_CompilerArgs;
	if ( preProcess )
	{
		if ( GetFlag( FLAG_MSVC ) )
		{
			fullArgs += " /E"; // run pre-processor only
		}
		else
		{
			ASSERT( GetFlag( FLAG_GCC ) );
			fullArgs += " -E"; // run pre-processor only
		}
	}

	Node * sourceFile = m_StaticDependencies[ 1 ];

	// substitute "function variables" if needed
	fullArgs.Replace( "%1", sourceFile->GetName().Get() );
	fullArgs.Replace( "%2", m_Name.Get() );
}

// BuildPreprocessedOutput
//------------------------------------------------------------------------------
bool ObjectNode::BuildPreprocessedOutput( const AString & fullArgs, Job * job ) const
{
	FileNode * compiler = m_StaticDependencies[ 0 ]->CastTo< FileNode >();
//	FileNode * sourceFile = m_StaticDependencies[ 1 ]->CastTo< FileNode >();

	EmitCompilationMessage( fullArgs );

	// spawn the process
	Process p;
	if ( false == p.Spawn( compiler->GetName().Get(), 
						   fullArgs.Get(),
						   nullptr, // use exe launch dir as the working dir
						   FBuild::Get().GetEnvironmentString() ) )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", GetName().Get() );
		return false;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memErr, memOut;
	uint32_t memErrSize = 0;
	uint32_t memOutSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	// Get result
	ASSERT( !p.IsRunning() );
	int result = p.WaitForExit();

	if ( result != 0 )
	{
		DumpOutput( memErr.Get(), memErrSize );
		FLOG_ERROR( "Failed to build Object (error %i) '%s'", result, GetName().Get() );
		return false;
	}

	// take a copy of the output because ReadAllData uses huge buffers to avoid re-sizing
	char * memCopy = new char[ memOutSize ];
	memcpy( memCopy, memOut.Get(), memOutSize );

	job->OwnData( memCopy, memOutSize );

	return true;
}

// WriteTmpFile
//------------------------------------------------------------------------------
bool ObjectNode::WriteTmpFile( Job * job, AString & tmpFileName ) const
{
	ASSERT( job->GetData() && job->GetDataSize() );

	Node * sourceFile = m_StaticDependencies[ 1 ];

	FileStream tmpFile;
	const char * tmpFileExt = sourceFile->GetName().FindLast( '.' ) + 1;

	if ( GetFlag( FLAG_GCC ) == true )
	{
		// GCC requires preprocessed output to be named a certain way
		tmpFileExt = ( AString::StrNCmpI( tmpFileExt, "c", 1 ) == 0 ) ? "i" : "ii";
	}

	void const * dataToWrite = job->GetData();
	size_t dataToWriteSize = job->GetDataSize();

	// handle compressed data
	Compressor c; // scoped here so we can access decompression buffer
	if ( job->IsDataCompressed() )
	{
		c.Decompress( dataToWrite );
		dataToWrite = c.GetResult();
		dataToWriteSize = c.GetResultSize();
	}

	WorkerThread::CreateTempFilePath( tmpFileExt, tmpFileName );
	if ( ( WorkerThread::CreateTempFile( tmpFileName, tmpFile ) == false ) ||
			( tmpFile.Write( dataToWrite, dataToWriteSize ) == false ) )
	{
		int error = GetLastError();
		FLOG_ERROR( "Failed to write to temp '%s' file to build '%s' (error %u)", tmpFileName.Get(), GetName().Get(), error );
		return NODE_RESULT_FAILED;
	}
	tmpFile.Close();

	FileIO::WorkAroundForWindowsFilePermissionProblem( tmpFileName );

	return true;
}

// BuildFinalOutput
//------------------------------------------------------------------------------
bool ObjectNode::BuildFinalOutput( Job * job, const AString & fullArgs ) const
{
	Node * compiler = m_StaticDependencies[ 0 ];

	// spawn the process
	Process p;
	if ( false == p.Spawn( compiler->GetName().Get(), 
						   fullArgs.Get(),
						   nullptr, // workingDir - use the exe launch dir
						   FBuild::IsValid() ? FBuild::Get().GetEnvironmentString() : nullptr ) )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", GetName().Get() );
		return false;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memErr, memOut;
	uint32_t memErrSize = 0;
	uint32_t memOutSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	// Get result
	ASSERT( !p.IsRunning() );
	int result = p.WaitForExit();

	if ( memErr.Get() )
	{ 
		DumpOutput( memErr.Get(), memErrSize );
	}

	if ( result != 0 )
	{
		DumpOutput( memOut.Get(), memOutSize );
		FLOG_ERROR( "Failed to build Object (error %i) '%s'", result, GetName().Get() );

		// for remote jobs, we must serialize the errors to return with the job
		if ( job->IsLocal() == false )
		{
			AutoPtr< char > mem( new char[ memOutSize + memErrSize ] );
			memcpy( mem.Get(), memOut.Get(), memOutSize );
			memcpy( mem.Get() + memOutSize, memErr.Get(), memErrSize );
			job->OwnData( mem.Release(), ( memOutSize + memErrSize ) );
		}

		return false;
	}

	return true;
}

//------------------------------------------------------------------------------
