// NodeGraph.h - interface to the dependency graph
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_GRAPH_NODEGRAPH_H
#define FBUILD_GRAPH_NODEGRAPH_H

// Includes
//------------------------------------------------------------------------------
#include <Core/Containers/Array.h>
#include <Core/Strings/AString.h>
#include <Core/Time/Timer.h>

// Forward Declaration
//------------------------------------------------------------------------------
class AliasNode;
class AString;
class CopyNode;
class CSNode;
class DirectoryListNode;
class ExecNode;
class FileNode;
class IOStream;
class LibraryNode;
class LinkerNode;
class ObjectNode;
class Node;
class TestNode;
class UnityNode;

// NodeGraphHeader
//------------------------------------------------------------------------------
class NodeGraphHeader
{
public:
	inline explicit NodeGraphHeader() {}
	inline explicit NodeGraphHeader( uint32_t crc )
	{
		m_Identifier[ 0 ] = 'N';
		m_Identifier[ 1 ] = 'G';
		m_Identifier[ 2 ] = 'D';
		m_Version = NODE_GRAPH_CURRENT_VERSION;
		m_CRC = crc;
	}
	inline ~NodeGraphHeader() {}

	enum { NODE_GRAPH_CURRENT_VERSION = 2 };

	bool IsValid( uint32_t crc ) const
	{
		return ( ( m_Identifier[ 0 ] == 'N' ) &&
				 ( m_Identifier[ 1 ] == 'G' ) &&
				 ( m_Identifier[ 2 ] == 'D' ) &&
				 ( m_Version == NODE_GRAPH_CURRENT_VERSION ) &&
				 ( m_CRC == crc ) );
	}
private:
	char		m_Identifier[ 3 ];
	uint8_t		m_Version;
	uint32_t	m_CRC;
};

// NodeGraph
//------------------------------------------------------------------------------
class NodeGraph
{
public:
	explicit NodeGraph();
	~NodeGraph();

	bool Initialize( const char * bffFile, const char * nodeGraphDBFile );

	bool Load( IOStream & stream );
	bool Save( IOStream & stream ) const;

	// access existing nodes
	Node * FindNode( const AString & nodeName ) const;

	// create new nodes
	CopyNode * CreateCopyNode( const AString & dstFileName, 
							   Node * sourceFile );
	ExecNode * CreateExecNode( const AString & dstFileName, 
							   FileNode * sourceFile, 
							   FileNode * executable, 
							   const char * arguments, 
							   const char * workingDir );
	FileNode * CreateFileNode( const AString & fileName, bool cleanPath = true );
	DirectoryListNode * CreateDirectoryListNode( const AString & name,
												 const AString & path,
												 const AString & wildCard,
												 bool recursive,
												 const AString & excludePath );
	LibraryNode *	CreateLibraryNode( const AString & libraryName,
									   Array< Node * > & inputNode,
									   FileNode * compilerNode,
									   const AString & compilerArgs,
									   const AString & compilerOutputPath,
									   const AString & linker,
									   const AString & linkerArgs,
									   ObjectNode * precompiledHeader );
	ObjectNode *	CreateObjectNode( const AString & objectName,
									  Node * inputNode,
									  Node * compilerNode,
									  const AString & compilerArgs,
									  Node * precompiledHeader,
									  uint32_t flags );
	AliasNode *		CreateAliasNode( const AString & aliasName,
									 const Array< Node * > & targets );
	LinkerNode *	CreateLinkerNode( const AString & linkerOutputName,
									  const Array< Node * > & inputLibraries,
									  const AString & linker,
									  const AString & linkerArgs );
	UnityNode *	CreateUnityNode( const AString & unityName,
								 const Array< DirectoryListNode * > & dirNodes,
								 const AString & outputPath,
								 const AString & outputPattern,
								 uint32_t numUnityFilesToCreate,
								 const AString & precompiledHeader,
								 const Array< AString > & filesToExclude );

	CSNode * CreateCSNode( const AString & compilerOutput,
						   const Array< Node * > & inputNodes,
						   const AString & compiler,
						   const AString & compilerOptions,
						   const Array< Node * > & extraRefs );
	TestNode * CreateTestNode( const AString & testOutput,
							   FileNode * testExecutable,
							   const AString & arguments,
							   const AString & workingDir );

	void DoBuildPass( Node * nodeToBuild );

	static void CleanPath( const AString & name, AString & fullPath );

private:
	friend class FBuild;

	void AddNode( Node * node );

	void BuildRecurse( Node * nodeToBuild );
	void UpdateBuildStatusRecurse( const Node * node, uint32_t & nodesBuiltTime, 
													  uint32_t & totalNodeTime,
													  uint32_t & totalNodes ) const;

	Node * FindNodeInternal( const AString & fullPath ) const;

	enum { NODEMAP_TABLE_SIZE = 65536 };
	Node *			m_NodeMap[ NODEMAP_TABLE_SIZE ];
	Array< Node * > m_AllNodes;

	uint32_t m_BFFCRC; // CRC of config used to generate the Dependency Graph

	Timer m_Timer;
};

//------------------------------------------------------------------------------
#endif // FBUILD_GRAPH_NODEGRAPH_H
