// Node.cpp - base interface for dependency graph nodes
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "NodeGraph.h"

#include <Tools/FBuild/FBuildCore/BFF/BFFParser.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/WorkerPool/JobQueue.h>

#include "AliasNode.h"
#include "CopyNode.h"
#include "CSNode.h"
#include "DirectoryListNode.h"
#include "ExecNode.h"
#include "FileNode.h"
#include "LibraryNode.h"
#include "LinkerNode.h"
#include "ObjectNode.h"
#include "TestNode.h"
#include "UnityNode.h"

#include <Core/Containers/AutoPtr.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Math/CRC32.h>
#include <Core/Process/Thread.h>
#include <Core/Strings/AStackString.h>
#include <Core/Tracing/Tracing.h>

#include <Windows.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
NodeGraph::NodeGraph()
: m_AllNodes( 1024, true )
{
	memset( m_NodeMap, 0, sizeof( Node * ) * NODEMAP_TABLE_SIZE );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
NodeGraph::~NodeGraph()
{
	Array< Node * >::Iter i = m_AllNodes.Begin();
	Array< Node * >::Iter end = m_AllNodes.End();
	for ( ; i != end; ++i )
	{
		delete ( *i );
	}
}

// Setup
//------------------------------------------------------------------------------
bool NodeGraph::Initialize( const char * bffFile,
							const char * nodeGraphDBFile )
{
	ASSERT( bffFile ); // must be supplied (or left as default)
	//ASSERT( nodeGraphDBFile ); // allowed to be nullptr (to force ignore it)

	// open the configuration file
	FLOG_INFO( "Loading BFF '%s'", bffFile );
	FileStream bffStream;
	if ( bffStream.Open( bffFile ) == false )
	{
		// missing bff is a fatal problem
		FLOG_ERROR( "Failed to open BFF '%s'", bffFile );
		return false;
	}

	// read entire config into memory
	uint32_t size = (uint32_t)bffStream.GetFileSize();
	AutoPtr< char > data( new char[ size + 1 ] ); // extra byte for null character sentinel
	if ( bffStream.Read( data.Get(), size ) != size )
	{
		FLOG_ERROR( "Error reading BFF '%s'", bffFile );
		return false;
	}
	m_BFFCRC = CRC32::Calc( data.Get(), size ); // CRC does not include null char

	// try to open the file dependency graph
	bool useNodeGraph = false;
	FileStream nodeGraphStream;
	if ( nodeGraphDBFile )
	{
		if ( nodeGraphStream.Open( nodeGraphDBFile, FileStream::READ_ONLY ) == true )
		{
			// there is a dep graph, was it generated from the same bff?
			NodeGraphHeader ngh;
			if ( nodeGraphStream.Read( &ngh, sizeof( ngh ) ) == sizeof( ngh ) )
			{
				if ( ngh.IsValid( m_BFFCRC ) )
				{
					FLOG_INFO( "NodeGraph db and BFF match" );
					useNodeGraph = true;
				}
			}
		}
		else
		{
			// BFF and NodeGraph db are out of sync, we can't use the DB
			// - BFF has changed (most likley)
			// - DB corrupt (unlikely, but possible)
			FLOG_INFO( "NodeGraph db and BFF are out of sync, build will be clean" );
		}
	}

	bool result;
	if ( useNodeGraph )
	{
		// we can use the node graph
		result = Load( nodeGraphStream );
		if ( result == false )
		{
			FLOG_ERROR( "Error reading BFF '%s' (corrupt?)", nodeGraphDBFile );
		}
	}
	else
	{
		// re-parse the BFF from scratch, clean build will result
		BFFParser bffParser;
		data.Get()[ size ] = '\0'; // data passed to parser must be NULL terminated
		result = bffParser.Parse( data.Get(), size, bffFile ); // pass size excluding sentinel
	}

	return result;
}

// Load
//------------------------------------------------------------------------------
bool NodeGraph::Load( IOStream & stream )
{
	// we expect to be past the header now
	ASSERT( stream.Tell() == sizeof( NodeGraphHeader ) );

	ASSERT( m_AllNodes.GetSize() == 0 );

	// Read nodes
	uint32_t numNodes;
	if ( stream.Read( numNodes ) == false )
	{
		return false;
	}
	if ( m_AllNodes.GetCapacity() < numNodes )
	{
		m_AllNodes.SetCapacity( numNodes );
	}
	for ( uint32_t i=0; i<numNodes; ++i )
	{
		Node * n = Node::Load( stream, false );
		if ( n == nullptr )
		{
			return false;
		}
		uint32_t lastTimeToBuild;
		if ( stream.Read( lastTimeToBuild ) == false )
		{
			return false;
		}
		n->SetLastBuildTime( lastTimeToBuild );
	}

	// TODO:C The serialization of these settings doesn't really belong here (not part of node graph)
	{
		// cachepath
		AStackString<> cachePath;
		if ( stream.Read( cachePath ) == false )
		{
			return false;
		}
		FBuild::Get().SetCachePath( cachePath );

		// environment
		uint32_t envStringSize = 0;
		if ( stream.Read( envStringSize ) == false )
		{
			return false;
		}
		if ( envStringSize > 0 )
		{
			AutoPtr< char > envString( new char[ envStringSize ] );
			if ( stream.Read( envString.Get(), envStringSize ) == false )
			{
				return false;
			}
			FBuild::Get().SetEnvironmentString( envString.Get(), envStringSize );
		}

		// worker list
		Array< AString > workerList( 0, true );
		if ( stream.Read( workerList ) == false )
		{
			return false;
		}
		FBuild::Get().SetWorkerList( workerList );
	}

	return true;
}

// Save
//------------------------------------------------------------------------------
bool NodeGraph::Save( IOStream & stream ) const
{
	// write header and version
	NodeGraphHeader header( m_BFFCRC );
	if ( stream.Write( (const void *)&header, sizeof( header ) ) == false )
	{
		return false;
	}

	// Write nodes
	size_t numNodes = m_AllNodes.GetSize();
	if ( stream.Write( (uint32_t)numNodes ) == false )
	{
		return false;
	}
	for ( size_t i=0; i<numNodes; ++i )
	{
		const Node * n = m_AllNodes[ i ];
		if ( Node::Save( stream, n ) == false )
		{
			return false;
		}
		uint32_t lastBuildTime = n->GetLastBuildTime();
		if ( stream.Write( lastBuildTime ) == false )
		{
			return false;
		}
	}

	// TODO:C The serialization of these settings doesn't really belong here (not part of node graph)
	{
		// cachpath
		if ( stream.Write( FBuild::Get().GetCachePath() ) == false )
		{
			return false;
		}

		// environment
		const uint32_t envStringSize = FBuild::Get().GetEnvironmentStringSize();
		if ( stream.Write( envStringSize ) == false )
		{
			return false;
		}
		if ( envStringSize > 0 )
		{
			const char * envString = FBuild::Get().GetEnvironmentString();
			if (  stream.Write( envString, envStringSize ) == false )
			{
				return false;
			}
		}

		// worker list
		const Array< AString > & workerList = FBuild::Get().GetWorkerList();
		if ( stream.Write( workerList ) == false )
		{
			return false;
		}
	}

	return true;
}

// FindNode (AString &)
//------------------------------------------------------------------------------
Node * NodeGraph::FindNode( const AString & nodeName ) const
{
	// try to find node 'as is'
	Node * n = FindNodeInternal( nodeName );
	if ( n )
	{
		return n;
	}

	// the expanding to a full path
	AStackString< 1024 > fullPath;
	CleanPath( nodeName, fullPath );
	return FindNodeInternal( fullPath );
}

// CreateCopyNode
//------------------------------------------------------------------------------
CopyNode * NodeGraph::CreateCopyNode( const AString & dstFileName, Node * sourceFile )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( sourceFile->IsAFile() );

	AStackString< 1024 > fullPathDst;
	CleanPath( dstFileName, fullPathDst );

	CopyNode * node = new CopyNode( fullPathDst, (FileNode *)sourceFile );
	AddNode( node );
	return node;
}

// CreateCopyNode
//------------------------------------------------------------------------------
ExecNode * NodeGraph::CreateExecNode( const AString & dstFileName, 
									  FileNode * sourceFile, 
									  FileNode * executable, 
									  const char * arguments,									  
									  const char * workingDir )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 512 > fullPath;
	CleanPath( dstFileName, fullPath );

	ExecNode * node = new ExecNode( fullPath, sourceFile, executable, arguments, workingDir );
	AddNode( node );
	return node;
}

// CreateFileNode
//------------------------------------------------------------------------------
FileNode * NodeGraph::CreateFileNode( const AString & fileName, bool cleanPath )
{
	ASSERT( Thread::IsMainThread() );

	FileNode * node;

	if ( cleanPath )
	{
		AStackString< 512 > fullPath;
		CleanPath( fileName, fullPath );
		node = new FileNode( fullPath );
	}
	else
	{
		node = new FileNode( fileName );
	}

	AddNode( node );
	return node;
}

// CreateDirectoryListNode
//------------------------------------------------------------------------------
DirectoryListNode * NodeGraph::CreateDirectoryListNode( const AString & name,
													    const AString & path,
													    const AString & wildCard,
													    bool recursive,
													    const AString & excludePath )
{
	ASSERT( Thread::IsMainThread() );

	// NOTE: DirectoryListNode assumes valid values from here
	// and will assert as such (so we don't check here)

	DirectoryListNode * node = new DirectoryListNode( name, path, wildCard, recursive, excludePath );
	AddNode( node );
	return node;
}

// CreateLibraryNode
//------------------------------------------------------------------------------
LibraryNode * NodeGraph::CreateLibraryNode( const AString & libraryName,
										    Array< Node * > & inputNodes,
											FileNode * compiler,
											const AString & compilerArgs,
											const AString & compilerOutputPath,
											const AString & linker,
											const AString & linkerArgs,
											ObjectNode * precompiledHeader )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( libraryName, fullPath );

	LibraryNode * node = new LibraryNode( fullPath, inputNodes, compiler, compilerArgs, compilerOutputPath, linker, linkerArgs, precompiledHeader );
	AddNode( node );
	return node;
}

// CreateObjectNode
//------------------------------------------------------------------------------
ObjectNode * NodeGraph::CreateObjectNode( const AString & objectName,
										  Node * inputNode,
										  Node * compilerNode,
										  const AString & compilerArgs,
										  Node * precompiledHeader,
										  uint32_t flags )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 512 > fullPath;
	CleanPath( objectName, fullPath );

	ObjectNode * node = new ObjectNode( fullPath, inputNode, compilerNode, compilerArgs, precompiledHeader, flags );
	AddNode( node );
	return node;
}

// CreateAliasNode
//------------------------------------------------------------------------------
AliasNode * NodeGraph::CreateAliasNode( const AString & aliasName,
										const Array< Node * > & targets )
{
	ASSERT( Thread::IsMainThread() );

	AliasNode * node = new AliasNode( aliasName, targets );
	AddNode( node );
	return node;
}

// CreateLinkerNode
//------------------------------------------------------------------------------
LinkerNode * NodeGraph::CreateLinkerNode( const AString & linkerOutputName,
										  const Array< Node * > & inputLibraries,
										  const AString & linker,
										  const AString & linkerArgs )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( inputLibraries.IsEmpty() == false );

	AStackString< 1024 > fullPath;
	CleanPath( linkerOutputName, fullPath );

	LinkerNode * node = new LinkerNode( fullPath, 
										inputLibraries,
										linker,
										linkerArgs );
	AddNode( node );
	return node;
}

// CreateUnityNode
//------------------------------------------------------------------------------
UnityNode *	NodeGraph::CreateUnityNode( const AString & unityName,
										const Array< DirectoryListNode * > & dirNodes,
										const AString & outputPath,
										const AString & outputPattern,
										uint32_t numUnityFilesToCreate,
										const AString & precompiledHeader,
										const Array< AString > & filesToExclude )
{
	ASSERT( Thread::IsMainThread() );

	UnityNode * node = new UnityNode( unityName,
									  dirNodes,
									  outputPath,
									  outputPattern,
									  numUnityFilesToCreate,
									  precompiledHeader,
									  filesToExclude );
	AddNode( node );
	return node;
}

CSNode * NodeGraph::CreateCSNode( const AString & compilerOutput,
								  const Array< Node * > & inputNodes,
								  const AString & compiler,
								  const AString & compilerOptions,
								  const Array< Node * > & extraRefs )
{
	ASSERT( Thread::IsMainThread() );
	ASSERT( inputNodes.IsEmpty() == false );

	AStackString< 1024 > fullPath;
	CleanPath( compilerOutput, fullPath );

	CSNode * node = new CSNode( fullPath,
								inputNodes,
								compiler,
								compilerOptions,
								extraRefs );
	AddNode( node );
	return node;
}

// CreateTestNode
//------------------------------------------------------------------------------
TestNode * NodeGraph::CreateTestNode( const AString & testOutput,
									  FileNode * testExecutable,
									  const AString & arguments,
									  const AString & workingDir )
{
	ASSERT( Thread::IsMainThread() );

	AStackString< 1024 > fullPath;
	CleanPath( testOutput, fullPath );

	TestNode * node = new TestNode( fullPath,
									testExecutable,
									arguments,
									workingDir );
	AddNode( node );
	return node;
}

// AddNode
//------------------------------------------------------------------------------
void NodeGraph::AddNode( Node * node )
{
	ASSERT( Thread::IsMainThread() );

	ASSERT( node );

	ASSERT( FindNodeInternal( node->GetName() ) == nullptr ); // node name must be unique

	// track in NodeMap
	const uint32_t crc = CRC32::CalcLower( node->GetName() );
	const size_t key = ( crc & 0xFFFF );
	node->m_Next = m_NodeMap[ key ];
	m_NodeMap[ key ] = node;

	// add to regular list
	m_AllNodes.Append( node );
}

// Build
//------------------------------------------------------------------------------
void NodeGraph::DoBuildPass( Node * nodeToBuild )
{
	// do a build pass
	BuildRecurse( nodeToBuild );
}

// BuildRecurse
//------------------------------------------------------------------------------
void NodeGraph::BuildRecurse( Node * nodeToBuild )
{
	ASSERT( nodeToBuild );

	// already building, or queued to build?
	if ( nodeToBuild->GetState() == Node::BUILDING )
	{
		return;
	}

	bool allDependenciesUpToDate = true;

	// test static dependencies first
	if ( nodeToBuild->GetState() == Node::NOT_PROCESSED )
	{
		const Array< Node * > & staticDeps = nodeToBuild->GetStaticDependencies();
		Array< Node * >::Iter i = staticDeps.Begin();
		Array< Node * >::Iter end = staticDeps.End();
		for ( ; i < end; ++i )
		{
			Node::State state = ( *i )->GetState();

			// recurse into nodes which have not been processed yet
			if ( state == Node::NOT_PROCESSED ||
				 state == Node::STATIC_DEPS_READY ||
				 state == Node::DYNAMIC_DEPS_DONE )
			{
				BuildRecurse( *i );
			}

			// dependency is uptodate, nothing more to be done
			state = ( *i )->GetState();
			if ( state == Node::UP_TO_DATE )
			{
				continue;
			}

			allDependenciesUpToDate = false;

			// dependency already building?
			if ( state == Node::BUILDING )
			{
				continue; // nothing more to be done
			}

			// dependency failed?
			if ( state == Node::FAILED )
			{
				// propogate failure state to this node
				nodeToBuild->SetState( Node::FAILED );
				return;
			}
		}

		// all static deps done?
		if ( allDependenciesUpToDate == false )
		{
			return; // no - have to wait some more
		}

		nodeToBuild->SetState( Node::STATIC_DEPS_READY );
	}

	ASSERT( ( nodeToBuild->GetState() == Node::STATIC_DEPS_READY ) ||
			( nodeToBuild->GetState() == Node::DYNAMIC_DEPS_DONE ) );

	if ( nodeToBuild->GetState() != Node::DYNAMIC_DEPS_DONE )
	{
		// static deps ready, update dynamic deps
		bool forceClean = FBuild::Get().GetOptions().m_ForceCleanBuild;
		if ( nodeToBuild->DoDynamicDependencies( forceClean ) == false )
		{
			nodeToBuild->SetState( Node::FAILED );
			return;		
		}

		nodeToBuild->SetState( Node::DYNAMIC_DEPS_DONE );
	}

	ASSERT( nodeToBuild->GetState() == Node::DYNAMIC_DEPS_DONE );

	// dynamic deps
	{
		const Array< Node * > & dynDeps = nodeToBuild->GetDynamicDependencies();
		Array< Node * >::Iter i = dynDeps.Begin();
		Array< Node * >::Iter end = dynDeps.End();
		for ( ; i < end; ++i )
		{
			Node::State state = ( *i )->GetState();

			// recurse into nodes which have not been processed yet
			if ( ( state == Node::NOT_PROCESSED ) ||
				 ( state == Node::STATIC_DEPS_READY ) ||
				 ( state == Node::DYNAMIC_DEPS_DONE ) )
			{
				BuildRecurse( *i );
			}

			// dependency is uptodate, nothing more to be done
			state = ( *i )->GetState();
			if ( state == Node::UP_TO_DATE )
			{
				continue;
			}

			allDependenciesUpToDate = false;

			// dependency already building?
			if ( state == Node::BUILDING )
			{
				continue; // nothing more to be done
			}

			// dependency failed?
			if ( state == Node::FAILED )
			{
				// propogate failure state to this node
				nodeToBuild->SetState( Node::FAILED );
				return;
			}
		}
	}

	// are all deps uptodate so we could build?
	if ( allDependenciesUpToDate == false )
	{
		return; // can't build this node yet
	}

	// dependencies are uptodate, so node can now tell us if it needs
	// building
	bool forceClean = FBuild::Get().GetOptions().m_ForceCleanBuild;
	nodeToBuild->SetStatFlag( Node::STATS_PROCESSED );
	if ( nodeToBuild->DetermineNeedToBuild( forceClean ) )
	{
		JobQueue::Get().QueueJob( nodeToBuild );
	}
	else
	{
		nodeToBuild->SetState( Node::UP_TO_DATE );
	}
}

// CleanPath
//------------------------------------------------------------------------------
/*static*/ void NodeGraph::CleanPath( const AString & name, AString & fullPath )
{
	ASSERT( &name != &fullPath );

	char * dst;

	//  - path should be fully qualified
	bool isFullPath = ( ( name.GetLength() > 1 ) && ( name[1] == ':' ) );
	if ( !isFullPath )
	{
		// make a full path by prepending working dir
		const AString & workingDir = FBuild::Get().GetWorkingDir();

		// we're making the assumption that we don't need to clean the workingDir
		ASSERT( workingDir.Find( '/' ) == nullptr );
		ASSERT( workingDir.Find( "\\\\" ) == nullptr );

		// build the start of the path
		fullPath = workingDir;
		fullPath += '\\';

		// concatenate
		uint32_t len = fullPath.GetLength();

		// make sure the dest will be big enough for the extra stuff
		fullPath.SetLength( fullPath.GetLength() + name.GetLength() ); 

		// set the output (which maybe a newly allocated ptr)
		dst = fullPath.Get() + len;
	}
	else
	{
		// make sure the dest will be big enough
		fullPath.SetLength( name.GetLength() ); 

		// copy from the start
		dst = fullPath.Get();
	}

	// the untrusted part of the path we need to copy/fix
	const char * src = name.Get();
	const char * const srcEnd = name.GetEnd();

	//  - TODO:C would be nice to handle "\..\" 

	// clean slashes
	while ( src < srcEnd )
	{
		const char thisChar = *src;

		// hit a slash?
		if ( ( thisChar == '\\' ) || ( thisChar == '/' ) )
		{
			// write it the correct way
			*dst = '\\';
			dst++;

			// skip until non-slashes
			while ( ( *src == '\\' ) || ( *src == '/' ) )
			{
				src++;
			}
			continue;
		}

		// write non-slash character
		*dst++ = *src++;
	}

	// correct length of destination
	fullPath.SetLength( (uint16_t)( dst - fullPath.Get() ) );
	ASSERT( AString::StrLen( fullPath.Get() ) == fullPath.GetLength() );

	// sanity checks
	ASSERT( fullPath.Find( '/' ) == nullptr );
	ASSERT( fullPath.Find( "\\\\" ) == nullptr );
}

//------------------------------------------------------------------------------
Node * NodeGraph::FindNodeInternal( const AString & fullPath ) const
{
	ASSERT( Thread::IsMainThread() );

	const uint32_t crc = CRC32::CalcLower( fullPath );
	const size_t key = ( crc & 0xFFFF );

	Node * n = m_NodeMap[ key ];
	while ( n )
	{
		if ( n->GetNameCRC() == crc )
		{
			if ( n->GetName().CompareI( fullPath ) == 0 )
			{
				return n;
			}
		}
		n = n->m_Next;
	}
	return nullptr;
}

// UpdateBuildStatusRecurse
//------------------------------------------------------------------------------
void NodeGraph::UpdateBuildStatusRecurse( const Node * node, uint32_t & nodesBuiltTime, 
															 uint32_t & totalNodeTime,
															 uint32_t & totalNodes ) const
{
	totalNodeTime += node->GetLastBuildTime();

	if ( node->GetState() == Node::UP_TO_DATE )
	{
		nodesBuiltTime += node->GetLastBuildTime();
	}

	const Array< Node * > & staticDeps = node->GetStaticDependencies();
	totalNodes += (uint32_t)staticDeps.GetSize();
	for ( Array< Node * >::Iter i = staticDeps.Begin();
		i != staticDeps.End();
		i++ )
	{
		UpdateBuildStatusRecurse( *i, nodesBuiltTime, totalNodeTime, totalNodes );
	}

	const Array< Node * > & dynamicDeps = node->GetDynamicDependencies();
	totalNodes += (uint32_t)dynamicDeps.GetSize();
	for ( Array< Node * >::Iter i = dynamicDeps.Begin();
		i != dynamicDeps.End();
		i++ )
	{
		UpdateBuildStatusRecurse( *i, nodesBuiltTime, totalNodeTime, totalNodes );
	}
}

//------------------------------------------------------------------------------
