// Node.h - base interface for dependency graph nodes
//------------------------------------------------------------------------------
#pragma once
#ifndef FBUILD_GRAPH_NODE_H
#define FBUILD_GRAPH_NODE_H

// Includes
//------------------------------------------------------------------------------
#include <Core/Containers/Array.h>
#include <Core/Strings/AString.h>

class IOStream;
class FileNode;
class Job;

// Load/SaveMacros
//------------------------------------------------------------------------------
#define NODE_SAVE( member ) if ( stream.Write( member ) == false ) { return false; }
#define NODE_SAVE_DEPS( depsArray )	if ( Node::SaveDepArray( stream, depsArray ) == false ) { return false; }
#define NODE_SAVE_NODE( node ) if ( Node::SaveNode( stream, node ) == false ) { return false; }

#define NODE_LOAD( type, member ) (void)remote; type member; if ( stream.Read( member ) == false ) { return nullptr; }
#define NODE_LOAD_DEPS( initialCapacity, depsArray ) \
	Array< Node * > depsArray( initialCapacity, true ); \
	if ( Node::LoadDepArray( stream, depsArray, remote ) == false ) { return nullptr; }
#define NODE_LOAD_NODE( type, node ) \
	type * node = nullptr; \
	if ( Node::LoadNode( stream, node ) == false ) { return nullptr; }

// FBuild
//------------------------------------------------------------------------------
class Node
{
public:
	enum Type
	{
		PROXY_NODE			= 0,
		COPY_NODE			= 1,
		DIRECTORY_LIST_NODE	= 2,
		EXEC_NODE			= 3,
		FILE_NODE			= 4,
		LIBRARY_NODE		= 5,
		OBJECT_NODE			= 6,
		ALIAS_NODE			= 7,
		LINKER_NODE			= 8,
		UNITY_NODE			= 9,
		CS_NODE				= 10,
		TEST_NODE			= 11,
		// Make sure you update 's_NodeTypeNames' in the cpp
		NUM_NODE_TYPES		// leave this last
	};

	enum ControlFlag
	{
		FLAG_NONE					= 0x00,
		FLAG_TRIVIAL_BUILD			= 0x01,	// DoBuild is performed locally in main thread
		FLAG_NO_DELETE_ON_FAIL		= 0x02, // Don't delete output file on failure (for Test etc)
	};

	enum StatsFlag
	{
		STATS_PROCESSED		= 0x01, // node was processed during the build
		STATS_BUILT			= 0x02,	// node needed building, and was built
		STATS_CACHE_HIT		= 0x04, // needed building, was cacheable & was retrieved from the cache
		STATS_CACHE_MISS	= 0x08, // needed building, was cacheable, but wasn't in cache
		STATS_CACHE_STORE	= 0x10, // needed building, was cacheable & was stored to the cache
		STATS_STATS_PROCESSED	= 0x8000 // mark during stats gathering (leave this last)
	};

	enum BuildResult
	{
		NODE_RESULT_FAILED		= 0,	// something went wrong building
		NODE_RESULT_NEED_SECOND_BUILD_PASS,	// needs build called again
		NODE_RESULT_OK,					// built ok
		NODE_RESULT_OK_CACHE			// retrieved from the cache
	};

	enum State
	{
		NOT_PROCESSED,		// no work done (either not part of this build, or waiting on static dependencies )
		STATIC_DEPS_READY,	// static dependencies are uptodate - we are ready to DoDynamicDeps
		DYNAMIC_DEPS_DONE,	// dynamic deps updated, waiting for dynamic deps to be ready
		BUILDING,			// in the queue for building
		FAILED,				// failed to build
		UP_TO_DATE,			// built, or confirmed as not needing building
	};

	explicit Node( const AString & name, Type type, uint32_t controlFlags );
	virtual ~Node();

	inline const AString & GetName() const { return m_Name; }
	inline uint32_t 	   GetNameCRC() const { return m_NameCRC; }
	inline Type GetType() const { return m_Type; }
	inline const char * GetTypeName() const { return s_NodeTypeNames[ m_Type ]; }
	inline static const char * GetTypeName( Type t ) { return s_NodeTypeNames[ t ]; }
	template < class T >
	inline T * CastTo() const;

	// each node must specify if it ouputs a file
	virtual bool IsAFile() const = 0;

	inline State GetState() const { return m_State; }

	inline bool GetStatFlag( StatsFlag flag ) const { return ( ( m_StatsFlags & flag ) != 0 ); }
	inline void SetStatFlag( StatsFlag flag )		{ m_StatsFlags |= flag; }

	inline uint32_t GetLastBuildTime() const	{ return m_LastBuildTimeMs; }
	inline uint32_t GetProcessingTime() const	{ return m_ProcessingTime; }

	virtual const Array< Node * > & GetStaticDependencies() const;
	virtual const Array< Node * > & GetDynamicDependencies() const;

	static Node *	Load( IOStream & stream, bool remote );
	static bool		Save( IOStream & stream, const Node * node );

	static bool EnsurePathExistsForFile( const AString & name );
protected:
	friend class FBuild;
	friend class JobQueue;
	friend class JobQueueRemote;
	friend class NodeGraph;
	friend class WorkerThread;

	void ReplaceDummyName( const AString & newName );

	virtual bool Save( IOStream & stream ) const = 0;

	inline uint32_t GetControlFlags() const { return m_ControlFlags; }

	inline void SetState( State state ) { m_State = state; }

	// each node must implement these core functions
	virtual bool DoDynamicDependencies( bool forceClean );
	virtual bool DetermineNeedToBuild( bool forceClean ) = 0;
	virtual BuildResult DoBuild( Job * job );
	virtual BuildResult DoBuild2( Job * job );

	inline void		SetLastBuildTime( uint32_t ms ) { m_LastBuildTimeMs = ms; }
	inline void		AddProcessingTime( uint32_t ms ){ m_ProcessingTime += ms; }

	// serialization helpers
	bool SaveDepArray( IOStream & stream, const Array< Node * > & depArray ) const;
	static bool LoadDepArray( IOStream & stream, Array< Node * > & deps, bool remote );

	bool SaveNode( IOStream & stream, const Node * node ) const;
	static bool LoadNode( IOStream & stream, Node * & node );
	static bool LoadNode( IOStream & stream, FileNode * & node );

	static void DumpOutput( const char * data, 
							uint32_t dataSize,
							const Array< AString > * exclusions = nullptr );

	State m_State;
	uint32_t		m_ControlFlags;
	uint32_t		m_StatsFlags;
	Type m_Type;
	Node *			m_Next; // node map linked list pointer
	uint32_t		m_NameCRC;
	uint32_t m_LastBuildTimeMs;	// time it took to do last known full build of this node
	uint32_t m_ProcessingTime;	// time spent on this node
	AString	m_Name;

	// for nodes to return to save storage for types that never
	// need certain types of dependencies
	static Array< Node * > s_NoDependencies;

	static const char * const s_NodeTypeNames[ NUM_NODE_TYPES ];
};

//------------------------------------------------------------------------------
template < class T >
inline T * Node::CastTo() const
{
	ASSERT( T::GetType() == GetType() );
	return (T *)this;
}

//------------------------------------------------------------------------------
#endif // FBUILD_GRAPH_NODE_H
