// Node.cpp - base interface for dependency graph nodes
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Node.h"
#include "FileNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/AliasNode.h>
#include <Tools/FBuild/FBuildCore/Graph/CopyNode.h>
#include <Tools/FBuild/FBuildCore/Graph/CSNode.h>
#include <Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h>
#include <Tools/FBuild/FBuildCore/Graph/ExecNode.h>
#include <Tools/FBuild/FBuildCore/Graph/FileNode.h>
#include <Tools/FBuild/FBuildCore/Graph/LibraryNode.h>
#include <Tools/FBuild/FBuildCore/Graph/LinkerNode.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeProxy.h>
#include <Tools/FBuild/FBuildCore/Graph/ObjectNode.h>
#include <Tools/FBuild/FBuildCore/Graph/TestNode.h>
#include <Tools/FBuild/FBuildCore/Graph/UnityNode.h>

#include <Core/Containers/Array.h>
#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/IOStream.h>
#include <Core/Math/CRC32.h>
#include <Core/Strings/AStackString.h>

// Static Data
//------------------------------------------------------------------------------
/*static*/ Array< Node * > Node::s_NoDependencies( 0, false );

/*static*/ const char * const Node::s_NodeTypeNames[ NUM_NODE_TYPES ] = 
{
	"Proxy",
	"Copy",
	"Directory",
	"Exec",
	"File",
	"Library",
	"Object",
	"Alias",
	"Linker",
	"Unity",
	"C#",
	"Test"
};

// CONSTRUCTOR
//------------------------------------------------------------------------------
Node::Node( const AString & name, Type type, uint32_t controlFlags )
	: m_State( NOT_PROCESSED )
	, m_ControlFlags( controlFlags )
	, m_StatsFlags( 0 )
	, m_Type( type )
	, m_Next( nullptr )
	, m_LastBuildTimeMs( 0 )
	, m_ProcessingTime( 0 )
	, m_Name( name )
{
	m_NameCRC = CRC32::CalcLower( name );
}

// DESTRUCTOR
//------------------------------------------------------------------------------
Node::~Node()
{
}

// GetStaticDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & Node::GetStaticDependencies() const
{
	return s_NoDependencies;
}

// GetDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & Node::GetDynamicDependencies() const
{
	return s_NoDependencies;
}

// DoDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ bool Node::DoDynamicDependencies( bool )
{
	return true;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult Node::DoBuild( Job * UNUSED( job ) )
{
	ASSERT( false ); // Derived class is missing implementation
	return Node::NODE_RESULT_FAILED;
}

// DoBuild2
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult Node::DoBuild2( Job * UNUSED( job ) )
{
	ASSERT( false ); // Derived class is missing implementation
	return Node::NODE_RESULT_FAILED;
}

//------------------------------------------------------------------------------
bool Node::SaveDepArray( IOStream & fileStream, const Array< Node * > & depArray ) const
{
	size_t numDeps = depArray.GetSize();
	if ( fileStream.Write( (uint32_t)numDeps ) == false )
	{
		return false;
	}
	for ( uint32_t i=0; i<numDeps; ++i )
	{
		if ( fileStream.Write( depArray[ i ]->GetName() ) == false )
		{
			return false;
		}
	}
	return true;
}

//------------------------------------------------------------------------------
/*static*/ bool Node::LoadDepArray( IOStream & fileStream, Array< Node * > & deps, bool remote )
{
	uint32_t numDeps;
	if ( fileStream.Read( numDeps ) == false )
	{
		return false;
	}
	if ( deps.GetCapacity() < deps.GetSize() + numDeps )
	{
		deps.SetCapacity( deps.GetSize() + numDeps );
	}
	for ( uint32_t i=0; i<numDeps; ++i )
	{
		AStackString< 512 > depName;
		if ( fileStream.Read( depName ) == false )
		{
			return false;
		}
		Node * n = nullptr;
		if ( remote )
		{
			n = new NodeProxy( depName );
		}
		else
		{
			NodeGraph & ng = FBuild::Get().GetDependencyGraph();
			n = ng.FindNode( depName );
			ASSERT( n );
		}
		deps.Append( n );
	}
	return true;
}

// SaveNode
//------------------------------------------------------------------------------
bool Node::SaveNode( IOStream & fileStream, const Node * node ) const
{
	// for null pointer, write an empty string
	if ( node == nullptr )
	{
		if ( fileStream.Write( AString::GetEmpty() ) == false )
		{
			return false;
		}
	}
	else
	{
		// for valid nodes, write the node name
		if ( fileStream.Write( node->GetName() ) == false )
		{
			return false;
		}
	}

	return true;
}

// LoadNode
//------------------------------------------------------------------------------
/*static*/ bool Node::LoadNode( IOStream & stream, Node * & node )
{
	// read the name of the node
	AStackString< 512 > nodeName;
	if ( stream.Read( nodeName ) == false )
	{
		return false;
	}

	// empty name means the pointer was null, which is supported
	if ( nodeName.IsEmpty() )
	{
		node = nullptr;
		return true;
	}

	// find the node by name - this should never fail
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * n = ng.FindNode( nodeName );
	if ( n == nullptr )
	{
		return false;
	}
	node = n;

	return true;
}

// LoadNode (FileNode)
//------------------------------------------------------------------------------
/*static*/ bool Node::LoadNode( IOStream & stream, FileNode * & fileNode )
{
	Node * node;
	if ( !LoadNode( stream, node ) )
	{
		return false;
	}
	if ( !node->IsAFile() )
	{
		return false;
	}
	fileNode = node->CastTo< FileNode >();
	return ( fileNode != nullptr );
}

// EnsurePathExistsForFile
//------------------------------------------------------------------------------
/*static*/ bool Node::EnsurePathExistsForFile( const AString & name )
{
	const char * lastSlash = name.FindLast( '\\' );
	ASSERT( lastSlash ); // should be guaranteed to be a full path
	AStackString<> pathOnly( name.Get(), lastSlash );
	if ( FileIO::EnsurePathExists( pathOnly ) == false )
	{
		FLOG_ERROR( "Failed to create path '%s'", pathOnly.Get() );
		return false;
	}
	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * Node::Load( IOStream & stream, bool remote )
{
	// read type
	uint32_t nodeType;
	if ( stream.Read( nodeType ) == false )
	{
		return nullptr;
	}

	// read contents
	Node * n = nullptr;
	switch ( (Node::Type)nodeType )
	{
		case Node::COPY_NODE:			n = CopyNode::Load( stream, remote );			break;
		case Node::DIRECTORY_LIST_NODE: n = DirectoryListNode::Load( stream, remote );	break;
		case Node::EXEC_NODE:			n = ExecNode::Load( stream, remote );			break;
		case Node::FILE_NODE:			n = FileNode::Load( stream, remote );			break;
		case Node::LIBRARY_NODE:		n = LibraryNode::Load( stream, remote );		break;
		case Node::OBJECT_NODE:			n = ObjectNode::Load( stream, remote );			break;
		case Node::ALIAS_NODE:			n = AliasNode::Load( stream, remote );			break;
		case Node::LINKER_NODE:			n = LinkerNode::Load( stream, remote );			break;
		case Node::CS_NODE:				n = CSNode::Load( stream, remote );				break;
		case Node::UNITY_NODE:			n = UnityNode::Load( stream, remote );			break;
		case Node::TEST_NODE:			n = TestNode::Load( stream, remote );			break;
		default:
		{
			ASSERT( false );
			break;
		}
	}
	return n;
}

// 
//------------------------------------------------------------------------------
/*static*/ bool Node::Save( IOStream & stream, const Node * node )
{
	ASSERT( node );

	// save type
	uint32_t nodeType = (uint32_t)node->GetType();
	if ( stream.Write( nodeType ) == false )
	{
		return false;
	}

	// save contents
	return node->Save( stream );
}

// ReplaceDummyName
//------------------------------------------------------------------------------
void Node::ReplaceDummyName( const AString & newName )
{
	ASSERT( m_Name == "$$dummyName$$" );
	m_Name = newName;
}

// DumpOutput
//------------------------------------------------------------------------------
/*static*/ void Node::DumpOutput( const char * data, 
								  uint32_t dataSize,
								  const Array< AString > * exclusions )
{
	if ( ( data == nullptr ) || ( dataSize == 0 ) )
	{
		return;
	}

	const char * end = data + dataSize;
	while( data < end )
	{
		// find the limits of the current line
		const char * lineStart = data;
		const char * lineEnd = lineStart;
		while ( lineEnd < end )
		{
			if ( ( *lineEnd == '\r' ) || ( *lineEnd == '\n' ) )
			{
				break;
			}
			lineEnd++;
		}
		if ( lineStart != lineEnd ) // ignore empty
		{
			// make a copy of the line to output
			AStackString< 1024 > copy( lineStart, lineEnd );

			// skip this line?
			bool skip = false;
			if ( exclusions )
			{
				auto iter = exclusions->Begin();
				auto endIter = exclusions->End();
				while ( iter != endIter )
				{
					if ( copy.BeginsWith( *iter ) )
					{
						skip = true;
						break;
					}
					iter++;
				}
			}
			if ( !skip )
			{
				copy.Replace( "\n", "" );
				copy.Replace( "\r", "" );
				copy += '\n';
				FLOG_ERROR_DIRECT( copy.Get() );
			}
		}
		data = ( lineEnd + 1 );
	}
}

//------------------------------------------------------------------------------
