// LinkerNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "LinkerNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/FileNode.h>
#include <Tools/FBuild/FBuildCore/Graph/LibraryNode.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Process/Process.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
LinkerNode::LinkerNode( const AString & linkerOutputName,
						 const Array< Node * > & inputLibraries,
						 const AString & linker,
						 const AString & linkerArgs )
: FileNode( linkerOutputName, Node::FLAG_NONE )
, m_StaticDependencies( inputLibraries.GetSize(), false )
{
	m_Type = LINKER_NODE;
	m_LastBuildTimeMs = 20000;

	// depend on everything we'll link together
	ASSERT( inputLibraries.IsEmpty() == false );
	m_StaticDependencies = inputLibraries;

	// store options we'll need to use dynamically
	m_Linker = linker; // TODO:C This should be a node
	m_LinkerArgs = linkerArgs;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
LinkerNode::~LinkerNode()
{
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult LinkerNode::DoBuild( Job * UNUSED( job ) )
{
	// need building...
	FLOG_BUILD( "Exe: %s\n", GetName().Get() );

	// Format compiler args string
	AStackString< 4 * KILOBYTE > fullArgs;
	fullArgs = m_LinkerArgs;

	// substitute "function variables" if needed
	AStackString< 4 * KILOBYTE > inputFiles;
	for ( Array< Node * >::Iter i = m_StaticDependencies.Begin();
		  i != m_StaticDependencies.End();
		  i++ )
	{
		inputFiles += ( *i )->GetName();
		inputFiles += ' ';
	}
	fullArgs.Replace( "%1", inputFiles.Get() );
	fullArgs.Replace( "%2", m_Name.Get() );

	// use the exe launch dir as the working dir
	const char * workingDir = nullptr;

	const char * environment = FBuild::Get().GetEnvironmentString();

	FLOG_INFO( "%s %s\n", m_Linker.Get(), fullArgs.Get() );

	// spawn the process
	Process p;
	bool spawnOK = p.Spawn( m_Linker.Get(),
							fullArgs.Get(),
							workingDir,
							environment );

	if ( !spawnOK )
	{
		FLOG_ERROR( "Failed to spawn process for Linker creation for '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	ASSERT( !p.IsRunning() );
	// Get result
	int result = p.WaitForExit();

	if ( result != 0 )
	{
		if ( memOut.Get() ) { FLOG_ERROR_DIRECT( memOut.Get() ); }
		if ( memErr.Get() ) { FLOG_ERROR_DIRECT( memErr.Get() ); }
	}

	// did the executable fail?
	if ( result != 0 )
	{
		FLOG_ERROR( "Failed to build Linker (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// record time stamp for next time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );
	ASSERT( m_TimeStamp );

	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & LinkerNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool LinkerNode::DetermineNeedToBuild( bool forceClean )
{
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	//uint64_t lastWriteTime = GetLastWriteTime();
	if ( m_TimeStamp == 0 )
	{
		// output file is missing, so it must be built
		return true;
	}

	if ( forceClean )
	{
		return true;
	}

	/*
	if ( GetTimeStamp() == 0 )
	{
		// file exists, be we don't know about it
		// (built by someone else, or we're doing a clean build)
		return true;
	}

	if ( lastWriteTime != GetTimeStamp() )
	{
		// file exists and we've built before, but they are out of sync
		// (probably modified or built by an external process)
		return true;
	}*/

	// we have a list of libraries - are any out of date or missing?
	//uint32_t filesMissing = 0;
	//uint32_t filesChanged = 0;
	for ( Array< Node * >::ConstIter it = m_StaticDependencies.Begin();
			it != m_StaticDependencies.End();
			it++ )
	{
		uint64_t stamp = 0;
		if ( ( *it )->GetType() == Node::FILE_NODE )
		{
			FileNode * fn = ( *it )->CastTo< FileNode >();
			stamp = fn->GetTimeStamp();
		}
		else if ( ( *it )->GetType() == Node::LIBRARY_NODE )
		{
			LibraryNode * ln = ( *it )->CastTo< LibraryNode >();
			stamp = ln->GetTimeStamp();			
		}
		else
		{
			ASSERT( false ); // should only have File and Library nodes!
		}

		if ( stamp == 0 )
		{
			// file missing
			return true;
			//filesMissing++;
			//continue;
		}

		if ( m_TimeStamp < stamp )
		{
			// file changed
			return true;
			//filesChanged++;
			//continue;
		}
	}

	//if ( ( filesChanged > 0 ) || ( filesMissing > 0 ) )
	//{
	//	return true;
	//}

	// nothing needs building
	return false;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * LinkerNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( AStackString<>,	linker );
	NODE_LOAD( AStackString<>,	linkerArgs );
	NODE_LOAD_DEPS( 0,			inputLibs);

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	return ng.CreateLinkerNode( name, inputLibs, linker, linkerArgs );
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool LinkerNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_Linker );
	NODE_SAVE( m_LinkerArgs );
	NODE_SAVE_DEPS( m_StaticDependencies );
	return true;
}

//------------------------------------------------------------------------------
