// LibraryNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "LibraryNode.h"
#include "DirectoryListNode.h"
#include "UnityNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/ObjectNode.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Process/Process.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
LibraryNode::LibraryNode( const AString & libraryName,
						  Array< Node * > & inputNodes,
						  FileNode * compiler,
						  const AString & compilerArgs,
						  const AString & compilerOutputPath,
						  const AString & linker,
						  const AString & linkerArgs,
						  ObjectNode * precompiledHeader )
: FileNode( libraryName, Node::FLAG_NONE )
, m_StaticDependencies( 2, true )
, m_DynamicDependencies( 0, true )
{
	m_Type = LIBRARY_NODE;
	m_LastBuildTimeMs = 1000; // higher default than a file node

	// depend on the input nodes
	ASSERT( !inputNodes.IsEmpty() );
	m_StaticDependencies = inputNodes;

	// store precompiled headers if provided
	m_PrecompiledHeader = precompiledHeader;

	// store options we'll need to use dynamically
	m_Compiler = compiler;
	m_CompilerArgs = compilerArgs;
	m_CompilerOutputPath = compilerOutputPath;
	m_LinkerPath = linker; // TODO:C This should be a node
	m_LinkerArgs = linkerArgs;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
LibraryNode::~LibraryNode()
{
}

// DoDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ bool LibraryNode::DoDynamicDependencies( bool forceClean )
{
	(void)forceClean; // dynamic deps are always re-added here, so this is meaningless

	// clear dynamic deps from previous passes
	m_DynamicDependencies.Clear();

	//Timer t;

	Node * pchCPP = nullptr;
	if ( m_PrecompiledHeader )
	{
		ASSERT( m_PrecompiledHeader->GetType() == Node::OBJECT_NODE );
		pchCPP = m_PrecompiledHeader->GetStaticDependencies()[ 1 ];
	}

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// if we depend on any directory lists, we need to use them to get our files
	for ( Array< Node * >::Iter i = m_StaticDependencies.Begin();
		  i != m_StaticDependencies.End();
		  i++ )
	{
		// is this a dir list?
		if ( ( *i )->GetType() == Node::DIRECTORY_LIST_NODE )
		{
			// get the list of files
			DirectoryListNode * dln = (*i)->CastTo< DirectoryListNode >();
			const Array< AString > & files = dln->GetFiles();
			m_DynamicDependencies.SetCapacity( m_DynamicDependencies.GetSize() + files.GetSize() );
			for ( Array< AString >::Iter fIt = files.Begin();
					fIt != files.End();
					fIt++ )
			{
				// Create the file node (or find an existing one)
				Node * n = ng.FindNode( *fIt );
				if ( n == nullptr )
				{
					n = ng.CreateFileNode( *fIt );
				}
				else if ( n->IsAFile() == false )
				{
					FLOG_ERROR( "Library() .CompilerInputFile '%s' is not a FileNode (type: %s)", n->GetName().Get(), n->GetTypeName() );
					return false;
				}

				// ignore the precompiled header as a convenience for the user
				// so they don't have to exclude it explicitly
				if ( pchCPP && ( n == pchCPP ) )
				{
					continue;
				}

				// create the object that will compile the above file
				if ( CreateDynamicObjectNode( n ) == false )
				{
					return false; // CreateDynamicObjectNode will have emitted error
				}
			}
		}
		else if ( ( *i )->GetType() == Node::FILE_NODE )
		{
			// a single file, create the object that will compile it
			if ( CreateDynamicObjectNode( *i ) == false )
			{
				return false; // CreateDynamicObjectNode will have emitted error
			}
		}
		else if ( ( *i )->GetType() == Node::UNITY_NODE )
		{
			// get the dir list from the unity node
			UnityNode * un = ( *i )->CastTo< UnityNode >();
			const Array< AString > & unityFiles = un->GetUnityFileNames();
			for ( Array< AString >::Iter it = unityFiles.Begin();
				  it != unityFiles.End();
				  it++ )
			{
				Node * n = ng.FindNode( *it );
				if ( n == nullptr )
				{
					n = ng.CreateFileNode( *it );
				}
				else if ( n->IsAFile() == false )
				{
					FLOG_ERROR( "Library() .CompilerInputUnity '%s' is not a FileNode (type: %s)", n->GetName().Get(), n->GetTypeName() );
					return false;
				}

				// create the object that will compile the above file
				if ( CreateDynamicObjectNode( n ) == false )
				{
					return false; // CreateDynamicObjectNode will have emitted error
				}
			}
		}
		else
		{
			ASSERT( false ); // unexpected node type
		}
	}

	//float time = t.GetElapsed();
	//FLOG_WARN( "DynamicDeps: %2.3f\t%s", time, GetName().Get() );

	// make sure we have something to build!
	if ( m_DynamicDependencies.GetSize() == 0 )
	{
		FLOG_ERROR( "No files found to build '%s'", GetName().Get() );
		return false;
	}
	return true;
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool LibraryNode::DetermineNeedToBuild( bool forceClean )
{
	// note the stamp of our output file for later comparisson to the dst
	// (will be 0 if file is missing)
	uint64_t lastWriteTime = FileIO::GetFileLastWriteTime( m_Name );

	if ( lastWriteTime == 0 )
	{
		// output file is missing, so it must be built
		return true;
	}

	if ( GetTimeStamp() == 0 )
	{
		// file exists, be we don't know about it
		// (built by someone else, or we're doing a clean build)
		return true;
	}

	if ( lastWriteTime != GetTimeStamp() )
	{
		// file exists and we've built before, but they are out of sync
		// (probably modified or built by an external process)
		return true;
	}

	if ( forceClean == true )
	{
		return true;
	}

	// work out the time stamp for the most recent thing we depend on
	for ( Array< Node * >::Iter i = m_DynamicDependencies.Begin();
			i != m_DynamicDependencies.End();
			i++ )
	{
		Node * n = *i;
		if ( ( n->GetType() == Node::FILE_NODE ) ||
				( n->GetType() == Node::OBJECT_NODE ) )
		{
			FileNode * fn = (FileNode *)n;
			if ( m_TimeStamp < fn->GetTimeStamp() )
			{
				return true;
			}
		}
	}

	return false;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult LibraryNode::DoBuild( Job * UNUSED( job ) )
{
	// need building...
	FLOG_BUILD( "Lib: %s\n", GetName().Get() );

	// Format compiler args string
	AStackString< 4 * KILOBYTE > fullArgs;
	fullArgs = m_LinkerArgs;

	// substitute "function variables" if needed
	AStackString< 4 * KILOBYTE > inputFiles;
	for ( Array< Node * >::Iter i = m_DynamicDependencies.Begin();
		  i != m_DynamicDependencies.End();
		  i++ )
	{
		inputFiles += ( *i )->GetName();
		inputFiles += ' ';
	}
	fullArgs.Replace( "%1", inputFiles.Get() );
	fullArgs.Replace( "%2", m_Name.Get() );

	// use the exe launch dir as the working dir
	const char * workingDir = nullptr;

	const char * environment = FBuild::Get().GetEnvironmentString();

	FLOG_INFO( "%s %s\n", m_LinkerPath.Get(), fullArgs.Get() );

	// spawn the process
	Process p;
	bool spawnOK = p.Spawn( m_LinkerPath.Get(),
							fullArgs.Get(),
							workingDir,
							environment );

	if ( !spawnOK )
	{
		FLOG_ERROR( "Failed to spawn process for Library creation for '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	ASSERT( !p.IsRunning() );
	// Get result
	int result = p.WaitForExit();

	if ( result != 0 )
	{
		if ( memOut.Get() ) { FLOG_ERROR_DIRECT( memOut.Get() ); }
		if ( memErr.Get() ) { FLOG_ERROR_DIRECT( memErr.Get() ); }
	}

	// did the executable fail?
	if ( result != 0 )
	{
		FLOG_ERROR( "Failed to build Library (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// record time stamp for next time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );
	ASSERT( m_TimeStamp );

	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & LibraryNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & LibraryNode::GetDynamicDependencies() const
{
	return m_DynamicDependencies;
}

// CreateDynamicObjectNode
//------------------------------------------------------------------------------
bool LibraryNode::CreateDynamicObjectNode( Node * inputFile )
{
	const AString & fileName = inputFile->GetName();

	// Transform src file to dst object path
	// get file name only (no path, no ext)
	const char * lastSlash = fileName.FindLast( '\\' );
	lastSlash = lastSlash ? ( lastSlash + 1 ) : fileName.Get();
	const char * lastDot = fileName.FindLast( '.' );
	lastDot = lastDot && ( lastDot > lastSlash ) ? lastDot : fileName.GetEnd();
	AStackString<> fileNameOnly( lastSlash, lastDot );
	AStackString<> objFile( m_CompilerOutputPath );
	objFile += fileNameOnly;
	objFile += ".obj"; // TODO:B Expose the ability to modify the extension

	// Create an ObjectNode to compile the above file
	// and depend on that
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * on = ng.FindNode( objFile );
	if ( on == nullptr )
	{
		// determine flags
		uint32_t flags = 0;
		if ( m_Compiler->GetName().EndsWithI( "\\cl.exe" ) )
		{
			flags |= ObjectNode::FLAG_MSVC;

			// FunctionLibrary should have ensured this is not used
			ASSERT( !m_CompilerArgs.BeginsWith( "/Yc" ) && !m_CompilerArgs.Find( " /Yc" ) );

			bool using7zDebugFormat = ( m_CompilerArgs.Find( " /Z7 " ) ||
										m_CompilerArgs.BeginsWith( "/Z7 " ) ||
										m_CompilerArgs.EndsWith( " /Z7" ) );

			if ( using7zDebugFormat )
			{
				// distribution requires Z7 format
				flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;

				// cache requires Z7 format
				flags |= ObjectNode::FLAG_CAN_BE_CACHED;
			}

			if ( m_CompilerArgs.BeginsWith( "/Yu" ) || m_CompilerArgs.Find( " /Yu" ) )
			{
				flags |= ObjectNode::FLAG_USING_PCH;
			}
		}
		else if ( m_Compiler->GetName().EndsWithI( "gcc" ) )
		{
			flags |= ObjectNode::FLAG_GCC;
			flags |= ObjectNode::FLAG_CAN_BE_DISTRIBUTED;
		}

		on = ng.CreateObjectNode( objFile, inputFile, m_Compiler, m_CompilerArgs, m_PrecompiledHeader, flags );
	}
	else if ( on->GetType() != Node::OBJECT_NODE )
	{
		FLOG_ERROR( "Node '%s' is not an ObjectNode (type: %s)", on->GetName().Get(), on->GetTypeName() );
		return false;
	}
	m_DynamicDependencies.Append( on );
	return true;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * LibraryNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( uint64_t,		timeStamp );
	NODE_LOAD_NODE( FileNode,	compilerNode );
	NODE_LOAD( AStackString<>,	compilerArgs );
	NODE_LOAD( AStackString<>,	compilerOutputPath );
	NODE_LOAD( AStackString<>,	linkerPath );
	NODE_LOAD( AStackString<>,	linkerArgs );
	NODE_LOAD_DEPS( 16,			staticDeps );
	NODE_LOAD_NODE( Node,		precompiledHeader );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	LibraryNode * n = ng.CreateLibraryNode( name, 
								 staticDeps, 
								 compilerNode, 
								 compilerArgs, 
								 compilerOutputPath, 
								 linkerPath, 
								 linkerArgs,
								 precompiledHeader ? precompiledHeader->CastTo< ObjectNode >() : nullptr );

	n->m_TimeStamp = timeStamp;

	// TODO:B Need to save the dynamic deps, for better progress estimates
	// but we can't right now because we rely on the nodes we depend on 
	// being saved before us which isn't the case for dynamic deps.
	//if ( Node::LoadDepArray( fileStream, n->m_DynamicDependencies ) == false )
	//{
	//	delete n;
	//	return nullptr;
	//}
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool LibraryNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_TimeStamp );
	NODE_SAVE_NODE( m_Compiler );
	NODE_SAVE( m_CompilerArgs );
	NODE_SAVE( m_CompilerOutputPath );
	NODE_SAVE( m_LinkerPath );
	NODE_SAVE( m_LinkerArgs );
	NODE_SAVE_DEPS( m_StaticDependencies );
	NODE_SAVE_NODE( m_PrecompiledHeader );

	// TODO:B Need to save the dynamic deps, for better progress estimates
	// but we can't right now because we rely on the nodes we depend on 
	// being saved before us which isn't the case for dynamic deps.
	// dynamic deps are saved for more accurate progress estimates in future builds
	//if ( Node::SaveDepArray( fileStream, m_DynamicDependencies ) == false )
	//{
	//	return false;
	//}

	return true;
}

//------------------------------------------------------------------------------
