// ExecNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "ExecNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Math/Conversions.h>
#include <Core/Strings/AStackString.h>
#include <Core/Process/Process.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
ExecNode::ExecNode( const AString & dstFileName,
					    FileNode * sourceFile,
						FileNode * executable,
						const char * arguments,
						const char * workingDir )
: FileNode( dstFileName, ControlFlag::FLAG_NONE )
, m_StaticDependencies( 2, true )
, m_SourceFile( sourceFile )
, m_Executable( executable )
, m_Arguments( arguments )
{
	ASSERT( sourceFile );
	ASSERT( executable );
	m_StaticDependencies.Append( sourceFile );
	m_StaticDependencies.Append( executable );
	if ( workingDir )
	{
		m_WorkingDir = workingDir;
	}
	m_Type = EXEC_NODE;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
ExecNode::~ExecNode()
{
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool ExecNode::DetermineNeedToBuild( bool forceClean )
{
	// note the stamp of our output file for later comparison to the dst
	// (will be 0 if file is missing)
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	// or older than the source, we are all good
	if ( ( m_TimeStamp < m_SourceFile->GetTimeStamp() ) ||
			( m_TimeStamp < m_Executable->GetTimeStamp() ) )
	{
		return true;
	}
	if ( forceClean )
	{
		return true;
	}
	return false;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult ExecNode::DoBuild( Job * UNUSED( job ) )
{
	// If the workingDir is empty, use the current dir for the process
	const char * workingDir = m_WorkingDir.IsEmpty() ? nullptr : m_WorkingDir.Get();

	// spawn the process
	Process p;
	bool spawnOK = p.Spawn( m_Executable->GetName().Get(),
							m_Arguments.Get(),
							workingDir,
							FBuild::Get().GetEnvironmentString() );

	if ( !spawnOK )
	{
		FLOG_ERROR( "Failed to spawn process for '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	ASSERT( !p.IsRunning() );
	// Get result
	int result = p.WaitForExit();
	if ( result != 0 )
	{
		// something went wrong, print details
		Node::DumpOutput( memOut.Get(), memOutSize );
		Node::DumpOutput( memErr.Get(), memErrSize );
	}

	// did the executable fail?
	if ( result != 0 )
	{
		FLOG_ERROR( "Execution failed (error %i) '%s'", result, GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// update the file's "last modified" time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );
	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & ExecNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * ExecNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	fileName );
	NODE_LOAD( AStackString<>,	sourceFile );
	NODE_LOAD( AStackString<>,	executable );
	NODE_LOAD( AStackString<>,	arguments );
	NODE_LOAD( AStackString<>,	workingDir );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * srcNode = ng.FindNode( sourceFile );
	ASSERT( srcNode ); // load/save logic should ensure the src was saved first
	ASSERT( srcNode->IsAFile() );
	Node * execNode = ng.FindNode( executable );
	ASSERT( execNode ); // load/save logic should ensure the src was saved first
	ASSERT( execNode->IsAFile() );
	Node * n = ng.CreateExecNode( fileName, 
								  (FileNode *)srcNode,
								  (FileNode *)execNode,
								  arguments.Get(),
								  workingDir.Get() );
	ASSERT( n );

	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool ExecNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_SourceFile->GetName() );
	NODE_SAVE( m_Executable->GetName() );
	NODE_SAVE( m_Arguments );
	NODE_SAVE( m_WorkingDir );
	return true;
}

//------------------------------------------------------------------------------
