// FileNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "CopyNode.h"

#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
CopyNode::CopyNode( const AString & dstFileName,
					FileNode * sourceFile )
: FileNode( dstFileName, ControlFlag::FLAG_NONE )
, m_SourceFile( sourceFile )
, m_StaticDependencies( 1, false )
{
	ASSERT( sourceFile );
	ASSERT( sourceFile->IsAFile() );
	m_StaticDependencies.Append( m_SourceFile );
	m_Type = Node::COPY_NODE;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
CopyNode::~CopyNode()
{
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
/*virtual*/ bool CopyNode::DetermineNeedToBuild( bool forceClean )
{
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	// if our timestamp is older than source
	if ( ( m_TimeStamp < m_SourceFile->GetTimeStamp() ) ||
		 ( forceClean == true ) )
	{
		return true;
	}
	return false;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult CopyNode::DoBuild( Job * UNUSED( job ) )
{
	// copy the file
	if ( FileIO::FileCopy( m_SourceFile->GetName().Get(), m_Name.Get() ) == false )
	{
		return NODE_RESULT_FAILED; // copy failed
	}

	if ( FileIO::SetReadOnly( m_Name.Get(), false ) == false )
	{
		return NODE_RESULT_FAILED; // failed to remove read-only
	}

	// update the file's "last modified" time to be equal to the src file
	m_TimeStamp = m_SourceFile->GetTimeStamp();
	ASSERT( m_TimeStamp );
	if ( FileIO::SetFileLastWriteTime( m_Name, m_TimeStamp ) == false )
	{
		m_TimeStamp = 0;
		return NODE_RESULT_FAILED; // failed to set the time
	}

	return NODE_RESULT_OK;
}

// GetDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & CopyNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}


// Load
//------------------------------------------------------------------------------
/*static*/ Node * CopyNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	fileName);
	NODE_LOAD( AStackString<>,	sourceFile );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * srcNode = ng.FindNode( sourceFile );
	ASSERT( srcNode ); // load/save logic should ensure the src was saved first
	ASSERT( srcNode->IsAFile() );
	Node * n = ng.CreateCopyNode( fileName, (FileNode *)srcNode );
	ASSERT( n );
	return n;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool CopyNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_SourceFile->GetName() );
	return true;
}

//------------------------------------------------------------------------------
